#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import pytest

from si_tests import settings
from si_tests import logger
from si_tests.managers import bootstrap_manager
from si_tests.managers import si_config_manager
from si_tests.clients.openstack import OpenstackClient


LOG = logger.logger


class TestProvisionBootstrapv2MgmtCluster():

    @pytest.mark.usefixtures("introspect_MKE_7914",
                             "introspect_PRODX_9238",
                             "introspect_openstack_seed_management_deploy_objects")
    def test_bootstrapv2_kaas(self, openstack_client: OpenstackClient, show_step):
        """Bootstrap V2 on OpenStack VMs

        Scenario:
            1. Check if a provider is eligible to start
            2. Generate Keycloak user credentials and store them in si-config.yaml before rendering templates
            3. Prepare templates to apply on bootstrap cluster
            4. Check bootstrap templates for management cluster
            5. Start bootstrap V2 management cluster deployment
            6. Create default MCC users, if requested
            7. Check versions
            8. Wait for all pods to be Running and Ready
            9. Check that all expected pods exist
            10. Download bootstrap artifacts
            11. Check bootFromVolume if enabled on OpenStack provider
            12. Check LDAP integration if enabled and required
            13. Erase created KaaS environment
            14. Check correct runtime
        """

        # step 001: Check if a provider is eligible to start
        show_step(1)
        LOG.info('Check provider enablement setting')

        endpoints = None
        if settings.KAAS_OPENSTACK_ENABLED:
            endpoints = openstack_client.get_os_cloud_endpoints()

        # TODO: (morgoev) check if kind cluster has bootstrapregions
        # if so drop and recreate the cluster

        # step 002: Generate Keycloak user credentials and store them in si-config.yaml before rendering templates
        show_step(2)
        LOG.info("Initialize bootstrap manager")
        seed_ip = openstack_client.get_seed_ip()
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        LOG.info("Add data to si config")
        si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
        si_config.store_seed_ip(seed_ip)
        bootstrap.add_credentials_data_to_si_config()

        # step 003: Prepare templates to apply on bootstrap cluster
        show_step(3)
        LOG.info("Prepare seed node templates")
        bootstrap.step_003_prepare_seed_node_templates()
        LOG.info("Patch release refs")
        bootstrap.step_003_patch_bootstrapv2_release_refs()

        # step 004: Check bootstrap templates for management cluster
        if settings.SKIP_BOOTSTRAP_TEMPLATES_CHECK:
            LOG.warning("Skip bootstrap templates check due "
                        "SKIP_BOOTSTRAP_TEMPLATES_CHECK flag is set")
            if not settings.KAAS_CUSTOM_LMA_LOGGING_ENABLED:
                bootstrap.disable_lma_logging()
            if settings.CORE_KEYCLOAK_LDAP_ENABLED:
                bootstrap.enable_keycloak_ldap()
            if settings.CORE_KAAS_NTP_ENABLED:
                bootstrap.set_ntp_servers()
        else:
            show_step(4)
            bootstrap.check_templates()

        # step 005: Start bootstrap V2 management cluster deployment
        show_step(5)
        bootstrap.step_004_deploy_kaas_cluster(
            endpoints=endpoints,
            seed_instance=openstack_client.get_seed_instance())

        # step 006: Create default MCC users, if requested
        if settings.CREATE_DEFAULT_MCC_USERS_VIA_CLI:
            show_step(6)
            bootstrap.step_004_create_default_mcc_users()

        if settings.MCC_UPGRADE_AUTO_DELAY_ENABLED:
            LOG.info("Enabling auto-delay for mcc upgrade")
            bootstrap.step_006_enable_mcc_upgrade_auto_delay()

        # step 007: Check versions
        show_step(7)
        bootstrap.step_005_check_versions()

        # step 008: Wait for all pods to be Running and Ready
        show_step(8)
        bootstrap.step_006_wait_for_pods()

        # step 009: Check that all expected pods exist
        show_step(9)
        bootstrap.step_006_postdeployment_checks()

        # step 010: Download bootstrap artifacts
        show_step(10)
        bootstrap.step_007_download_bootstrap_artifacts()

        # step 011: Check bootFromVolume if enabled on OpenStack provider
        if settings.OPENSTACK_BOOT_FROM_VOLUME:
            show_step(11)
            bootstrap.check_boot_from_volume(openstack_client,
                                             boot_volume_size=settings.OPENSTACK_BOOT_VOLUME_SIZE)

        # step 012: Check LDAP integration if enabled and required
        # NOTE: for some reason there is no check for ldap on azure atm
        # in v1 scenario
        if settings.KEYCLOAK_LDAP_ENABLED and not settings.KAAS_AZURE_ENABLED:
            show_step(12)
            bootstrap.check_keystone_ldap_integration()

        # step 013: Erase created KaaS environment
        if settings.KEEP_ENV_AFTER:
            LOG.warning("Skip erase due KEEP_ENV_AFTER flag is set")
        else:
            # Erase kaas environment after bootstrap is passed
            show_step(13)
            bootstrap.step_007_erase_env_after()

        # step 014: Check runtime
        if settings.DESIRED_RUNTIME:
            show_step(14)
            bootstrap.compare_cluster_runtime_with_desired()
