import yaml
import os
import sys

from itertools import groupby
from functools import lru_cache
from si_tests.utils import packaging_version as version

from si_tests import logger
from si_tests import settings
from si_tests.utils import utils
import si_tests.utils.templates as template_utils

LOG = logger.logger


def str_presenter(dumper, data):
    if len(data.splitlines()) > 1:
        return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='|')
    return dumper.represent_scalar('tag:yaml.org,2002:str', data)


yaml.add_representer(str, str_presenter)


class ClusterReleaseVersionGenerators(object):
    """Collection of ClusterRelease version generators and filters for parallel and upgrade tests"""

    def __init__(self, kaas_manager):
        self.__kaas_manager = kaas_manager

    @property
    def kaas_manager(self):
        return self.__kaas_manager

    # Invoked from si_tests/utils/gen_child_cluster_release_versions.py
    def gen_child_cluster_release_versions(self, release_type, active_mcc_release_only=False):
        """Get the latest clusterrelease versions for each major release

        :rtype dict: {<major version>: <clusterrelease>, ...}
            Example for release_type='mke' : {'5': 'mke-5-20-0-3-3-12', '7': 'mke-7-3-0-3-4-5'}
            Example for release_type='mos' : {'6': 'mos-6-20-0-21-6'}
        """

        versions = self.kaas_manager.get_clusterrelease_names()
        kaas_releases = self.kaas_manager.get_kaasreleases()
        if active_mcc_release_only:
            kaas_releases = [self.kaas_manager.get_active_kaasrelease()]
            supported_active = kaas_releases[0].data.get('spec').get('supportedClusterReleases')
            versions = [v for v in versions if v in [r['name'] for r in supported_active]]

        release = [x for x in versions if x.startswith(f"{release_type}-")]
        for r in kaas_releases:
            supported = r.data.get('spec').get('supportedClusterReleases')
            for rel in supported:
                for cluster_release in release:
                    if cluster_release == rel.get('name'):
                        if 'byo' in rel.get('providers', {}).get('supported', []):
                            release.remove(cluster_release)

        grouped_names = groupby(release, key=lambda x: "-".join(x.split("-")[1:3]))
        grouped_names_dict = {k: list(v) for k, v in grouped_names}
        max_ver = [max(list(v), key=version.parse) for v in grouped_names_dict.values()]

        return max_ver

    def gen_child_cluster_release_mke_versions_with_pr(self, provider):
        """Get the clusterrelease versions(major+pr) for deploy

        Get clusterrelease versions for each major release
        Get latest patch-release clusterrelease versions for each major release
        """
        supported_clusterreleases = self.kaas_manager.get_supported_clusterreleases(provider)
        names = [supported_clusterreleases[x]['name'] for x in supported_clusterreleases if
                 supported_clusterreleases.get(x)['name'].startswith("mke-")]
        #########################################################
        # WA for vsphere provider
        excluded_releases = []
        # Remove not supported releases
        names = [name for name in names if name not in excluded_releases]
        #########################################################

        # Get min and max version(major and path) for each release (12.x 14.x 16.x)
        grouped_names = groupby(names, key=lambda x: "-".join(x.split("-")[1:3]))
        grouped_names_dict = {k: list(v) for k, v in grouped_names}
        max_ver = [max(list(v), key=version.parse) for v in grouped_names_dict.values()]
        min_ver = [min(list(v), key=version.parse) for v in grouped_names_dict.values()]

        combined_data = set(max_ver + min_ver)
        result = sorted(combined_data, key=version.parse)
        return result

    def gen_latest_supported_clusterreleases(self, provider, prefix):
        """Get latest clusterrelease versions for deploy

        Get clusterrelease versions for each major release
        Starting from 2.27.0 we support development in two branches
        For example:
            for 16.1.x:
                mke-16-1-0-3-7-5
                mke-16-1-5-3-7-8
                mke-16-1-6-3-7-10
                mke-16-1-7-3-7-11
            for 16.2.x
                mke-16-2-0-3-7-8
                mke-16-2-1-3-7-10
                mke-16-2-2-3-7-11

        In this case clusters mke-16-1-7-3-7-11 and mke-16-2-2-3-7-11 are being tested
        and this method returns both of latest clusters as list
        This method is aligned to situation when we have transition from 16.1.x -> 16.2.x
        In this case 16.1.x is not counted as latest anymore and will be removed from
        result list. As an example, for mcc 2.27.3 we will have 16.1.7 and 16.2.3 versions
        but 16.1.7 can be upgraded to 16.2.3. In this case 16.1.7 is not counted as latest
        and the result list will contain only 16.2.3 version as latest. This will help
        automaticaly choose correct version during Major releases testing
        """

        supported_clusterreleases = self.kaas_manager.get_supported_clusterreleases(provider)
        names = [supported_clusterreleases[x]['name'] for x in supported_clusterreleases if
                 supported_clusterreleases.get(x)['name'].startswith(prefix)]

        # Get max version for each release (like 16.1.x 16.2.x)
        grouped_names = groupby(names, key=lambda x: "-".join(x.split("-")[1:3]))
        grouped_names_dict = {k: list(v) for k, v in grouped_names}
        max_ver = [max(list(v), key=version.parse) for v in grouped_names_dict.values()]

        # Now we need to check that all of versions do not have available upgrades
        # to be sure that these versions are end versions for every minor version.
        # Else, we should remove this version from list, because it is not counting as latest
        # available anymore
        kaas_release_data = self.kaas_manager.get_active_kaasrelease().data
        supported = kaas_release_data['spec']['supportedClusterReleases']
        max_ver_data = [r for r in supported if r['name'] in max_ver]
        for i in max_ver_data:
            if i.get('availableUpgrades', []):
                max_ver.remove(i['name'])
        return max_ver

    # Invoked from si_tests/utils/gen_supported_ucp_versions.py
    def gen_supported_mke_versions(self):
        """Get the list of the supported MKE versions

        :rtype list:
        """
        cr_list = self.kaas_manager.api.kaas_clusterreleases.list_raw().to_dict()['items']
        crs = [x for x in cr_list
               if 'lcmType' in x['spec'] and 'byo' in
               x['spec']['lcmType'] and
               x['metadata']['name'].startswith("mke")]
        ucp_version = []
        kaas_release = max(self.kaas_manager.get_kaasreleases(), key=lambda x: version.parse(x.name))
        for cr in crs:
            for r in kaas_release.data.get('spec').get('supportedClusterReleases'):
                if cr['metadata']['name'] == r.get('name'):
                    machinetypes_params = cr['spec']['machineTypes']['control'] + \
                                          cr['spec']['machineTypes']['worker']
                    ucp_tag = ''
                    for ctl in machinetypes_params:
                        if 'ucp_tag' in ctl['params'].keys():
                            if not ucp_tag:
                                ucp_tag = ctl['params']['ucp_tag']
                            else:
                                if ucp_tag != ctl['params']['ucp_tag']:
                                    break
                        else:
                            if ucp_tag:
                                ucp_version.append(ucp_tag)
        return sorted(set(ucp_version), key=version.parse)

    # Invoked from si_tests/tests/deployment/child-cluster/test_create_physical_child_ceph_lma_istio_harbor.py
    def get_child_cluster_release_name(self, release_name=settings.KAAS_CHILD_CLUSTER_RELEASE_NAME):
        """Get latest child release depending on keyword in release_name

        Recognisable keywords in release_name:

        * Empty string, 'default' or 'ucp' or 'mke' are keywords
          to get latest (by version) release with 'ucp-' or 'mke-' prefix in name
        * 'current' is a keyword to choose release from 'clusterRelease' field of
          KaasRelease
        * 'kubernetes', 'kubespray', 'k8s', 'kaas' are keywords to get latest
          (by version) release with 'kubernetes-' prefix in name
        * 'openstack' is a keyword to get latest (by version) release with
          'openstack-' prefix in name

        If none of above keywords provided release_name returned as is.
        """
        # Get KaasRelease
        mgm_cluster = self.kaas_manager.get_mgmt_cluster()
        deployed_kaas_release_name = mgm_cluster.spec[
            'providerSpec']['value']['kaas']['release']
        kaas_release = self.kaas_manager.get_kaasrelease(deployed_kaas_release_name)
        kaas_release = kaas_release.data
        supported_releases = kaas_release['spec']['supportedClusterReleases']
        # Try to guess correct child release name
        if not release_name or release_name in ['default', 'ucp', 'mke']:
            # Get latest ucp (mke) release,
            # which means release name has prefix 'ucp' or 'mke'
            ucp_releases = [r for r in supported_releases
                            if (r['name'].startswith('ucp-') or
                                r['name'].startswith('mke-'))]
            ucp_releases.sort(key=lambda r: version.Version(r['version']))
            release_name = ucp_releases[-1]['name']
        # Support plenty of magic words for the same thing
        elif release_name in ['kubernetes', 'kubespray', 'k8s', 'kaas']:
            # Get latest kubespray release, which means release name has prefix
            # 'kubernetes'
            kubernetes_releases = [r for r in supported_releases
                                   if r['name'].startswith('kubernetes-')]
            kubernetes_releases.sort(key=lambda r: version.Version(r['version']))
            release_name = kubernetes_releases[-1]['name']
        elif release_name in ['current']:
            # Use clusterRelease from KaasRelease spec
            release_name = kaas_release['spec']['clusterRelease']
        elif release_name in ['openstack']:
            # Get latest openstack release, which means name has such prefix
            openstack_releases = [r for r in supported_releases if r['name'].startswith('openstack-')]
            openstack_releases.sort(key=lambda r: version.Version(r['version']))
            release_name = openstack_releases[-1]['name']
        elif release_name in ['mos']:
            # TODO: (@ovyblov) probably need merge 'openstack' and 'mos' variants
            # Get latest mos release, which means name has such prefix
            openstack_releases = [r for r in supported_releases if r['name'].startswith('mos-')]
            openstack_releases.sort(key=lambda r: version.Version(r['version']))
            release_name = openstack_releases[-1]['name']
        # Otherwise use release_name as is
        return release_name

    # not used
    def get_upgrade_path_combinations(self, **kwargs):
        """Generate upgrade path combinations from all to the latest version

        Produces 2^(n-1) lists of versions, where 'n' is amount
        of clusterreleases versions.

        :return: yields comma-separated string with release names
                 where the first release is the initial version to deploy,
                 and all others are the versions to upgrade in sequence.
        """

        def combinations(data):
            for i, d in enumerate(data):
                yield [d]
                for x in combinations(data[i + 1:]):
                    yield [d] + x

        cr_names = self.kaas_manager.get_clusterrelease_names()

        for x in combinations(cr_names[:-1]):
            path = x + [cr_names[-1]]
            yield ",".join(path)

    # not used
    def get_upgrade_path_single_sequence(self, **kwargs):
        """Generate upgrade path as a single sequence through all clusterreleases

        Produces release pairs like: "n,n+1,n+2,n+3,..."

        :return: yields comma-separated string with release name pairs
                 where the first release is the initial version to deploy,
                 and the second is the versions to upgrade.
        """
        cr_names = self.kaas_manager.get_clusterrelease_names()
        yield ",".join(cr_names)

    # not used
    def get_upgrade_path_single_sequence_pairs(self, **kwargs):
        """Generate upgrade path pairs

        Produces release pairs like: "n,n+1", "n+1,n+2", "n+2,n+3", ...

        :return: yields comma-separated string with release name pairs
                 where the first release is the initial version to deploy,
                 and the second is the versions to upgrade.
        """
        cr_names = self.kaas_manager.get_clusterrelease_names()
        for i in range(len(cr_names) - 1):
            yield f"{cr_names[i]},{cr_names[i + 1]}"

    # not used
    def get_upgrade_path_minor_major_series(self, **kwargs):
        """Generate upgrade path respecting minor/major versions

        For kaas with zero-leading versions: 0.x.y
        For further kaas versions: x.y.y

        The minor versions sequence "y" is available only inside the constant
        major version "x":  0.x.y-2 > 0.x.y-1 > 0.x.y

        If the release version has the highest available minor version, it can
        be upgraded only to the highest available minor version of the next
        major release: 0.x-2.y > 0.x-1,y > 0.x.y

        For example, given that the clusterreleases are:
        - ['0.2.1', '0.2.2', '0.2.4', '0.3.1', '0.3.2', '0.3.4']

        , there will be two upgrade paths:
        - ['0.2.1', '0.2.2', '0.2.4', '0.3.4']  # latest 0.2.x -> latest 0.3.x
        - ['0.3.1', '0.3.2', '0.3.4']

        :return: yields comma-separated string with release name pairs
                 where the first release is the initial version to deploy,
                 and all the others are the versions to upgrade in a sequence.
        """
        crs = [cr.data for cr in self.kaas_manager.get_clusterreleases()]

        versions = {}
        for cr in crs:
            ver = cr['spec']['version'].split("+")[0]
            repr_ver = str(version.parse(ver))
            versions[repr_ver] = cr['metadata']['name']

        # sort x.y.z release versions
        pversions = sorted([version.parse(v) for v in versions.keys()])

        # generate version sequences inside each major release
        # keys: major release number,
        # values: list of release versions
        sequences = {}

        major_versions = []  # major versions in the increasing order
        for pver in pversions:
            ver = str(pver)

            if ver.startswith("0"):
                # 0.x.y : x =  major, y = minor version
                v = ver.split(".")
                major = ".".join(v[0:2])
            else:
                # x.y.y : x =  major, y.y = minor version
                v = ver.split(".")
                major = v[0]

            if major in sequences:
                sequences[major].append(ver)
            else:
                sequences[major] = [ver]
                major_versions.append(major)

        # Add the latest versions of each release sequence to the previous
        # major releases.
        # In the result, each sequences of previous major releases can
        # continue upgrade using the latest versions of the next major releases.
        major_versions.reverse()
        latest_versions = []
        for major in major_versions:
            latest_in_sequence = sequences[major][-1]
            sequences[major] += latest_versions
            latest_versions = [latest_in_sequence] + latest_versions

        for major in sequences:
            sequence = [versions[ver] for ver in sequences[major]]
            yield ",".join(sequence)

    # not used
    def get_upgrade_path_minor_major_pairs(self, **kwargs):
        """Generate upgrade path pairs respecting minor/major versions

        The same paths as self.get_upgrade_path_minor_major_series(), but
        in pairs of release: start child version, upgrade version

        :return: yields comma-separated string with release name pairs
                 where the first release is the initial version to deploy,
                 and the second is the versions to upgrade.
        """
        upgrade_paths = []
        for path in self.get_upgrade_path_minor_major_series():
            cr_names = path.split(",")
            for i in range(len(cr_names) - 1):
                p = f"{cr_names[i]},{cr_names[i + 1]}"
                # filter out duplicate pairs of releases
                if p not in upgrade_paths:
                    upgrade_paths.append(p)

        for upgrade_path in upgrade_paths:
            yield upgrade_path

    def _get_release_upgrade_path(self, skip_list=None, **kwargs):
        skip_list = skip_list or []
        mgm_cluster = self.kaas_manager.get_mgmt_cluster()
        deployed_kaas_release_name = mgm_cluster.spec[
            'providerSpec']['value']['kaas']['release']
        kaas_release = self.kaas_manager.get_kaasrelease(deployed_kaas_release_name)
        kaas_release = kaas_release.data
        supported_releases = kaas_release['spec']['supportedClusterReleases']
        upgrade_paths = []
        for r in supported_releases:
            # var assignment for better understanding
            _current_version = r['version']
            _available_upgrade = r['availableUpgrades'][0]['version'] if \
                'availableUpgrades' in r else _current_version
            _supported_providers = r.get('providers', {}).get('supported', [])
            _needed_providers = kwargs.get('providers', [])

            # Upgrade sequences logic
            if _supported_providers:
                if not _needed_providers:
                    raise ValueError(f"Release {r} has supported provider(s), "
                                     f"but pipeline ran current script without "
                                     f"--providers argument\n"
                                     f"Supported releases:\n"
                                     f"{supported_releases}")
                else:
                    for provider in _needed_providers:
                        if provider in _supported_providers:
                            upgrade_paths.append((_current_version,
                                                  _available_upgrade))
            else:
                upgrade_paths.append((_current_version,
                                      _available_upgrade))

        @lru_cache(maxsize=8)
        def get_releases_name(version):
            filtered_releases = [r['name'] for r in supported_releases
                                 if r['version'] == version]
            if filtered_releases:
                return filtered_releases[0]
            else:
                raise Exception(f"There is no release for version '{version}' in supportedClusterReleases "
                                f"in the KaaSRelease object {deployed_kaas_release_name}")

        upgrade_paths = [(
            get_releases_name(upgrade_pair[0]),
            get_releases_name(upgrade_pair[1]))
            for upgrade_pair in upgrade_paths]

        for upgrade_pair in upgrade_paths:
            if not all(upgrade_pair):
                raise ValueError(f"Not all upgrades have a target: {upgrade_pair}")

        for upgrade_pair in upgrade_paths:
            if any((upgrade_pair[0].startswith(x) for x in skip_list)):
                print(f"Skip deploy for release: {upgrade_pair[0]}",
                      file=sys.stderr)
                continue
            elif any((upgrade_pair[1].startswith(x) for x in skip_list)):
                print(f"Skip upgrade to release: {upgrade_pair[1]}",
                      file=sys.stderr)
                yield (upgrade_pair[0], upgrade_pair[0])
            else:
                yield (upgrade_pair[0], upgrade_pair[1])

    # not used
    def get_kaas_defined_release_upgrade_path(self, **kwargs):
        # skip_list - list of release names or name prefixes
        # that should be skipped
        skip_list = [
            'mke-',
            'ucp-',
            'openstack-',
        ]
        providers = kwargs.get('providers', [])
        print(f"Get ClusterReleases upgrade paths for 'kubernetes-' releases with providers {providers}",
              file=sys.stderr)
        for pair in self._get_release_upgrade_path(skip_list=skip_list, **kwargs):
            yield pair

    def get_ucp_defined_release_upgrade_path(self, **kwargs):
        # skip_list - list of release names or name prefixes
        # that should be skipped
        skip_list = [
            'kubernetes-',
            'openstack-',
            'mosk-',
            'mos',
        ]
        providers = kwargs.get('providers', [])
        print(f"Get ClusterReleases upgrade paths for 'mke-' releases with providers {providers}",
              file=sys.stderr)
        for pair in self._get_release_upgrade_path(skip_list=skip_list, **kwargs):
            yield pair

    # Invoked from si_tests/utils/gen_upgrade_paths.py
    def get_openstack_defined_release_upgrade_path(self, **kwargs):
        # skip_list - list of release names or name prefixes
        # that should be skipped
        skip_list = [
            'mke-',
            'ucp-',
            'kubernetes-',
        ]
        providers = kwargs.get('providers', [])
        print(f"Get ClusterReleases upgrade paths for 'mos-' releases with providers {providers}",
              file=sys.stderr)
        for pair in self._get_release_upgrade_path(skip_list=skip_list, **kwargs):
            yield pair

    # not used
    def get_kaas_defined_release_upgrade_path_str(self, **kwargs):
        for pair in self.get_kaas_defined_release_upgrade_path(**kwargs):
            yield ",".join(pair)

    # Invoked from si_tests/utils/gen_upgrade_paths.py
    def get_ucp_defined_release_upgrade_path_str(self, **kwargs):
        for pair in self.get_ucp_defined_release_upgrade_path(**kwargs):
            yield ",".join(pair)

    # Invoked from si_tests/utils/gen_upgrade_paths.py
    def get_openstack_defined_release_upgrade_path_str(self, **kwargs):
        for pair in self.get_openstack_defined_release_upgrade_path(**kwargs):
            yield ",".join(pair)


# iterators for plain jobs:
# - static (TARGET_CLUSTER + TARGET_NAMESPACE) * num clusters (or mapped from scenario metadata file)
# iterators for parallel jobs:
# - TARGET_CLUSTER + TARGET_NAMESPACE from  TARGET_CLUSTERS
# - KAAS_CHILD_CLUSTER_RELEASE_NAME + KAAS_CHILD_CLUSTER_UPDATE_RELEASE_NAME + cluster name + cluster namespace
# - ( KAAS_CHILD_CLUSTER_RELEASE_NAME + KAAS_CHILD_CLUSTER_UPDATE_RELEASE_NAME + cluster name + cluster namespace )
#     * multiple providers
# -  ['ovs', 'tf'] + cluster name + cluster namespace
# -  ['reboot', 'shutdown'] + cluster name + cluster namespace

"""
clusters:  # let the scenario to decide: parallel or sequence
  parallel_method1:
    - TARGET_CLUSTER: a
      TARGET_NAMESPACE: a
      KAAS_CHILD_CLUSTER_RELEASE_NAME: a
    - TARGET_CLUSTER: b
      TARGET_NAMESPACE: b
      KAAS_CHILD_CLUSTER_RELEASE_NAME: b
  parallel_method2:
    - TARGET_CLUSTER: a
      TARGET_NAMESPACE: a
      KAAS_CHILD_CLUSTER_RELEASE_NAME: a
      KAAS_CHILD_CLUSTER_UPDATE_RELEASE_NAME: a
    - TARGET_CLUSTER: b
      TARGET_NAMESPACE: b
      KAAS_CHILD_CLUSTER_RELEASE_NAME: b
      KAAS_CHILD_CLUSTER_UPDATE_RELEASE_NAME: b
  linear_method1:
    - TARGET_CLUSTER: a
      TARGET_NAMESPACE: a
      KAAS_CHILD_CLUSTER_RELEASE_NAME: a
      VLAN_ID: a
    - TARGET_CLUSTER: b
      TARGET_NAMESPACE: b
      KAAS_CHILD_CLUSTER_RELEASE_NAME: b
      VLAN_ID: b
"""


class SIConfigManager(object):

    def __init__(self, kaas_manager=None, si_config_path=settings.SI_CONFIG_PATH):
        # <kaas_manager> provides methods to access 'kaasrelease' and 'clusterrelease' objects
        # TODO(ddmitriev): check why we need to access mgmt_cluster objects in generators
        #                  other than kaasrelease/clusterrelease, and avoid that
        # TODO(ddmitriev): make a new class with the similar interface like for <kaas_manager>,
        #                  which works with kaas/releases YAMLs instead of mgmt cluster
        self.__kaas_manager = kaas_manager
        self.__version_generators = ClusterReleaseVersionGenerators(kaas_manager)
        self.si_config_path = si_config_path
        if self.si_config_path and os.path.isfile(self.si_config_path):
            with open(self.si_config_path, 'r') as f:
                self.__data = yaml.load(f.read(), Loader=yaml.SafeLoader)
        else:
            self.__data = {}

    @property
    def kaas_manager(self):
        return self.__kaas_manager

    @property
    def version_generators(self):
        return self.__version_generators

    @property
    def data(self):
        """Raw data from SI_CONFIG
            :rtype: dict
        """
        return self.__data

    @data.setter
    def data(self, new_data):
        """Update data in the si-config.yaml with extra_dict data"""
        self.__data = new_data
        if self.si_config_path and os.path.isfile(self.si_config_path):
            with open(self.si_config_path, 'w') as f:
                f.write(str(yaml.dump(self.__data)))

    def create_si_scenario(self, extra_options=None):
        scenario_path = os.path.join(settings.SI_SCENARIOS_PATH, settings.SI_SCENARIO)
        if not os.path.isfile(scenario_path):
            raise Exception(f"SI_SCENARIO YAML '{settings.SI_SCENARIO}' not found in '{settings.SI_SCENARIOS_PATH}'")

        LOG.info(f"Reading SI scenario from {scenario_path}")
        options = self.get_scenario_template_options(extra_options)
        scenario_yaml = template_utils.render_template(scenario_path, options)
        LOG.debug(scenario_yaml)
        scenario = yaml.load(scenario_yaml, Loader=yaml.SafeLoader) or {}
        LOG.info(f"Created SI scenario:\n{yaml.dump(scenario)}")
        data = self.data
        data.update(scenario)
        self.data = data
        return scenario

    def get_scenario_template_options(self, extra_options=None):
        """Read kaas/cluster releases and init variables for render the scenario template"""
        extra_options = extra_options or {}
        ver_gen = self.version_generators
        major_release_versions_mke = ver_gen.gen_child_cluster_release_versions('mke')
        major_release_versions_mos = ver_gen.gen_child_cluster_release_versions('mos')
        mos_releases_upgrade_path_openstack = ver_gen.get_openstack_defined_release_upgrade_path(
            providers=['openstack'])
        mke_releases_upgrade_path_openstack = ver_gen.get_ucp_defined_release_upgrade_path(
            providers=['openstack'])
        mke_releases_upgrade_path_aws = ver_gen.get_ucp_defined_release_upgrade_path(
            providers=['aws'])
        mke_releases_upgrade_path_equinixmetal = ver_gen.get_ucp_defined_release_upgrade_path(
            providers=['equinixmetal'])
        mke_releases_upgrade_path_equinixmetalv2 = ver_gen.get_ucp_defined_release_upgrade_path(
            providers=['equinixmetalv2'])
        mke_releases_upgrade_path_vsphere = ver_gen.get_ucp_defined_release_upgrade_path(
            providers=['vsphere'])
        supported_mke_versions = ver_gen.gen_supported_mke_versions()

        options = {
            'gen_random_string': utils.gen_random_string,
            'major_release_versions_mke': list(major_release_versions_mke),
            'major_release_versions_mos': list(major_release_versions_mos),
            'mos_releases_upgrade_path_openstack': list(mos_releases_upgrade_path_openstack),
            'mke_releases_upgrade_path_openstack': list(mke_releases_upgrade_path_openstack),
            'mke_releases_upgrade_path_aws': list(mke_releases_upgrade_path_aws),
            'mke_releases_upgrade_path_equinixmetal': list(mke_releases_upgrade_path_equinixmetal),
            'mke_releases_upgrade_path_equinixmetalv2': list(mke_releases_upgrade_path_equinixmetalv2),
            'mke_releases_upgrade_path_vsphere': list(mke_releases_upgrade_path_vsphere),
            'supported_mke_versions': list(supported_mke_versions),
        }
        options.update(extra_options)
        return options

    def create_equinixmetalv2_network_config(self, region, management_network_config,
                                             region_network_configs, child_network_configs):
        data = self.data
        network_config = data.setdefault('network_config', {})
        # management_network_config is already a dict with a key <cluster_namespace>/<cluster_name>
        network_config.update(management_network_config)

        region_cluster_params = data.get("region_cluster_params", {})
        child_cluster_params = data.get("child_cluster_params", {})

        if management_network_config:
            management_cluster_key = f"{settings.CLUSTER_NAMESPACE}/{settings.CLUSTER_NAME}"
            management_region = region
            mgmt_metro = management_network_config[management_cluster_key]['metro']
        else:
            management_region = region
            mgmt_metro = None

        # Create a dict for all mgmt/region child metros in keys
        child_metros = {mgmt_metro: []}
        # Create a dict for all child metros in keys where no region
        child_multimetros = {}
        region_metros = {}
        for region in region_network_configs:
            child_metros[region['metro']] = []
            region_metros[region['metro']] = region

        for child in child_network_configs:
            if child['metro'] in child_metros:
                # child cluster is in same metro as the mgmt or region cluster
                child_metros[child['metro']].append(child)
            else:
                # child cluster is in different metro with no region cluster
                # Map such childs on the mgmt cluster MCC region below,
                # storing child network configs as dependants from mgmt_metro
                child_multimetros.setdefault(child['metro'], []).append(child)

        # LOG.debug(f"Network configs by regions:\n{yaml.dump(child_metros)}")

        # Create a dict for child cluster names in the specified regions
        child_cluster_regions = {}
        # Create a dict for child cluster names in the management region but different metros
        child_cluster_multimetros = {}
        for child_cluster_param in child_cluster_params:
            cluster_name = child_cluster_param['KAAS_CHILD_CLUSTER_NAME']
            cluster_namespace = child_cluster_param['KAAS_NAMESPACE']
            cluster_key = f"{cluster_namespace}/{cluster_name}"
            cluster_region = child_cluster_param.get("EQUINIXMETALV2_REGION", management_region)
            cluster_mgmt_metro = child_cluster_param.get("EQUINIXMETALV2_MGMT_METRO", True)
            if cluster_region != management_region or (cluster_region == management_region and cluster_mgmt_metro):
                child_cluster_regions.setdefault(cluster_region, []).append(cluster_key)
            elif cluster_region == management_region and not cluster_mgmt_metro:
                # Looks like the child MCC region is set as for management cluster,
                # but EQUINIXMETALV2_MGMT_METRO is False, which means that the child cluster
                # should be created in a different equinix Metro than for management cluster.
                child_cluster_multimetros.setdefault(cluster_region, []).append(cluster_key)
            else:
                raise Exception(f"Should never reach there. Something wrong with the conditions above "
                                f"or the child parameters: {child_cluster_param}")

        # Create a dict for region cluster names in the specified regions
        region_cluster_regions = {}
        for region_cluster_param in region_cluster_params:
            cluster_name = region_cluster_param['REGIONAL_CLUSTER_NAME']
            cluster_namespace = settings.CLUSTER_NAMESPACE  # 'default'
            cluster_key = f"{cluster_namespace}/{cluster_name}"
            cluster_region = region_cluster_param.get("EQUINIXMETALV2_REGION")
            assert cluster_region is not None, (
                f"Missing the parameter 'EQUINIXMETALV2_REGION' for the region cluster '{cluster_key}'. "
                f"Please check the scenario '{settings.SI_SCENARIO}'")
            assert cluster_region not in region_cluster_regions, (
                f"The same 'EQUINIXMETALV2_REGION' for the region clusters '{cluster_key}' and "
                f"'{region_cluster_regions[cluster_region]}'. "
                f"Please check the scenario '{settings.SI_SCENARIO}'")
            region_cluster_regions[cluster_region] = cluster_key
            child_cluster_regions.setdefault(cluster_region, [])

        # Map childs from management_region on the mgmt_metro network configs
        mgmt_child_names = child_cluster_regions.pop(management_region, [])
        mgmt_metro_network_configs = child_metros.pop(mgmt_metro, [])
        assert len(mgmt_metro_network_configs) >= len(mgmt_child_names), (
            f"Not enought network configs created for child clusters in the MCC region '{management_region}' "
            f"in 'equinixmetalv2' provider: there are {len(mgmt_metro_network_configs)} available network configs "
            f"for {len(mgmt_child_names)} child clusters. "
            f"Please check the scenario '{settings.SI_SCENARIO}'")
        for n, child_name in enumerate(mgmt_child_names):
            network_config[child_name] = mgmt_metro_network_configs[n]

        # Map childs from management_region on the multimetro network configs
        multimetro_child_names = child_cluster_multimetros.pop(management_region, [])
        mgmt_multimetro = list(child_multimetros.keys()).pop() if len(child_multimetros.keys()) > 0 else None
        mgmt_multimetro_network_configs = child_multimetros.pop(mgmt_multimetro, [])
        assert len(mgmt_multimetro_network_configs) >= len(multimetro_child_names), (
            f"Not enought network configs created for child clusters in the MCC region '{management_region}' "
            f"in 'equinixmetalv2' provider: there are {len(mgmt_multimetro_network_configs)} available "
            f"network configs for {len(multimetro_child_names)} child clusters. "
            f"Please check the scenario '{settings.SI_SCENARIO}'")
        for n, child_name in enumerate(multimetro_child_names):
            network_config[child_name] = mgmt_multimetro_network_configs[n]

        # Check that there is enought network configs for region clusters
        assert len(region_network_configs) >= len(region_cluster_params), (
            "Not enought network configs created for region clusters in 'equinixmetalv2' provider: "
            f"there are {len(region_network_configs)} available network configs "
            f"for {len(region_cluster_params)} region clusters. "
            f"Please check the scenario '{settings.SI_SCENARIO}'")

        # Map MCC region names on the metros with enought network configs
        sorted_child_cluster_regions = sorted([k for k in child_cluster_regions.keys()],
                                              key=lambda x: len(child_cluster_regions[x]))
        sorted_child_metros = sorted([k for k in child_metros.keys()],
                                     key=lambda x: len(child_metros[x]))
        # Check that there is enought metros for child cluster regions
        assert len(sorted_child_metros) >= len(sorted_child_cluster_regions), (
            "Not enought metros created for child clusters from different regions in 'equinixmetalv2' provider: "
            f"there are {sorted_child_metros} available metros "
            f"for {sorted_child_cluster_regions} child cluster regions. "
            f"Please check the scenario '{settings.SI_SCENARIO}'")
        for region, metro in zip(sorted_child_cluster_regions, sorted_child_metros):
            # Check that there is enought metros for child cluster regions
            assert len(child_metros[metro]) >= len(child_cluster_regions[region]), (
                f"Not enought network configs created in metro '{metro}' for child clusters from MCC region "
                f"'{region}' in 'equinixmetalv2' provider: there are {len(child_metros[metro])} available "
                f"network configs for {len(child_cluster_regions[region])} child clusters. "
                f"Please check the scenario '{settings.SI_SCENARIO}'")
            assert region in region_cluster_regions, (
                f"Missing region cluster for the MCC region '{region}' to deploy "
                f"{len(child_cluster_regions[region])} child clusters. "
                f"Please check the scenario '{settings.SI_SCENARIO}'"
            )
            # Add network config for region cluster
            network_config[region_cluster_regions[region]] = region_metros[metro]
            # Add network configs for child clusters in the current region
            for n, child_name in enumerate(child_cluster_regions[region]):
                network_config[child_name] = child_metros[metro][n]

        self.data = data

    def store_seed_ip(self, seed_ip):
        LOG.info('Writing seed ip to SI_CONFIG')
        data = self.data
        data.setdefault('run_on_remote', {})['SEED_STANDALONE_EXTERNAL_IP'] = seed_ip
        self.data = data

    def update_run_on_remote(self, run_on_remote=False):
        LOG.info(f"Set RUN_ON_REMOTE='{run_on_remote}' in SI_CONFIG")
        data = self.data
        data.setdefault('run_on_remote', {})['RUN_ON_REMOTE'] = run_on_remote
        self.data = data

    def get_keycloak_user_password(self, user):
        """Get the password for the specified keycloak user from si-config.yaml"""
        assert 'keycloak_users' in self.data, "No Keycloak users found in SI_CONFIG"
        password = self.data.get('keycloak_users', {}).get(user, 'password')
        return password

    def get_heat_stack_info(self) -> dict:
        """Get stored heat stack info from si-config.yaml"""
        assert 'si_heat_stack_info' in self.data, "No si_heat_stack_info found in SI_CONFIG"
        return self.data.get('si_heat_stack_info', dict())

    def get_ansible_state_env_config(self) -> dict:
        """Get stored ansible_state_env_config info from si-config.yaml"""
        if 'ansible_state_env_config' not in self.data:
            LOG.warning("No ansible_state_env_config found in SI_CONFIG")
            return dict()
        return self.data.get('ansible_state_env_config', dict())
