#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
from si_tests import logger, settings
import json

LOG = logger.logger


class OvsOvnMigrationManager(object):

    stages = ["10_PREPARE", "20_DEPLOY_OVN_DB", "30_DEPLOY_OVN_CONTROLLERS",
              "40_MIGRATE_DATAPLANE", "50_FINALIZE_MIGRATION", "60_CLEANUP"]

    def __init__(self, os_manager):
        self.os_manager = os_manager
        self.osctlm = os_manager.osctlm

    def backup(self):
        cmd = "osctl-ovs-ovn-migrate backup_db"
        LOG.info("Running database backups")
        self.osctlm.exec(cmd, timeout=1800, verbose=True)
        LOG.info("Finished database backups")

    def pre_flight_checks(self, fail_on_check=True):
        cmd = "osctl-ovs-ovn-migrate preflight_checks"
        LOG.info("Running preflight checks")
        self.osctlm.exec(cmd, timeout=1800, raise_on_error=fail_on_check, verbose=True)
        LOG.info("Finished preflight checks")

    def migrate(self):
        cmd = "osctl-ovs-ovn-migrate migration --non-interactive"
        LOG.info("Running migration")
        self.osctlm.exec(cmd, timeout=settings.OPENSTACK_OVN_MIGRATION_TIMEOUT, verbose=True)
        LOG.info("Finished migration")

    def check_state(self, status='completed'):
        LOG.info("Checking migration state configmap")
        state_cm = self.os_manager.api.configmaps.get("ovs-ovn-migration-state", "openstack")
        state = state_cm.read().data
        assert self.stages == list(state.keys()), f"Migration state is incorrect, state is {state}"
        for stage in self.stages:
            assert json.loads(state[stage]) == {"status": status, "error": None}, \
                f"Expected {stage} status is {status}, actual status is {state[stage]}"
        LOG.info(f"Openstack OVN migration is in expected state {state}")

    def collect_logs(self):
        LOG.info("Collecting migration tool logs and reports")
        cmd = 'tar -C /tmp -czvf /tmp/ovs-ovn-migration.tar.gz ovs-ovn-migration'
        self.osctlm.exec(cmd, verbose=True)
        self.osctlm.client.cp_from_pod(source_dir="/tmp", source_file="ovs-ovn-migration.tar.gz")
        LOG.info("Finished collecting migration tool logs and reports")
