import yaml
from retry import retry

import packet
from exec_helpers import Subprocess, SSHClient, SSHAuth

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters, utils

LOG = logger.logger


class BootstrapManagerPacket(object):

    def __init__(self, auth_token=settings.KAAS_BM_CI_EQUINIX_API_TOKEN,
                 project_id=settings.KAAS_BM_CI_EQUINIX_PROJECT_ID,
                 hostname=settings.ENV_NAME,
                 plan=settings.KAAS_BM_CI_EQUINIX_PLAN,
                 metro=settings.KAAS_BM_CI_EQUINIX_METRO,
                 facility=settings.KAAS_BM_CI_EQUINIX_FACILITY,
                 operating_system=settings.KAAS_BM_CI_EQUINIX_OS,
                 ):
        self.manager = packet.Manager(
            auth_token=auth_token)
        self.project_id = project_id
        self.hostname = hostname
        self.plan = plan
        self.metro = metro
        self.facility = facility
        self.operating_system = operating_system

        raise Exception("BootstrapManagerPacket is deprecated")

    def get_seed_ip(self, dev_id):
        # need to wait sometime a little bit
        waiters.wait_pass(lambda: self.manager.get_device(dev_id), interval=3,
                          timeout=60,
                          timeout_msg=f"Waiting for device:{dev_id} failed")
        d = self.manager.get_device(dev_id)
        # search public and ipv4 ip only. Yes, we hope for only 1
        return [x['address'] for x in d.ip_addresses if
                x['public'] and x['address_family'] == 4][0]

    def get_device_id_by_hostname(self, hostname):
        all_hosts = [{'hostname': dev.hostname,
                      'ip': self.get_seed_ip(dev.id),
                      'id': dev.id}
                     for dev in
                     self.manager.list_devices(self.project_id)]
        # self-check for already created duplicates
        hosts = [i for i in all_hosts if
                 i['hostname'] == hostname]
        if len(hosts) > 1:
            raise Exception(f'Detected more then one node with name:{hostname}'
                            f'\n All devices: {all_hosts}')
        elif len(hosts) == 1:
            host = hosts[0]
            LOG.info(f"Found requested node:{host}")
            return host['id']
        elif not len(hosts):
            LOG.info(f"Device with hostname {hostname} not found")
            return False

    def delete_device(self, dev_id):
        LOG.warning(f'Deleting device {dev_id}')
        self.manager.get_device(dev_id).delete()
        return

    def delete_node_by_hostname(self, hostname):
        dev_id = self.get_device_id_by_hostname(hostname)
        if dev_id:
            LOG.warning(f"Deleting device with hostname{hostname}")
            self.delete_device(dev_id)
        else:
            LOG.warning(f"device with hostname:{hostname} not found")

    def is_device_active(self, dev_id):
        state = self.manager.get_device(dev_id).state
        LOG.info(f"Device {dev_id} state: {state}")
        if state != 'active':
            return False
        return True

    def push_docker_image_to_seed(self, dev_id):
        """ Tar local docker image, and sync it to target"""
        seed_ip = self.get_seed_ip(dev_id)
        r = Subprocess()
        r.logger.addHandler(logger.console)
        _image = 'docker-dev-kaas-local.docker.mirantis.net/mirantis/kaas/si-test:master'
        cmd = f"docker save {_image} | pigz | " \
              f"ssh -i {settings.SEED_SSH_PRIV_KEY_FILE} " \
              f"-o StrictHostKeyChecking=no " \
              f"-o UserKnownHostsFile=/dev/null " \
              f"{settings.SEED_SSH_LOGIN}@{seed_ip} 'pigz -d | docker load' "
        LOG.info(f"Attempt to run cmd: {cmd}")
        r.check_call(cmd, raise_on_err=True, verbose=True)

    @retry(Exception, delay=1, tries=3, jitter=1, logger=LOG)
    def remote_seed(self, dev_id):
        key = utils.load_keyfile(settings.SEED_SSH_PRIV_KEY_FILE)
        key = utils.get_rsa_key(key["private"])
        seed_ip = self.get_seed_ip(dev_id)
        remote = SSHClient(
            host=seed_ip,
            port=22,
            auth=SSHAuth(
                username=settings.SEED_SSH_LOGIN,
                key=key))
        remote.logger.addHandler(logger.console)
        return remote

    def create_seed_node(self):
        LOG.info("Attempt to create seed node on Equinix")
        if self.get_device_id_by_hostname(self.hostname):
            raise Exception(
                f"Device with hostname {self.hostname} already exist!")
        userdata = '#cloud-config\n'
        with open(settings.KAAS_BM_CI_EQUINIX_USERDATA, 'r') as f:
            userdata += yaml.dump(yaml.load(f.read(), Loader=yaml.SafeLoader))
        LOG.info(f"Creating HW node with hostname {self.hostname}")
        device = self.manager.create_device(
            project_id=self.project_id,
            hostname=self.hostname,
            plan=self.plan,
            metro=self.metro,
            facility=self.facility,
            operating_system=self.operating_system,
            userdata=userdata)
        waiters.wait(lambda: self.is_device_active(device.id),
                     interval=30,
                     timeout=60 * 10)
        seed_ip = self.get_seed_ip(device.id)
        LOG.info(f"Waiting to ping {seed_ip}")
        waiters.wait(lambda: waiters.icmp_ping(seed_ip), interval=5,
                     timeout=60 * 2)
        return device

    def step_002_provision_seed_vm(self):
        device_id = self.get_device_id_by_hostname(self.hostname)
        if settings.KEEP_ENV_BEFORE and device_id:
            LOG.info('Skip erase due KEEP_ENV_BEFORE flag is set')
        else:
            self.delete_node_by_hostname(self.hostname)
            device_id = self.create_seed_node().id
        seed_ip = self.get_seed_ip(device_id)
        LOG.info("Seed node ip address is {0}".format(seed_ip))
        assert seed_ip, "seed_ip is empty"
        remote = self.remote_seed(dev_id=device_id)
        # TODO(alexz): eq actually have analog for ring-to-home bell.
        # need to refactor.
        LOG.info('wait for cloud-init finish')
        waiters.wait(lambda: remote.isfile('/var/lib/cloud/si-cinit-finished'),
                     interval=10,
                     timeout=60 * 5,
                     timeout_msg="Failed to wait cloud-init")
        # upstream docker.io now complains about restart. so, need to cover
        # it with non-interactive.
        with remote.sudo(enforce=True):
            remote.check_call('export DEBIAN_FRONTEND=noninteractive ;'
                              'export DEBCONF_NONINTERACTIVE_SEEN=true ;'
                              'export LANG=C.UTF-8;'
                              'apt-get update;'
                              'apt install docker.io')
            remote.check_call(
                f"usermod -aG docker {settings.SEED_SSH_LOGIN}")
        self.push_docker_image_to_seed(device_id)
        with open(f'{settings.ARTIFACTS_DIR}/seed_node_ip',
                  mode='w') as f:
            f.write(str(seed_ip))
        with open(f'{settings.ARTIFACTS_DIR}/seed_node_dev_id',
                  mode='w') as f:
            f.write(str(device_id))
        with open(f'{settings.ARTIFACTS_DIR}/env_name',
                  mode='w') as f:
            f.write(settings.ENV_NAME)
        LOG.info(f"Provisioning done. SeedIP: {seed_ip}")
