
from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters

LOG = logger.logger


def perform_upgrade(os_manager, upgrade_version):
    osdpl_name = settings.OSH_DEPLOYMENT_NAME

    # Timeout for lcm operations
    timeout = settings.OPENSTACK_LCM_OPERATIONS_TIMEOUT

    # Upgrade OpenStack timeout
    upgrade_timeout = settings.OPENSTACK_UPGRADE_TIMEOUT

    LOG.info("Check that all fingerprints in actual state before upgrade")
    os_manager.wait_openstack_helmbundles_fingerprint(osdpl_name)

    osdpl = os_manager.get_openstackdeployment(osdpl_name)

    # Save OpenStack deployment before upgrade
    osdpl_before_upgrade = osdpl.read()

    # Apply upgrade for os version
    osdpl.patch({'spec': {'openstack_version': upgrade_version}})

    LOG.info("Waiting osdpl fingerprint updated")
    waiters.wait((lambda expected:
                  expected != os_manager.get_openstackdeployment(
                      osdpl_name).read().status['fingerprint']),
                 predicate_kwargs={
                     'expected': osdpl_before_upgrade.status['fingerprint']},
                 timeout=timeout,
                 timeout_msg="OSDpl fingerprint wasn't updated")

    LOG.info("Check all helmbundles fingerprints equals new fp of osdpl")
    os_manager.wait_openstack_helmbundles_fingerprint(
        osdpl_name,
        timeout=upgrade_timeout)

    os_controller_version = os_manager.os_controller_version()
    LOG.info(f"OpenStack controller version: {os_controller_version}")

    os_manager.wait_os_deployment_status(timeout=timeout, status="APPLIED")
    os_manager.wait_osdpl_services()

    LOG.info("Wait until all services in health section"
             " get status `Ready`")
    os_manager.wait_openstackdeployment_health_status()

    LOG.info("Check that OpenStack pods are active.")
    os_manager.wait_os_resources(timeout=timeout, interval=20)


def upgrade_openstack(os_manager, openstack_client_manager):
    """Upgrade OS version.
       Parameters required for test execution:
         - KUBECONFIG
         - OPENSTACK_UPGRADE_VERSION
    """
    upgrade_version = settings.OPENSTACK_UPGRADE_VERSION

    if not upgrade_version == "stein":
        perform_upgrade(os_manager, upgrade_version)
        return

    vm_name = "test-vm"

    with openstack_client_manager.provide_vm(vm_name):
        deployed_server = openstack_client_manager.server.show(
                [vm_name])

        deployed_host = deployed_server["OS-EXT-SRV-ATTR:hypervisor_hostname"]

        providers = openstack_client_manager.resource_provider.list([])
        provider_id = list(filter(
            lambda x: x["name"] == deployed_host, providers))[0]["uuid"]

        providers = {item["uuid"]: item["name"] for item in providers}
        usage = openstack_client_manager.resource_provider_usage.show(
            [provider_id])

        usage = {item["resource_class"]: item["usage"] for item in usage}

        perform_upgrade(os_manager, upgrade_version)

        openstack_client_manager.reload_client()
        deployed_server = openstack_client_manager.server.show(
            [vm_name])

        assert deployed_host == deployed_server["OS-EXT-SRV-ATTR:hypervisor_hostname"]

        new_providers = openstack_client_manager.resource_provider.list([])

        for item in new_providers:
            assert item["uuid"] in providers
            assert item["name"] == providers[item["uuid"]]

        new_usage = openstack_client_manager.resource_provider_usage.show(
            [provider_id])

        for item in new_usage:
            assert item["usage"] == usage[item["resource_class"]]

    LOG.info("OpenStack upgrade finished successfully.")
