#    Copyright 2019 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from contextlib import contextmanager
import shutil
import signal
import os
import paramiko
from io import StringIO

from si_runtest import logger

LOG = logger.logger


def generate_keys():
    file_obj = StringIO()
    key = paramiko.RSAKey.generate(1024)
    key.write_private_key(file_obj)
    public = key.get_base64()
    private = file_obj.getvalue()
    file_obj.close()
    return {'private': private,
            'public': public}


def load_keyfile(file_path):
    with open(file_path, 'r') as private_key_file:
        private = private_key_file.read()
    key = paramiko.RSAKey(file_obj=StringIO(private))
    public = key.get_base64()
    return {'private': private,
            'public': public}


def get_rsa_key(private_key):
    f = StringIO(private_key)
    return paramiko.rsakey.RSAKey.from_private_key(f)


def dump_keyfile(file_path, key):
    key = paramiko.RSAKey(file_obj=StringIO(key['private']))
    key.write_private_key_file(file_path)
    os.chmod(file_path, 0o644)


def clean_dir(dirpath):
    shutil.rmtree(dirpath)


def make_export_env_strting(envs):
    envs_string = '; '.join(
        ["export {}='{}'".format(k, envs[k]) for k in envs])
    return envs_string


def merge_dicts(src, dst, path=None):
    path = path or []
    for key in dst:
        if key in src:
            if isinstance(src[key], dict) and isinstance(dst[key], dict):
                merge_dicts(src[key], dst[key], path + [str(key)])
            elif src[key] == dst[key]:
                pass  # same leaf value
            else:
                src[key] = dst[key]
        else:
            src[key] = dst[key]
    return src


@contextmanager
def pushd(path):
    current_dir = os.getcwd()
    try:
        os.chdir(os.path.expanduser(path))
        yield
    finally:
        os.chdir(current_dir)


class RunLimit(object):
    # pity copy-paste from  si_tests/utils/waiters.py
    def __init__(self, timeout=60, timeout_msg='Timeout',
                 status_msg_function=None):
        self.seconds = int(timeout)
        self.error_message = timeout_msg
        self.status_msg_function = status_msg_function
        LOG.debug("RunLimit.__init__(timeout={0}, timeout_msg='{1}'"
                  .format(timeout, timeout_msg))

    def handle_timeout(self, signum, frame):
        LOG.debug("RunLimit.handle_timeout reached!")
        err_msg = self.error_message.format(spent=self.seconds)
        if self.status_msg_function is not None:
            err_msg += str(self.status_msg_function())

        raise TimeoutError(err_msg)

    def __enter__(self):
        signal.signal(signal.SIGALRM, self.handle_timeout)
        signal.alarm(self.seconds)
        LOG.debug("RunLimit.__enter__(seconds={0}".format(self.seconds))

    def __exit__(self, exc_type, value, traceback):
        time_remained = signal.alarm(0)
        LOG.debug("RunLimit.__exit__ , remained '{0}' sec"
                  .format(time_remained))
