from concurrent.futures import ThreadPoolExecutor
import logging
import sys
import time

import pytest
from texttable import Texttable

import utils
from utils import helpers
from utils import os_client
from utils import ssh


logger = logging.getLogger(__name__)


def test_vm2vm(openstack_clients, pair, os_resources, request, html_report):
    """
    Simplified Performance Tests VM to VM test in different topologies
    1. Create 4 VMs admin project
    2. Associate floating IPs to the VMs
    3. Connect to each VM via SSH and install iperf3
    4. Measure VM to VM on same node via Private IP, 1 thread
    5. Measure VM to VM on different HW nodes via Private IP, 1 thread
    6. Measure VM to VM on different HW nodes via Private IP, multiple threads
       (10 by default)
    7. Measure VM to VM on different HW nodes via Floating IP, 1 thread
    8. Measure VM to VM on different HW nodes, each VM is in separate network,
       the networks are connected using Router via Private IP, 1 thread
    9. Draw the table with all pairs and results
    """
    os_actions = os_client.OSCliActions(openstack_clients)
    config = utils.get_configuration()
    timeout = int(config.get('nova_timeout', 30))
    iperf_time = int(config.get('iperf_time', 60))
    private_key = os_resources['keypair'].private_key
    ssh_timeout = int(config.get('ssh_timeout', 500))
    threads = int(config.get('multiple_threads_number', 10))
    iperf_utility = config.get('multiple_threads_iperf_utility', 'iperf3')
    custom_mtu = config.get('custom_mtu') or 'default'
    utils.check_iperf_utility(iperf_utility)
    result_table = Texttable(max_width=120)

    try:
        zone1 = [service.zone for service in
                 openstack_clients.compute.services.list() if
                 service.host == pair[0]]
        zone2 = [service.zone for service in
                 openstack_clients.compute.services.list()
                 if service.host == pair[1]]

        # create 4 VMs
        logger.info("Creating 4 VMs...")
        vm1 = os_actions.create_basic_server(
            os_resources['image_id'], os_resources['flavor_id'],
            os_resources['net1'], '{0}:{1}'.format(zone1[0], pair[0]),
            [os_resources['sec_group']['name']], os_resources['keypair'].name)
        logger.info("Created VM {}.".format(vm1.id))

        vm2 = os_actions.create_basic_server(
            os_resources['image_id'], os_resources['flavor_id'],
            os_resources['net1'], '{0}:{1}'.format(zone1[0], pair[0]),
            [os_resources['sec_group']['name']], os_resources['keypair'].name)
        logger.info("Created VM {}.".format(vm2.id))

        vm3 = os_actions.create_basic_server(
            os_resources['image_id'], os_resources['flavor_id'],
            os_resources['net1'], '{0}:{1}'.format(zone2[0], pair[1]),
            [os_resources['sec_group']['name']], os_resources['keypair'].name)
        logger.info("Created VM {}.".format(vm3.id))

        vm4 = os_actions.create_basic_server(
            os_resources['image_id'], os_resources['flavor_id'],
            os_resources['net2'], '{0}:{1}'.format(zone2[0], pair[1]),
            [os_resources['sec_group']['name']], os_resources['keypair'].name)
        logger.info("Created VM {}.".format(vm4.id))

        vm_info = []
        vms = []
        vms.extend([vm1, vm2, vm3, vm4])
        fips = []
        time.sleep(5)

        # Associate FIPs and check VMs are Active
        logger.info("Creating Floating IPs and associating them...")
        for i in range(len(vms)):
            fip = os_actions.create_floating_ip(os_resources['ext_net']['id'])
            fips.append(fip['id'])
            os_actions.check_vm_is_active(vms[i].id, timeout=timeout)
            vms[i].add_floating_ip(fip['floating_ip_address'])
            private_address = vms[i].addresses[
                list(vms[i].addresses.keys())[0]][0]['addr']
            vm_info.append({'vm': vms[i], 'fip': fip['floating_ip_address'],
                            'private_address': private_address,
                            'private_key': private_key})
        # Set custom MTU if required
        if os_actions.is_cloud_tf() and (custom_mtu != "default"):
            logger.info("Setting up custom MTU at network ports...")
            for vm in vms:
                os_actions.update_network_port_with_custom_mtu(vm.id,
                                                               custom_mtu)

        # Check VMs are reachable and prepare iperf3
        transport1 = ssh.SSHTransport(vm_info[0]['fip'], 'ubuntu',
                                      password='dd', private_key=private_key)
        logger.info("Checking VMs are reachable via SSH...")
        for i in range(len(vms)):
            transport1.check_vm_is_reachable_ssh(
                floating_ip=vm_info[i]['fip'], timeout=ssh_timeout)
        with ThreadPoolExecutor() as executor:
            futures = [
                executor.submit(ssh.install_iperf_at_vm, vm_info)
                for vm_info in vm_info]
        logger.info("Getting MTUs at VMs...")
        mtus = []
        for vm in vm_info:
            mtus.append(ssh.get_mtu_at_vm(vm))
        logger.info("MTU at networks: {}, {}".format(
            os_resources['net1']['mtu'], os_resources['net2']['mtu']))
        # Prepare the result table and run iperf3
        table_rows = [[
            'Test Case', 'Host 1', 'Host 2', 'MTU at VMs', 'Result'
        ]]
        # Do iperf3 measurement #1
        measurement1 = ("VM to VM in same project on same node via Private "
                        "IP, 1 thread; iperf3")
        logger.info("Doing '{}' measurement...".format(measurement1))
        result1 = transport1.exec_command(
            'iperf3 -c {} -t {} | grep sender | tail -n 1'.format(
                vm_info[1]['private_address'], iperf_time))
        res1 = (b" ".join(result1.split()[-4:-2:])).decode('utf-8')
        logger.info("Result #1 is {}".format(res1))
        table_rows.append([measurement1,
                           "{}".format(pair[0]),
                           "{}".format(pair[0]),
                           "{}, {}".format(mtus[0], mtus[1]),
                           "{}".format(res1)])

        # Do iperf3 measurement #2
        measurement2 = ("VM to VM in same project on different HW nodes via "
                        "Private IP, 1 thread; iperf3")
        logger.info("Doing '{}' measurement...".format(measurement2))
        result2 = transport1.exec_command(
            'iperf3 -c {} -t {} | grep sender | tail -n 1'.format(
                vm_info[2]['private_address'], iperf_time))
        res2 = (b" ".join(result2.split()[-4:-2:])).decode('utf-8')
        logger.info("Result #2 is {}".format(res2))
        table_rows.append([measurement2,
                           "{}".format(pair[0]),
                           "{}".format(pair[1]),
                           "{}, {}".format(mtus[0], mtus[2]),
                           "{}".format(res2)])

        # Do iperf/iperf3 measurement #3
        measurement3 = ("VM to VM in same project on different HW nodes via "
                        "Private IP, {} threads; {}"
                        "".format(threads, iperf_utility))
        logger.info("Doing '{}' measurement...".format(measurement3))
        if iperf_utility == "iperf3":
            result3 = transport1.exec_command(
                '{} -c {} -P {} -t {} | grep sender | tail -n 1'
                ''.format(iperf_utility, vm_info[2]['private_address'],
                          threads, iperf_time))
            res3 = (b" ".join(result3.split()[-4:-2:])).decode('utf-8')
        else:
            iperf_utility = "iperf"
            result3 = transport1.exec_command(
                '{} -c {} -P {} -t {} | tail -n 1'.format(
                    iperf_utility, vm_info[2]['private_address'], threads,
                    iperf_time))
            res3 = (b" ".join(result3.split()[-2::])).decode('utf-8')
        logger.info("Result #3 is {}".format(res3))
        table_rows.append([measurement3,
                           "{}".format(pair[0]),
                           "{}".format(pair[1]),
                           "{}, {}".format(mtus[0], mtus[2]),
                           "{}".format(res3)])

        # Do iperf3 measurement #4
        measurement4 = ("VM to VM in same project via Floating IP and VMs "
                        "are on different nodes, 1 thread; iperf3")
        logger.info("Doing '{}' measurement...".format(measurement4))
        result4 = transport1.exec_command(
            'iperf3 -c {} -t {} | grep sender | tail -n 1'.format(
                vm_info[2]['fip'], iperf_time))
        res4 = (b" ".join(result4.split()[-4:-2:])).decode('utf-8')
        logger.info("Result #4 is {}".format(res4))
        table_rows.append([measurement4,
                           "{}".format(pair[0]),
                           "{}".format(pair[1]),
                           "{}, {}".format(mtus[0], mtus[2]),
                           "{}".format(res4)])

        # Do iperf3 measurement #5
        measurement5 = ("VM to VM in same project, different HW nodes and "
                        "each VM is connected to separate network which are "
                        "connected using Router via Private IP, 1 thread; "
                        "iperf3")
        logger.info("Doing '{}' measurement...".format(measurement5))
        result5 = transport1.exec_command(
            'iperf3 -c {} -t {} | grep sender | tail -n 1'.format(
                vm_info[3]['private_address'], iperf_time))
        res5 = (b" ".join(result5.split()[-4:-2:])).decode('utf-8')
        logger.info("Result #5 is {}".format(res5))
        table_rows.append([measurement5,
                           "{}".format(pair[0]),
                           "{}".format(pair[1]),
                           "{}, {}".format(mtus[0], mtus[3]),
                           "{}".format(res5)])

        logger.info("Drawing the table with iperf results...")
        result_table.add_rows(table_rows)
        sys.stdout.write('\n{}\n'.format(result_table.draw()))

        # Send the results to CSV file at reports/ directory
        helpers.create_test_result_table_csv_file(
            table_rows, request.node.name)

        logger.info("Removing VMs and FIPs...")
        for vm in vms:
            openstack_clients.compute.servers.delete(vm)
        logger.info("Removing FIPs...")
        for fip in fips:
            os_actions.delete_floating_ip(fip)
    except Exception as e:
        sys.stdout.write("\n{}".format(e))
        sys.stdout.write("\nSomething went wrong\n")
        if 'vms' in locals():
            logger.info("Removing VMs...")
            for vm in vms:
                openstack_clients.compute.servers.delete(vm)
            if 'fips' in locals():
                logger.info("Removing FIPs...")
                for fip in fips:
                    os_actions.delete_floating_ip(fip)
        else:
            sys.stdout.write("\nSkipping cleaning, VMs were not created")
        pytest.fail("Something went wrong")
