#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
from datetime import datetime
from urllib import parse as urlparse

from si_tests import logger
from si_tests import settings
from si_tests.managers import bootstrap_manager
from si_tests.utils import utils
from si_tests.utils.certs import CertManager
from si_tests.utils import waiters
from si_tests.utils.utils import Provider

LOG = logger.logger


def test_set_cache_custom_cert(kaas_manager, show_step):
    """Add custom cache cert to the regional cluster.

    Scenario:
        1. Collecting env data
        2. Init bootstrap using SI_CONFIG
        3. Generate mcc-cache TLS settings
        4. Get time stamped before applying cert
        5. Apply mcc-cache TLS settings
        6. Validate mcc-cache connection on child clusters

    """

    show_step(1)
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    show_step(2)
    bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()

    host = urlparse.urlsplit(cluster.mcc_cache).hostname
    ips = None
    dns_names = None
    if utils.is_valid_ip(host):
        ips = [host]
    else:
        dns_names = [host]

    show_step(3)
    app = "cache"
    ca_dir_suffix = datetime.now().strftime('%d-%-H-%-M')
    ca_path = f'{app}-{ca_dir_suffix}'
    cert_dir = cluster.name

    cert_pem, key_pem, ca_pem = bootstrap.generate_cert(ips=ips, dns_names=dns_names,
                                                        ca_path=ca_path, cert_dir=cert_dir)
    cert_manager = CertManager(kaas_manager=kaas_manager)

    show_step(4)
    child_clusters = kaas_manager.get_child_clusters_in_region(
        cluster.region_name)
    for child_cluster in child_clusters:
        machines = child_cluster.get_machines()
        datetime_cmd = "date -u +'%Y-%m-%d %H:%M:%S'"
        time_before_update = machines[0].exec_pod_cmd(datetime_cmd)['logs']

    show_step(5)
    cert_manager.apply_cert_for_app(app=app, cluster=cluster, hostname=host,
                                    cert_pem=cert_pem, key_pem=key_pem, ca_pem=ca_pem)

    cluster.check.check_cluster_readiness()

    show_step(5)
    for child_cluster in child_clusters:
        child_cluster_provider = child_cluster.provider
        waiters.wait(
            lambda: all(child_cluster.check.check_expected_phases_on_machines(
                machines, time_before_update, 'reconfigure')) is True,
            timeout=600*2 if child_cluster_provider == Provider.baremetal else 600,
            interval=20,
            timeout_msg="Reconfigure phase started not on all machines",
        )

        child_cluster.check.validate_machines_mcc_ca(ca=ca_pem)
        child_cluster.check.check_machines_status("Ready")

        # Negative check
        # after applying the certificate, the cluster child should not have a Deploy status
        assert (all(value is False for value in child_cluster.check.check_expected_phases_on_machines(
                machines, time_before_update, 'deploy'))),\
            'One of the machines has been in deploy status since the certificate was applied.'

        child_cluster.check.check_cluster_readiness()
