import random
import string
import pytest

from si_tests import logger
from si_tests.lcm.openstack_lcm.policies import Policy
from si_tests.utils.utils import verify

LOG = logger.logger

letters = string.ascii_lowercase
# creating an unique part of a network name to delete it after the test
name_postfix = ''.join(random.choice(letters) for i in range(5))


@pytest.fixture(scope="function")
def delete_resources(openstack_client_manager):
    """Function to delete the networks that were created during the test"""

    # executing test_crud_operations_network_policy
    yield

    # networks deletion
    network_list = openstack_client_manager.network.list([])
    for network in network_list:
        if name_postfix in network.get('Name'):
            openstack_client_manager.network.delete([network.get('ID')])


@pytest.mark.usefixtures('delete_resources')
def test_crud_operations_network_policy(openstack_client_manager, policy_user):
    """Test CRUD operations for network policy
       https://mirantis.testrail.com/index.php?/cases/view/4963696

       Parameters required for test execution:
         - KUBECONFIG

       Setup environment:
       Create user with role member
    """
    user_name, password = policy_user
    os_service = 'neutron'
    os_pod = 'neutron-server'
    policy = Policy(os_service, os_pod)
    check_policy_file = ['/bin/sh', '-c', ' cat /etc/neutron/policy.d/02-custom.yaml']
    http_forbidden_exception = "Exception: 403"

    try:
        # checking than the policy defining file is not changed before the test
        get_policy_file_before_update = ['/bin/sh', '-c',
                                         r' grep -irn "create_network:\|get_network:"'
                                         ' /etc/neutron/policy.d/02-custom.yaml']
        policy.check_policy_absent_before_update(get_policy_file_before_update)

        LOG.info("Step 1: Checking that policies are absent. Admin and user can create a network and see network"
                 " details")
        step = 1

        # creating a network with the admin credentials
        network = openstack_client_manager.network.create([f"sitest_net_no_policy_admin_{name_postfix}"],
                                                          combined_output=True)
        verify(http_forbidden_exception not in network["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. Admin can create a new network")

        # checking the possibility to get the network details with the admin credentials
        network_details = openstack_client_manager.network.show([f"sitest_net_no_policy_admin_{name_postfix}"],
                                                                combined_output=True)
        verify(http_forbidden_exception not in network_details["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. Admin can see the network details")

        # creating a network with the user credentials
        network = openstack_client_manager.network.create(
            [f"sitest_net_no_policy_user_{name_postfix}", f"--os-username={user_name}", f"--os-password={password}"],
            combined_output=True)
        verify(http_forbidden_exception not in network["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. User can create a new network")

        # checking the possibility to get the network details with the user credentials
        network_details = openstack_client_manager.network.show(
            [f"sitest_net_no_policy_user_{name_postfix}", f"--os-username={user_name}", f"--os-password={password}"],
            combined_output=True)
        verify(http_forbidden_exception not in network_details["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. User can see the network details")

        LOG.info("Step 2: Adding new rules to OSDPL file")
        step = 2

        # changing the policy defining file, no one will be able to create a network, no one will be able to watch
        # network details, except admin with their superuser rights
        policy_add_role = {'create_network': '!', 'get_network': '!'}
        policy.update_policy_osdpl(policy_add_role, step)

        LOG.info("Step 3: Checking that the policies were added to Neutron API pod")
        step = 3

        # checking that the policies were updated
        policy.wait_policy_updated_on_pod(check_policy_file, policy_add_role, step)

        LOG.info("Step 4: The policies were added. Checking that only admin can create a new network and see"
                 " network details")
        step = 4

        # creating a network with the admin credentials
        network = openstack_client_manager.network.create([f"sitest_net_add_policy_admin_{name_postfix}"],
                                                          combined_output=True)
        verify(http_forbidden_exception not in network["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies disallow everyone except the admin to create a new network. Admin can create a"
               f" new network")

        # checking the possibility to get the network details with the admin credentials
        network_details = openstack_client_manager.network.show([f"sitest_net_add_policy_admin_{name_postfix}"],
                                                                combined_output=True)
        verify(http_forbidden_exception not in network_details["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies disallow everyone except the admin to see network details. Admin can see"
               f" the network details")

        # checking the impossibility of network creating with the user credentials
        error_output = openstack_client_manager.network.create(
            [f"sitest_net_add_policy_user_{name_postfix}", f"--os-username={user_name}", f"--os-password={password}"],
            combined_output=True)
        verify(http_forbidden_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies disallow everyone except the admin to create a new network. User cannot create"
               f" a new network")

        LOG.info("Step 5: Updating policies. Checking that the policies were added to Neutron API pod")
        step = 5

        # changing the policy defining file, only admin will be able to create a network, everyone will be able to
        # watch network details
        policy_update_role = {'create_network': 'rule:context_is_admin', 'get_network': '@'}
        policy.update_policy_osdpl(policy_update_role, step)
        policy.wait_policy_updated_on_pod(check_policy_file, policy_update_role, step)

        LOG.info("Step 6: The policies were updated. Checking that only admin can create a new network, everyone can"
                 " see network details")
        step = 6

        # creating a network with the admin credentials
        network = openstack_client_manager.network.create([f"sitest_net_updated_policy_admin_{name_postfix}"],
                                                          combined_output=True)
        verify(http_forbidden_exception not in network["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies allow admin to create a new network, admin can create a new network")

        # checking the possibility to get the network details with the admin credentials
        network_details = openstack_client_manager.network.show([f"sitest_net_updated_policy_admin_{name_postfix}"],
                                                                combined_output=True)
        verify(http_forbidden_exception not in network_details["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies allow admin to see network details, admin can see the network details")

        # checking the impossibility of network creating with the user credentials
        error_output = openstack_client_manager.network.create(
            [f"sitest_net_updated_policy_user_{name_postfix}", f"--os-username={user_name}",
             f"--os-password={password}"], combined_output=True)
        verify(http_forbidden_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies allow only admin to create a new network, user cannot create a new network")

        # checking the possibility to get the network details with the user credentials
        network_details = openstack_client_manager.network.show(
            [f"sitest_net_no_policy_user_{name_postfix}", f"--os-username={user_name}",
             f"--os-password={password}"], combined_output=True)
        verify(http_forbidden_exception not in network_details["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies allow everyone to see network details, user can see the network details")

    finally:
        try:
            LOG.info("Step 7: Deleting policies. Checking that the policies were deleted and"
                     " the result is the same as in the first step")
            step = 7

            # changing the policy defining file to the pre-test condition
            policy_not_role = None
            policy.update_policy_osdpl(policy_not_role, step)
            policy.wait_policy_updated_on_pod(check_policy_file, {}, step)

            # creating a network with the admin credentials
            network = openstack_client_manager.network.create([f"sitest_net_deleted_policy_admin_{name_postfix}"],
                                                              combined_output=True)
            verify(http_forbidden_exception not in network["stderr"],
                   f"[Step {step}]: Policies are not required for the test",
                   f"[Step {step}]: Policies are absent. Admin can create a new network")

            # checking the possibility to get the network details with the admin credentials
            network_details = openstack_client_manager.network.show([
                f"sitest_net_deleted_policy_admin_{name_postfix}"],
                combined_output=True)
            verify(http_forbidden_exception not in network_details["stderr"],
                   f"[Step {step}]: Policies are not required for the test",
                   f"[Step {step}]: Policies are absent. Admin can see the network details")

            # creating a network with the user credentials
            network = openstack_client_manager.network.create(
                [f"sitest_net_deleted_policy_user_{name_postfix}", f"--os-username={user_name}",
                 f"--os-password={password}"], combined_output=True)
            verify(http_forbidden_exception not in network["stderr"],
                   f"[Step {step}]: Policies are not required for the test",
                   f"[Step {step}]: Policies are absent. User can create a new network")

            # checking the possibility to get the network details with the user credentials
            network_details = openstack_client_manager.network.show(
                [f"sitest_net_deleted_policy_user_{name_postfix}", f"--os-username={user_name}",
                 f"--os-password={password}"], combined_output=True)
            verify(http_forbidden_exception not in network_details["stderr"],
                   f"[Step {step}]: Policies are not required for the test",
                   f"[Step {step}]: Policies are absent. User can see the network details")

        finally:
            LOG.info("Deleting created resources")
