import ast
import os
import uuid
import paramiko
import pytest

from si_tests import settings
from si_tests.deployments.utils import (
    commons,
    kubectl_utils,
    file_utils,
)
from si_tests.utils import (waiters, utils)


def test_manila_shared_file_system_nfs(openstack_client_manager, request):
    """
    1.Create heat stack with manila shared file system (NFS)
    2.Check created stack status
    3.Check created manila NFS mount
    """

    kubectl = kubectl_utils.Kubectl()
    ns = settings.OSH_NAMESPACE
    osdpl_name = settings.OSH_DEPLOYMENT_NAME
    command = '{} -o yaml'.format(osdpl_name)
    osdpl = kubectl.get('osdpl', command, ns).result_yaml
    if 'shared-file-system' not in osdpl['spec']['features']['services']:
        message = "Manila service is not enabled"
        commons.LOG.warning(message)
        pytest.skip(message)
    elif osdpl['spec']['features']['neutron'].get('backend') == 'tungstenfabric':
        message = "Manila is not supported with TF networking"
        commons.LOG.warning(message)
        pytest.skip(message)
    else:
        template = file_utils.join(
            os.path.dirname(os.path.abspath(__file__)), "templates/manila.yaml"
        )

    stack_name = f"manila-{uuid.uuid4()}"

    openstack_client_manager.create_stack(request, stack_name, template, finalizer=False)

    stack_data = openstack_client_manager.stack.show([stack_name])

    commons.LOG.info("Try to check NFS mount and its CRUD operations")
    server_floating = None
    share_mount_path_list = ''
    for output in stack_data['outputs']:
        if output['output_key'] == 'server_fip':
            server_floating = output['output_value']
        elif output['output_key'] == 'share_path':
            share_mount_path_list = output['output_value']

    assert server_floating, \
        ("Unable to get IP address from stack output.\n"
         f"Floating ip of VM: {server_floating}\n")

    assert share_mount_path_list, \
        ("Unable to get mount path from stack output.\n"
         f"Returned mount path: {share_mount_path_list}\n")

    share_mount_path = ast.literal_eval((share_mount_path_list))[0]

    check_nfs_mount_cmd = f"""
sudo mount -t nfs {share_mount_path} /mnt/manila_share
sleep 4
dd bs=1024 count=10485 </dev/urandom > /mnt/manila_share/testfile10Mb
    """

    waiters.wait_tcp(server_floating, 22, timeout=900,
                     timeout_msg=f"Waiting {server_floating} port 22 timed out")
    try:
        mount_result = utils.basic_ssh_command(check_nfs_mount_cmd, server_floating, 'ubuntu', 'qalab')
    except (paramiko.ssh_exception.SSHException, TimeoutError):
        commons.LOG.error(
            "Unable connect to instance by SSH. Floating address is unreachable."
        )
        raise Exception("Unable connect to instance by SSH. Floating address unreachable.")

    assert mount_result.exit_code == 0, "Unable to check NFS mount for manila share"

    commons.LOG.info(f"Delete heat stack {stack_name}")
    openstack_client_manager.stack.delete([stack_name, "-y", "--wait"])
