import ast
import os
import time

from si_tests import settings
from si_tests.deployments.deploy_workloads import WorkloadBase
from si_tests.deployments.utils import kubectl_utils, commons, file_utils
from si_tests.deployments.utils.namespace import NAMESPACE
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.utils import utils


class RefappBase(WorkloadBase):
    def __init__(self, client, kubeconfig=None):
        self.ostcm = client
        self.kubeconfig = os.path.abspath(kubeconfig or os.environ.get("KUBECONFIG"))
        self.kubectl = kubectl_utils.Kubectl(NAMESPACE.openstack, self.kubeconfig)
        self._url = None
        self._db_passwords = None
        self.os_manager = OpenStackManager(kubeconfig=self.kubeconfig)

    def _request(self, request):
        response = self.ostcm.exec(request)
        try:
            return ast.literal_eval(response)
        except Exception:
            commons.LOG.error(f"Request {request} failed with: {response}")
            return {}

    def get_node_identifier(self):
        request = ["curl", "-s", self.url]
        return self._request(request).get("host", None)

    def create_record(self, data='{"record":{}}'):
        request = [
            "curl",
            "-X",
            "POST",
            "-s",
            f"{self.url}/records",
            "-H",
            "Content-Type: application/json",
            "--data",
            data,
        ]
        return self._request(request).get("record", {})

    def get_record_by_id(self, record_id):
        url = f"{self.url}/records/{record_id}"
        return self._request(["curl", "-s", url]).get('record', {})

    def get_records(self):
        url = f"{self.url}/records"
        request = ["curl", "-s", url]
        return self._request(request).get('records', {})

    def check_get_node_identifier(self, collect=False, requests_count=50):
        hosts = set()
        response_count = 0
        for counter in range(requests_count):
            host = self.get_node_identifier()
            if host:
                hosts.add(host)
                response_count += 1
            if len(hosts) == 3 and not collect:
                break
            time.sleep(1)
        else:
            if collect:
                return requests_count, response_count
            assert False, "App did not have 3 distinct GET responses " \
                          "from APIs within 50 retries"

    def check_create_record(self, collect=False, requests_count=50):
        records_file = settings.OPENSTACK_REFAPP_RECORDS_FILE
        hosts = set()
        records = list()
        response_count = 0
        for counter in range(requests_count):
            record = self.create_record()
            if record:
                response_count += 1
                hosts.add(record["host"])
                records.append(record)
                file_utils.save_to_yaml(records, records_file, mode='a')

            if len(hosts) == 3 and not collect:
                break
            time.sleep(1)
        else:
            if collect:
                return requests_count, response_count
            assert False, "App did not have 3 distinct POST responses " \
                          "from APIs within 50 retries"

    def check_get_record_by_id(self, collect=False):
        records_file = settings.OPENSTACK_REFAPP_RECORDS_FILE
        records = (
            file_utils.get_yaml_content(
                records_file, file_not_found_fail=False
            )
            or []
        )
        request_count = len(records)
        response_count = 0

        for record in records:
            record_by_id = self.get_record_by_id(record_id=record["id"])
            if collect:
                if record == record_by_id:
                    response_count += 1
            else:
                assert record == record_by_id, f"Expected {record}, got {record_by_id}"
        return request_count, response_count

    def check(self):
        """Check working capacity of RefApp.
            - Check already created records
            - Check that RefApp did have 3 distinct GET responses
            - Check that RefApp did have 3 distinct POST responses
            - Save created records.
        """
        self.ostcm.reload_client()  # After LCM operations, old keystone pod object may be outdated, need refresh
        commons.LOG.info("Check get RefApp records by id if "
                         "file with records exists.")
        self.check_get_record_by_id()
        commons.LOG.info("Check GET RefApp requests.")
        self.check_get_node_identifier()
        commons.LOG.info("Check POST RefApp requests.")
        self.check_create_record()
        commons.LOG.info("Refapp check finished.")

    def collect_statistic(self):
        retries = settings.COLLECT_REFAPP_STATISTIC_RETRIES
        commons.LOG.info("Starting refapp statistic collection ...")
        for i in range(retries):
            get_host_req, get_host_resp = self.check_get_node_identifier(
                collect=True, requests_count=3
            )
            create_req, create_resp = self.check_create_record(
                collect=True, requests_count=1
            )
            id_req, id_resp = self.check_get_record_by_id(collect=True)
            commons.LOG.warning(
                "RefApp collected requests/responses counts:\n"
                f"Get host requests/responses: {get_host_req}/{get_host_resp}\n"
                f"Create record requests/responses: {create_req}/{create_resp}\n"
                f"Get record by ID requests/responses: {id_req}/{id_resp}",
            )
            commons.LOG.warning("RETRY %s", i)

    @property
    def db_passwords(self):
        if self._db_passwords is None:
            self._db_passwords = {
                db: utils.gen_random_password(16) for db in ("admin", "app")
            }
        return self._db_passwords
