#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.clients.k8s.base import K8sBaseManager
import si_tests.utils.waiters as helpers
from si_tests import logger
from kubernetes.client.rest import ApiException

LOG = logger.logger


class K8sStatefulSet(K8sNamespacedResource):
    resource_type = 'statefulset'

    def _read(self, **kwargs):
        return self._manager.api.read_namespaced_stateful_set(
            self.name, self.namespace, **kwargs)

    def _create(self, body, **kwargs):
        return self._manager.api.create_namespaced_stateful_set(
            self.namespace, body, **kwargs)

    def _patch(self, body, **kwargs):
        return self._manager.api.patch_namespaced_stateful_set(
            self.name, self.namespace, body, **kwargs)

    def _replace(self, body, **kwargs):
        return self._manager.api.replace_namespaced_stateful_set(
            self.name, self.namespace, body, **kwargs)

    def _delete(self, **kwargs):
        self._manager.api.delete_namespaced_stateful_set(
            self.name, self.namespace, **kwargs)

    def patch_scale(self, body, **kwargs):
        return self._manager.api.patch_namespaced_stateful_set_scale(
            self.name, self.namespace, body, **kwargs)

    def replace_scale(self, body, **kwargs):
        return self._manager.api.replace_namespaced_stateful_set_scale(
            self.name, self.namespace, body, **kwargs)

    def read_scale(self, **kwargs):
        return self._manager.api.read_namespaced_stateful_set_scale(
            self.name, self.namespace, **kwargs)

    def read_status(self, **kwargs):
        return self._manager.api.read_namespaced_stateful_set_status(
            self.name, self.namespace, **kwargs)

    def wait_readiness(self, timeout=500, interval=10):
        desired_replicas = self._read().spec.replicas

        def detect_statefulset_progressing():
            if not self.read().status:
                raise Exception("Statefulset {0} in {1} does not have status"
                                "yet.".format(self.name, self.namespace))
            ready_replicas = self.read().status.ready_replicas
            if ready_replicas is None:
                ready_replicas = 0
            if ready_replicas != desired_replicas:
                LOG.info("Ready replicas {0} of {1} in StatefulSet {2}. "
                         "Waiting...".format(ready_replicas,
                                             desired_replicas, self.name))
                raise Exception("Statefulset {0} in namespace {1} "
                                "is not ready yet.".format(
                                 self.name, self.namespace))
            LOG.info("Statefulset {0} in namespace {1} is ready.".format(
                self.name, self.namespace))
        LOG.info("Wait {0} seconds until Statefulset replicas "
                 "will ready".format(timeout))
        helpers.wait_pass(detect_statefulset_progressing,
                          timeout=timeout,
                          interval=interval,
                          expected=(Exception, ApiException))
        return self

    @property
    def desired_replicas(self):
        return self.read().spec.replicas

    @property
    def ready_replicas(self):
        status = self.read().status
        if status:
            return status.ready_replicas
        else:
            LOG.info("Status of statefulset {0}/{1} not populated yet".format(self.namespace, self.name))
            # If status does not present - no ready replicas expected
            return 0

    @property
    def data(self):
        """Returns dict of k8s object

        Data contains keys like api_version, kind, metadata,
        spec, status or items
        """
        return self.read().to_dict()

    @property
    def ready(self):
        obj = self.data
        if obj["status"]["observed_generation"] >= obj["metadata"]["generation"]:
            if (obj["status"].get("updated_replicas") or 0) == obj['spec']['replicas']:
                if (obj["status"].get("ready_replicas") or 0) == obj['spec']['replicas']:
                    return True

        LOG.debug("StatefulSet {}/{} \nstatus.observed_generation: '{}', metadata.generation: '{}'\n"
                  "status.updated_replicas: '{}', expected: '{}' \nstatus.ready_replicas: '{}', expected: '{}'"
                  "".format(self.namespace, self.name,
                            obj["status"]["observed_generation"], obj["metadata"]["generation"],
                            obj["status"].get("updated_replicas"), obj['spec']['replicas'],
                            obj["status"].get("ready_replicas"), obj['spec']['replicas']))
        return False


class K8sStatefulSetManager(K8sBaseManager):
    resource_class = K8sStatefulSet

    @property
    def api(self):
        return self._cluster.api_apps

    def _list(self, namespace, **kwargs):
        return self.api.list_namespaced_stateful_set(namespace, **kwargs)

    def _list_all(self, **kwargs):
        return self.api.list_stateful_set_for_all_namespaces(**kwargs)
