#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from si_tests.clients.k8s import base
from si_tests import logger
from si_tests.utils import waiters

LOG = logger.logger


class V2HelmRelease(base.BaseModel):
    pass


class V2HelmReleaseList(base.BaseModelList):
    pass


class HelmRelease(base.K8sNamespacedResource):
    resource_type = 'helmrelease'
    model = V2HelmRelease

    @property
    def ready(self):
        result = self.get_conditions(verbose=True)
        if result['not_ready']:
            return False
        else:
            return True

    def get_conditions(self, verbose):
        result = {
            'ready': [],
            'not_ready': []
        }
        data_status = self.data.get('status') or {}
        conditions = data_status.get('conditions', [])
        if not conditions:
            result['not_ready'].append("No conditions in helmrelease status yet")

        for condition in conditions:
            if condition['status'] in ['True', 'False']:
                ready = eval(condition['status'])
            else:
                ready = False

            condition_type = condition.get('type')
            if ready:
                result['ready'].append(condition_type)
            else:
                result['not_ready'].append(condition_type)

            if verbose:
                condition_message = condition.get('message', '')
                LOG.info(f"Condition ({condition_type}) have message ({condition_message}) and "
                         f"status ({condition['status']})")
        if verbose:
            message = (f"{self.data['kind']} {self.namespace}/"
                       f"{self.data['metadata']['name']}: {result}")
            LOG.info(message)
        return result

    def _delete(self, async_del=True, timeout=300, body=None, **kwargs):
        LOG.info(f"Removing HelmRelease {self.namespace}/{self.name}")
        if not body:
            body = {}
        self._manager.api.delete_namespaced_custom_object(
            group=self._manager.resource_group,
            version=self._manager.resource_version,
            namespace=self.namespace,
            plural=self._manager.resource_plural,
            name=self.name,
            body=body,
            **kwargs)
        if not async_del:
            waiters.wait(lambda: not self._manager.present(self.name, self.namespace),
                         timeout=timeout,
                         timeout_msg=f"Timeout waiting for HelmRelease deletion {self.namespace}/{self.name}")
        LOG.info(f"HelmRelease {self.namespace}/{self.name} has been deleted.")


class HelmReleaseManager(base.K8sBaseManager):
    model = V2HelmReleaseList
    resource_class = HelmRelease
    resource_group = 'helm.toolkit.fluxcd.io'
    resource_version = 'v2'
    resource_plural = 'helmreleases'
