from abc import ABC, abstractmethod
from kubernetes.client.rest import ApiException
import pytest
import random
import yaml
import time
import base64

from kubernetes.client.exceptions import ApiException as k8sApiException
from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import waiters

LOG = logger.logger


class AbstractModule(ABC):
    @abstractmethod
    def get_config(self, *args, **kwargs):
        raise NotImplementedError("Abstract method")

    @abstractmethod
    def create(self, *args, **kwargs):
        raise NotImplementedError("Abstract method")

    @abstractmethod
    def delete(self, *args, **kwargs):
        raise NotImplementedError("Abstract method")


class BaseModule(AbstractModule):

    def __init__(self, cluster, ns):
        self.cluster = cluster
        self.ns = ns

    def get_config(self, name):
        result = self.ns.get_hostosconfiguration(name=name)
        if not result:
            raise Exception(f"HostOSConfiguration '{name}' not found")
        return result

    def create(self, name, configs, machines_type='control'):
        """Create HostOSConfiguration"""
        match_labels = self.get_match_labels(machines_type)

        hostoscfg_data = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "HostOSConfiguration",
            "metadata": {
                "name": name,
                "namespace": self.cluster.namespace,
            },
            "spec": {
                "configs": configs,
                "machineSelector": {
                    "matchLabels": match_labels,
                }
            }
        }
        hostoscfg = self.ns.create_hostosconfiguration_raw(hostoscfg_data)
        return hostoscfg

    def get_cleanup_bad_values(self) -> dict:
        return dict()

    def patch(self, name, configs, machines_type='control'):
        """Create HostOSConfiguration"""
        match_labels = self.get_match_labels(machines_type)

        hostoscfg_data = {
            "spec": {
                "configs": configs,
                "machineSelector": {
                    "matchLabels": match_labels,
                }
            }
        }
        obj = self.get_config(name)
        hostoscfg = obj.patch(body=hostoscfg_data)
        return hostoscfg

    def delete(self, name, timeout=60 * 10, interval=10):
        """Delete HostOSConfiguration"""
        if not self.ns.hostosconfiguration_is_present(name=name):
            LOG.warning(f"HostOSConfiguration {name} not found, so will not be removed")
            return
        existing_config = self.ns.get_hostosconfiguration(name=name)
        LOG.info(f"Deleting HostOSConfiguration '{existing_config.name}'")
        existing_config.delete(async_req=True)
        timeout_msg = f"HostOSConfiguration {name} was not deleted in {timeout}s"
        waiters.wait(lambda: not bool(self.ns.hostosconfiguration_is_present(name=name)),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)

    def get_match_labels(self, machine_type='control'):
        if self.cluster.is_management:
            # Management cluster have only 'control' node type
            match_labels_by_types = {
                'control': {
                    "cluster.sigs.k8s.io/control-plane": "true",
                },
                'worker': {
                    "kaas.mirantis.com/provider": "baremetal",
                },
            }
            assert machine_type in match_labels_by_types, (
                f"Machine type '{machine_type}' is not supported, please update the test")
            match_labels = match_labels_by_types.get(machine_type)
        else:
            match_labels_by_types = {
                'control': {
                    "hostlabel.bm.kaas.mirantis.com/controlplane": "controlplane",
                },
                'worker': {
                    "hostlabel.bm.kaas.mirantis.com/worker": "worker",
                },
                'storage': {
                    "hostlabel.bm.kaas.mirantis.com/storage": "storage",
                },
            }
            assert machine_type in match_labels_by_types, (
                f"Machine type '{machine_type}' is not supported, please update the test")
            match_labels = match_labels_by_types.get(machine_type)

        match_labels['cluster.sigs.k8s.io/cluster-name'] = self.cluster.name
        return match_labels

    def get_machines_distro_version(self, machines_type='control'):
        result = {}
        match_labels = self.get_match_labels(machines_type)
        for m in self.cluster.get_machines():
            if m.has_machine_labels(match_labels):
                result[m.name] = m.get_distribution()
        return result

    def apply_new_config(self, name, configs, machines_type='control', check_result=True):
        # Remember LCMMachines timestamps before creating HostOSConfiguration
        lcmmachines_timestamps_before = self.cluster.get_cluster_lcmmachines_timestamps()

        hostoscfg = self.create(name=name, configs=configs, machines_type=machines_type)

        # Wait for the selected Machines in the hostosconfiguration status
        LOG.info("Check that machines from hostosconfiguration status field have labels used for machineSelector")
        self.cluster.check.check_hostosconfig_machine_selector(hostoscfg)
        LOG.info("Check that new items added into machineTypes in LCMCluster")
        self.cluster.check.wait_lcmcluster_day2_machinetypes(hostoscfg)
        LOG.info("Check that new items added into stateItems in LCMMachine")
        self.cluster.check.wait_lcmmachine_day2_stateitems(hostoscfg, lcmmachines_timestamps_before)
        LOG.info("Check the result on the machines")
        machine_names = self.cluster.check.get_hostosconfig_machines_status(hostoscfg)
        if check_result:
            for module_spec in hostoscfg.data.get('spec', {}).get('configs', []):
                self.cluster.check.check_day2_module_results(module_spec['module'], hostoscfg, machine_names)

    def apply_new_config_negative(self, name, configs, machines_type='control'):

        # Remember LCMMachines timestamps before creating HostOSConfiguration
        lcmmachines_timestamps_before = self.cluster.get_cluster_lcmmachines_timestamps()

        hostoscfg = self.create(name=name, configs=configs, machines_type=machines_type)

        # Wait for the selected Machines in the hostosconfiguration status
        LOG.info("Check that machines from hostosconfiguration status field have labels used for machineSelector")
        self.cluster.check.check_hostosconfig_machine_selector(hostoscfg)
        LOG.info("Check that new items added into machineTypes in LCMCluster")
        self.cluster.check.wait_lcmcluster_day2_machinetypes(hostoscfg)

        LOG.info("Check that new items added into stateItems in LCMMachine but stateItemStatuses contain errors")
        self.cluster.check.wait_lcmmachine_day2_stateitems(
            hostoscfg, lcmmachines_timestamps_before, expected_error_result=True)

    def apply_patch(self, name, configs, machines_type='control', check_result=True):

        # Remember LCMMachines timestamps before creating HostOSConfiguration
        lcmmachines_timestamps_before = self.cluster.get_cluster_lcmmachines_timestamps()

        hostoscfg = self.patch(name=name, configs=configs, machines_type=machines_type)

        # Wait for the selected Machines in the hostosconfiguration status
        LOG.info("Check that machines from hostosconfiguration status field have labels used for machineSelector")
        self.cluster.check.check_hostosconfig_machine_selector(hostoscfg)
        LOG.info("Check that new items added into machineTypes in LCMCluster")
        self.cluster.check.wait_lcmcluster_day2_machinetypes(hostoscfg)
        LOG.info("Check that new items added into stateItems in LCMMachine")
        self.cluster.check.wait_lcmmachine_day2_stateitems(hostoscfg, lcmmachines_timestamps_before)
        LOG.info("Check the result on the machines")
        machine_names = self.cluster.check.get_hostosconfig_machines_status(hostoscfg)
        if check_result:
            for module_spec in hostoscfg.data.get('spec', {}).get('configs', []):
                self.cluster.check.check_day2_module_results(module_spec['module'], hostoscfg, machine_names)


class SysctlModule(BaseModule):
    """Test 'sysctl' module"""

    def create(self, *args, **kwargs):
        return super().create(*args, **kwargs)

    def delete(self, *args, **kwargs):
        return super().delete(*args, **kwargs)

    def get_values_to_create(self):
        return {
            "filename": "si_tests",
            "cleanup_before": True,
            "options": {
                "kernel.printk_delay": str(random.randint(10, 30)),  # string value
                "fs.nr_open": str(1048576 + random.randint(1000, 1999))
            }
        }

    def get_new_values_to_create(self):
        return {
            "filename": "si_tests",
            "cleanup_before": True,
            "options": {
                "fs.pipe-user-pages-soft": str(random.randint(17000, 18000)),
            }
        }

    def get_new_values_to_update(self):
        return {
            "filename": "si_tests",
            "cleanup_before": True,
            "options": {
                "kernel.printk_delay": str(random.randint(50, 90)),  # string value
                "fs.pipe-user-pages-soft": str(random.randint(18100, 19000)),  # override some old value
                "net.ipv4.route.mtu_expires": str(600 + random.randint(10, 50)),  # add new value
            }
        }

    def get_values_for_stateitemsoverwrites_after_hoc_deletion_test(self):
        return {
            "filename": "si_tests",
            "cleanup_before": True,
            "options": {
                "kernel.watchdog_thresh": str(random.randint(11, 19)),
            }
        }

    def get_bad_values(self):
        return {
            "filename": "si_tests",
            "cleanup_before": True,
            "options": {
                "kernel.aaa.bbb": str(random.randint(1000, 1999)),
            }
        }

    def get_cleanup_bad_values(self) -> dict:
        return {
            "filename": "si_tests",
            "cleanup_before": True,
            "options": {},
        }

    def get_secret_values_to_create(self):
        sysctl_opts = {"kernel.printk_delay": str(random.randint(700, 799))}
        return {
            "filename": "si_tests",
            "cleanup_before": True,
            "options": base64.b64encode(yaml.dump(sysctl_opts).encode('ascii')).decode()
        }


class PackageModule(BaseModule):
    """Test 'package' module"""

    def create(self, *args, **kwargs):
        return super().create(*args, **kwargs)

    def delete(self, *args, **kwargs):
        return super().delete(*args, **kwargs)

    def get_values_to_create(self):
        return {
            "packages": [
                {
                    "name": "elinks",
                },
            ],
        }

    def get_new_values_to_create(self):
        return {
            "packages": [
                {
                    "name": "lynx",
                },
            ],
        }

    def get_new_values_to_update(self):
        return {
            "packages": [
                {
                    "name": "htop",
                    "state": "absent",
                    "purge": "yes",
                },
                {
                    "name": "tshark",
                },
            ],
        }

    def get_values_for_stateitemsoverwrites_after_hoc_deletion_test(self):
        return {
            "packages": [
                {
                    "name": "bc",
                },
            ],
        }

    def get_bad_values(self):
        return {
            "packages": [
                {
                    "name": "system-integration-2024-" + str(random.randint(10, 99)),
                },
            ],
        }

    def get_secret_values_to_create(self):
        return {
            "packages": [
                {
                    "name": base64.b64encode("dnstop".encode('ascii')).decode(),
                },
            ],
        }


class TmpfileModule(BaseModule):
    """Test 'package' module"""

    def create(self, *args, **kwargs):
        return super().create(*args, **kwargs)

    def delete(self, *args, **kwargs):
        return super().delete(*args, **kwargs)

    def get_values_to_create(self):
        return {
            "filename": "foobar-" + str(int(time.time())),
        }

    def get_new_values_to_create(self):
        return {
            "filename": "spameggs-" + str(int(time.time())),
        }

    def get_new_values_to_update(self):
        return {
            "filename": "updated_tmpfile-" + str(int(time.time())),
        }

    def get_values_for_stateitemsoverwrites_after_hoc_deletion_test(self):
        return {
            "filename": "hoc_deletion_tmpfile-" + str(int(time.time())),
        }

    def get_bad_values(self):
        return {
            "filename": "",
        }

    def get_secret_values_to_create(self):
        return {
            "filename": base64.b64encode(f"secret_tmpfile-{str(int(time.time()))}".encode('ascii')).decode(),
        }


class IrqBalanceModule(BaseModule):
    """Test 'IrqBalance' module"""

    def create(self, *args, **kwargs):
        return super().create(*args, **kwargs)

    def delete(self, *args, **kwargs):
        return super().delete(*args, **kwargs)

    def get_values_to_create(self):
        return {
            "banned_cpulist": "3,5",
        }

    def get_new_values_to_create(self):
        return {
            "banned_cpulist": "4,6",
        }

    def get_secret_values_to_create(self):
        return {
            "update_apt_cache": False,
        }

    def get_new_values_to_update(self):
        return {
            "banned_cpulist": "2,4",
            "args": "--journal",
        }

    def get_bad_values(self):
        return {
            "args": "--policyscript=/etc/default/irqbalance-numa-nonexisting-script.sh",
            "policy_script": "!/bin/bash echo 'numa_node=-1'",
            "policy_script_filepath": "/non/existing/folder/irqbalance-numa.sh",
        }

    def get_values_for_stateitemsoverwrites_after_hoc_deletion_test(self):
        return {
            "banned_cpulist": "1,7",
        }


class GrubSettingsModule(BaseModule):
    """Test 'grub_settings' module"""

    def create(self, *args, **kwargs):
        return super().create(*args, **kwargs)

    def delete(self, *args, **kwargs):
        return super().delete(*args, **kwargs)

    def get_values_to_create(self):
        return {
            "options": {
                "grub_timeout": random.randint(10, 20),
                "grub_cmdline_linux": [
                    "cgroup_enable=memory",
                    "intel_iommu=off",
                ],
            },
            "disable_reboot_request": False,
        }

    def get_new_values_to_create(self):
        return {
            "options": {
                "grub_hidden_timeout": random.randint(30, 40),
                "grub_cmdline_linux_default": [
                    "cgroup.memory=nobpf",
                    "intel_iommu=off",
                    "panic=5",
                ]
            }
        }

    def get_new_values_to_update(self):
        return {
            "options": {
                "grub_timeout": random.randint(40, 50),
            }
        }

    def get_values_for_stateitemsoverwrites_after_hoc_deletion_test(self):
        return {
            "options": {
                "grub_disable_os_prober": True,
            }
        }

    def get_bad_values(self):
        return {
            "options": {
                "grub_hidden_timeout": random.randint(60, 70),
            },
            "grub_reset_to_defaults": True,
        }

    def get_secret_values_to_create(self):
        return {
            "options": {
                "grub_recordfail_timeout": base64.b64encode(str(random.randint(60, 70)).encode('ascii')).decode(),
            }
        }


class CPUShieldModule(BaseModule):
    """Test 'cpushield' module"""

    def create(self, *args, **kwargs):
        return super().create(*args, **kwargs)

    def delete(self, *args, **kwargs):
        return super().delete(*args, **kwargs)

    def get_values_to_create(self):
        return {
            "system_cpus": "0-10",
            "systemd_units_to_pin": [
                "system.slice",
                "user.slice",
                "kubepods.slice",
            ]
        }

    def get_new_values_to_create(self):
        return {
            "system_cpus": "0-10,12",
            "systemd_units_to_pin": [
                "system.slice",
                "user.slice",
            ]
        }

    def get_new_values_to_update(self):
        return {
            "system_cpus": "0-12",
            "system_mem_numas": "0",
            "systemd_units_to_pin": [
                "system.slice",
                "user.slice",
            ]
        }

    def get_values_for_stateitemsoverwrites_after_hoc_deletion_test(self):
        return {
            "system_cpus": "0-10,13",
            "systemd_units_to_pin": [
                "system.slice",
                "user.slice",
                "kubepods.slice",
            ]
        }

    def get_bad_values(self):
        return {
            "system_mem_numas": "2",
            "systemd_units_to_pin": [
                "system.slice",
                "user.slice",
                "kubepods.slice",
            ]
        }

    def get_secret_values_to_create(self):
        return {
            "system_cpus": base64.b64encode(str("0,2-10,14").encode('ascii')).decode(),
            "systemd_units_to_pin": [
                "system.slice",
                "user.slice",
                "kubepods.slice",
            ]
        }


def _get_testobject_class(module_name):
    _modules = {
        'sysctl': SysctlModule,
        'package': PackageModule,
        'tmpfile': TmpfileModule,
        'irqbalance': IrqBalanceModule,
        'grub_settings': GrubSettingsModule,
        'cpushield': CPUShieldModule,
    }
    assert module_name in _modules, (f"Undefined module name {module_name}")
    testobject_class = _modules[module_name]
    return testobject_class


def _check_custom_hostoscfg_module_status(name_prefix, kaas_manager):
    custom_hostoscfg_module = kaas_manager.get_hostosconfigurationmodule(name=name_prefix)
    if custom_hostoscfg_module.data.get('status', {}):
        return True
    return False


def get_mcc_module_versions():
    kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_PATH)
    mcc_modules_versions = {}
    hostoscfgmodules_list = kaas_manager.get_hostosconfigurationmodules()
    for hostoscfgmodules in hostoscfgmodules_list:
        hostoscfgmodules_data = hostoscfgmodules.data or dict()
        # Collect modules from spec
        hostoscfg_modules = hostoscfgmodules_data.get('spec', dict()).get('modules', list()) or list()
        for hostoscfg_module in hostoscfg_modules:
            hostoscfg_module_name = hostoscfg_module.get("name")
            hostoscfg_module_version = hostoscfg_module.get("version")
            mcc_modules_versions.setdefault(hostoscfg_module_name, []).append(hostoscfg_module_version)
    return mcc_modules_versions


def generate_module_data(use_custom_tmpfile_module=True, supported_module_versions=None, skip_module_versions=None):
    """Generates data structure which contains modules to be tested
    Parameters:
    - use_custom_tmpfile_module - if True, enable emulation of customer's hosted module
    Currently we have tmpfile module only which just creates /tmp/<filename>
    - supported_module_versions - dict, containing lists of allowed versions per module, f.e.
    {'sysctl': ['1.0.1', '1.1.1']}. Other module versions will be ignored.
    If module name is absent in this structure - all its versions from all hocm
    objects will be IGNORED.
    """

    if supported_module_versions is None:
        supported_module_versions = {}
    if skip_module_versions is None:
        skip_module_versions = {}
    kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_PATH)
    cluster_name = settings.TARGET_CLUSTER
    cluster_ns = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace=cluster_ns)
    cluster = ns.get_cluster(cluster_name)

    # Create HostOSConfigurationModule resource emulating customer-hosted module
    custom_hostoscfg_module_resource_name = 'custom-modules'
    if use_custom_tmpfile_module:
        LOG.banner("Create a custom 'tmpfile' HOC module")
        custom_hostoscfg_module_name = 'tmpfile'
        custom_hostoscfg_module_version = '1.0.0'
        custom_hostoscfg_module_sha256 = '0eda55e339a5eced109b3d906fcb84efeb0f5a8f79292e10a6f537c093119dee'
        custom_hostoscfg_module_url = (
            'https://artifactory.mcp.mirantis.net/binary-dev-kaas-local'
            '/bm/bin/custom-host-os-modules/tmpfile-1.0.0.tgz'
        )
        custom_hostoscfg_module_data = {
            "apiVersion": "kaas.mirantis.com/v1alpha1",
            "kind": "HostOSConfigurationModules",
            "metadata": {
                "name": custom_hostoscfg_module_resource_name,
            },
            "spec": {
                "modules": [
                    {
                        "name": custom_hostoscfg_module_name,
                        "version": custom_hostoscfg_module_version,
                        "sha256sum": custom_hostoscfg_module_sha256,
                        "url": custom_hostoscfg_module_url,
                    },
                ],
            }
        }
        custom_hostoscfg_module_list = kaas_manager.get_hostosconfigurationmodules()
        if custom_hostoscfg_module_resource_name not in [m.name for m in custom_hostoscfg_module_list]:
            LOG.info(f'Creating new hostosconfigurationmodule: {custom_hostoscfg_module_resource_name}')
            try:
                kaas_manager.create_hostosconfigurationmodule_raw(custom_hostoscfg_module_data)
            # stupid WA if 2 day2 tests ran at same time.
            # the issue behind - one of the paraless might create object exactly between
            # 'custom_hostoscfg_module_list' and 'if custom_hostoscfg_module_resource_name not in'
            # checks.
            except k8sApiException as e:
                if e.status == 409 and settings.CHECK_SKIP_DAY2_HOCM_CREATE_CONFLICT:
                    LOG.warning(f'HOCM create conflict detected for {custom_hostoscfg_module_resource_name}')
                    pass
            timeout = 60 * 5
            interval = 10
            timeout_msg = (f"HostOSConfigurationModule {custom_hostoscfg_module_resource_name} "
                           f"was not created in {timeout}s")
            waiters.wait(lambda: _check_custom_hostoscfg_module_status(
                name_prefix=custom_hostoscfg_module_resource_name,
                kaas_manager=kaas_manager),
                         timeout=timeout,
                         interval=interval,
                         timeout_msg=timeout_msg)

    # Collect modules data from hostosconfigurationmodules objects
    LOG.banner("Collect HOC modules data from hostosconfigurationmodules objects")
    hostoscfgmodules_list = kaas_manager.get_hostosconfigurationmodules()
    test_modules = {}
    for hostoscfgmodules in hostoscfgmodules_list:
        # Wait for hoc module status
        timeout = 60 * 5
        interval = 10
        timeout_msg = (f"HostOSConfigurationModule {custom_hostoscfg_module_resource_name} "
                       f"still has None status for last {timeout}s")
        waiters.wait(lambda: _check_custom_hostoscfg_module_status(name_prefix=hostoscfgmodules.name,
                                                                   kaas_manager=kaas_manager),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)

        hostoscfgmodules_data = hostoscfgmodules.data or dict()
        hostoscfg_modules_statuses = hostoscfgmodules_data.get('status', {}).get('modules', [])
        # Collect modules from spec
        hostoscfg_modules = hostoscfgmodules_data.get('spec', dict()).get('modules', list()) or list()
        for hostoscfg_module in hostoscfg_modules:
            hostoscfg_module_name = hostoscfg_module.get("name")
            hostoscfg_module_version = hostoscfg_module.get("version")
            hostoscfg_module_allowed_versions = supported_module_versions.get(hostoscfg_module_name, [])
            if (not hostoscfg_module_allowed_versions
                    or hostoscfg_module_version not in hostoscfg_module_allowed_versions):
                LOG.warning(f'Skipping module {hostoscfg_module_name}-{hostoscfg_module_version} '
                            f'as there is no tests available for it')
                continue
            hostoscfg_module_skip_versions = skip_module_versions.get(hostoscfg_module_name, [])
            if (hostoscfg_module_skip_versions and hostoscfg_module_version in hostoscfg_module_skip_versions):
                LOG.warning(f'Skipping module {hostoscfg_module_name}-{hostoscfg_module_version} '
                            f'because it is in <skip_module_versions> list')
                continue

            hostoscfg_module_key = f"{hostoscfg_module_name}-{hostoscfg_module_version}"
            assert hostoscfg_module_key not in test_modules.keys(), (
                f"Duplicate definition of the module '{hostoscfg_module_name}:{hostoscfg_module_version}' "
                f"in both HostOSConfigurationModule objects '{hostoscfgmodules.name}' and "
                f"'{test_modules[hostoscfg_module_key]['object_ref']}'")

            testobject_class = _get_testobject_class(hostoscfg_module_name)
            testobject = testobject_class(cluster, ns)

            hostoscfg_module_status = {}
            for hocm_status in hostoscfg_modules_statuses:
                if (hostoscfg_module_name == hocm_status.get("name") and
                        hostoscfg_module_version == hocm_status.get("version")):
                    hostoscfg_module_status = hocm_status

            test_modules[hostoscfg_module_key] = {
                'object_ref': hostoscfgmodules.name,
                'key': hostoscfg_module_key,
                'spec': hostoscfg_module,
                'testobject': testobject,
                'failed_test': '',
                'status': hostoscfg_module_status,
            }

    return test_modules.values()


if settings.HOC_USE_BUILTIN_MCC_MODULE_VERSIONS:
    # SI modules versions taken from installed MCC cluster
    supported_module_versions = get_mcc_module_versions()
    hocm_msg_suffix = 'HOC modules present in current MCC cluster:'
else:
    # BM supported module versions
    supported_module_versions = settings.HOC_FIXED_SUPPORTED_MODULE_VERSIONS
    # supported_module_versions = {'sysctl': ['1.1.0'],
    #                             'package': ['1.1.0'],
    #                             'tmpfile': ['1.0.0'],
    #                             'irqbalance': ['1.0.0'],
    #                             'grub_settings': ['1.0.2']}
    hocm_msg_suffix = 'fixed list of HOC modules (see <supported_module_versions> variable):'
if settings.HOC_USE_SKIP_MODULE_VERSIONS:
    skip_module_versions = settings.HOC_SKIP_MODULE_VERSIONS
else:
    skip_module_versions = {}

LOG.warning(f'Test supposed to work with {hocm_msg_suffix}\n{supported_module_versions}')
modules_data = list(generate_module_data(use_custom_tmpfile_module=settings.HOC_USE_CUSTOM_TMPFILE_MODULE,
                                         supported_module_versions=supported_module_versions,
                                         skip_module_versions=skip_module_versions))


@pytest.fixture(scope='function', params=modules_data,
                ids=[f"MODULE={x['key']}" for x in modules_data])
def module_data(request, func_name):
    _data = request.param
    if _data['failed_test']:
        msg = f"Skip '{func_name}' because previous test for the module {_data['key']} has been failed"
        LOG.info(msg)
        pytest.skip(msg)

    LOG.info(f"Run test {func_name} for module {_data['key']}")
    yield _data
    # Check the result of the current step
    test_passed = (hasattr(request.node, 'rep_call') and
                   (request.node.rep_call.passed or request.node.rep_call.skipped))
    if not test_passed:
        _data['failed_test'] = func_name


@pytest.fixture(scope='module')
def configobject(kaas_manager):
    # BaseModule testobject to create hoc resources with few modules
    cluster_name = settings.TARGET_CLUSTER
    cluster_ns = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace=cluster_ns)
    cluster = ns.get_cluster(cluster_name)
    configobject = BaseModule(cluster, ns)
    yield configobject


def test_010_host_os_configuration_bm_module_status(module_data, kaas_manager):
    """Test HostOSConfigurations and HostOSConfigurationModules
        8. TODO(ddmitriev): Test applying 'values' with the same module name but with higher module version

    """
    module_state = module_data['status'].get('state')
    module_error = module_data['status'].get('error')
    assert module_state in ('available', 'deprecated'), (
        f"HostOSConfigurationModule '{module_data['key']}' is in wrong state '{module_state}': {module_error}\n"
        f"Module spec:\n{yaml.safe_dump(module_data['spec'])}\n"
        f"Module status:\n{yaml.safe_dump(module_data['status'])}")
    LOG.info(f'Pass for\n{module_data}')


def test_020_host_os_configuration_si_negative_conflict_options(module_data, kaas_manager):
    """Negative: create HostOSConfiguration with 'values' and 'secretValues' at the same time should not be allowed
    Expected result: admission controller does not accept such set of options, and return HTTP status = 400
    """
    hostoscfg_module_name = module_data['spec']['name']
    hostoscfg_module_version = module_data['spec']['version']
    testobject = module_data['testobject']
    hostoscfg_name = f"test-020-negative-{module_data['key']}-{testobject.cluster.name}".replace('_', '-')
    # Remove existing HostOSConfiguration objects with such names
    testobject.delete(name=hostoscfg_name)
    hostoscfg_configs = [
        {
            "module": hostoscfg_module_name,
            "moduleVersion": hostoscfg_module_version,
            "values": testobject.get_values_to_create(),
            "secretValues": testobject.get_secret_values_to_create(),
        },
    ]
    with pytest.raises(ApiException) as exc_info:
        # Try to create hoc object, expecting result status 400
        testobject.create(name=hostoscfg_name, configs=hostoscfg_configs, machines_type='control')
        pytest.fail(f"Expected result with 'Bad Request', but object '{hostoscfg_name}' has been succesfully created")
    assert exc_info.value.status == 400 and exc_info.value.reason == 'Bad Request', (
        f"Expected reason 'Bad Request' when module config contains both 'values' and 'secretValues', "
        f"but got:\n{exc_info.value.body}")
    LOG.info(f"Got the following response: {exc_info.value.body}")


def test_021_host_os_configuration_bm_negative_bad_values(module_data, kaas_manager):
    """Negative: create HostOSConfiguration with bad values

    Expected result: HostOSConfiguration is created, but bad values cause errors in LCMMachines stateItemStatuses
    """

    hostoscfg_module_name = module_data['spec']['name']
    if hostoscfg_module_name in ('tmpfile', 'grub_settings', 'cpushield'):
        msg = (f"Skip negative_bad_values test for {hostoscfg_module_name} because it does not have bad values")
        LOG.info(msg)
        pytest.skip(msg)
    else:
        LOG.info(f'test_021_host_os_configuration_bm_negative_bad_values => for {hostoscfg_module_name}')
    hostoscfg_module_version = module_data['spec']['version']
    testobject = module_data['testobject']
    hostoscfg_name = f"test-021-negative-{module_data['key']}-{testobject.cluster.name}".replace('_', '-')
    LOG.info("Create HostOSConfiguration with bad values and ensure that "
             "related LCMMachines got errors for the related stateItemStatuses")
    hostoscfg_configs = [
        {
            "module": hostoscfg_module_name,
            "moduleVersion": hostoscfg_module_version,
            "values": testobject.get_bad_values(),
        },
    ]

    try:
        testobject.apply_new_config_negative(name=hostoscfg_name, configs=hostoscfg_configs,
                                             machines_type='control')
    finally:
        cleanup_values = testobject.get_cleanup_bad_values()
        LOG.info(f"Updating  {hostoscfg_name} with cleanup values")
        if cleanup_values:
            hostoscfg_configs = [
                {
                    "module": hostoscfg_module_name,
                    "moduleVersion": hostoscfg_module_version,
                    "values": cleanup_values,
                },
            ]
            testobject.apply_patch(name=hostoscfg_name, configs=hostoscfg_configs, check_result=False)
        LOG.info("Remove created HostOSConfiguration object")
        testobject.delete(name=hostoscfg_name)


def test_030_host_os_configuration_bm_first_config(configobject, kaas_manager):
    """Test applying 'values' in a new HostOSConfiguration"""
    hostoscfg_name = f"test-030-first-config-{configobject.cluster.name}".replace('_', '-')
    machines_type = 'control'
    machines_distro = configobject.get_machines_distro_version(machines_type)
    # Remove existing HostOSConfiguration objects with such names
    configobject.delete(name=hostoscfg_name)

    hostoscfg_configs = []
    for module_data in modules_data:
        hostoscfg_module_name = module_data['spec']['name']
        hostoscfg_module_version = module_data['spec']['version']
        module_testobject = module_data['testobject']
        # Skip cpushield module testing for Ubuntu 20.04
        if hostoscfg_module_name == 'cpushield' and machines_distro and 'ubuntu/focal' in machines_distro.values():
            LOG.info(f"Skipping cpushield module testing because some machines "
                     f"of machines_type {machines_type} are provisoned with Ubuntu 20.04")
            continue
        hostoscfg_configs += [
            {
                "module": hostoscfg_module_name,
                "moduleVersion": hostoscfg_module_version,
                "values": module_testobject.get_values_to_create(),
            },
        ]

    # Create a HostOSConfiguration on 'control' nodes on Management or Child cluster
    try:
        configobject.apply_new_config(name=hostoscfg_name, configs=hostoscfg_configs,
                                      machines_type=machines_type)
    except Exception as e:
        LOG.error(f"Error during {hostoscfg_name} hoc object application: {e}")
        configobject.delete(name=hostoscfg_name)
        raise e


def test_040_host_os_configuration_bm_second_config(configobject, kaas_manager):
    """Test applying new HostOSConfiguration object with 'values' for already existing parameters."""
    hostoscfg_name = f"test-040-second-config-{configobject.cluster.name}".replace('_', '-')
    machines_type = 'control'
    machines_distro = configobject.get_machines_distro_version(machines_type)
    # Remove existing HostOSConfiguration objects with such names
    configobject.delete(name=hostoscfg_name)

    hostoscfg_configs = []
    for module_data in modules_data:
        hostoscfg_module_name = module_data['spec']['name']
        hostoscfg_module_version = module_data['spec']['version']
        module_testobject = module_data['testobject']
        # Skip cpushield module testing for Ubuntu 20.04
        if hostoscfg_module_name == 'cpushield' and machines_distro and 'ubuntu/focal' in machines_distro.values():
            LOG.info(f"Skipping cpushield module testing because some machines "
                     f"of machines_type {machines_type} are provisoned with Ubuntu 20.04")
            continue
        hostoscfg_configs += [
            {
                "module": hostoscfg_module_name,
                "moduleVersion": hostoscfg_module_version,
                "values": module_testobject.get_new_values_to_create(),
            },
        ]

    # TODO(ddmitriev): also change the machine_type for this object from 'control' to 'storage' for child
    try:
        configobject.apply_new_config(name=hostoscfg_name, configs=hostoscfg_configs,
                                      machines_type=machines_type)
    finally:
        configobject.delete(name=hostoscfg_name)


def test_050_host_os_configuration_bm_update_first_config(configobject, kaas_manager):
    """Test that new 'values' added to an existing HostOSConfiguration object are applied on the Machines"""
    machines_type = 'control'
    machines_distro = configobject.get_machines_distro_version(machines_type)

    hostoscfg_configs = []
    for module_data in modules_data:
        hostoscfg_module_name = module_data['spec']['name']
        hostoscfg_module_version = module_data['spec']['version']
        module_testobject = module_data['testobject']
        # Skip cpushield module testing for Ubuntu 20.04
        if hostoscfg_module_name == 'cpushield' and machines_distro and 'ubuntu/focal' in machines_distro.values():
            LOG.info(f"Skipping cpushield module testing because some machines "
                     f"of machines_type {machines_type} are provisoned with Ubuntu 20.04")
            continue
        hostoscfg_configs += [
            {
                "module": hostoscfg_module_name,
                "moduleVersion": hostoscfg_module_version,
                "values": module_testobject.get_new_values_to_update(),
            },
        ]

    # Use existing HostOSConfiguration from previous test to update it's values
    hostoscfg_name = f"test-030-first-config-{configobject.cluster.name}".replace('_', '-')

    # TODO(ddmitriev): also change the machine_type for this object from 'control' to 'worker' for child
    try:
        configobject.apply_patch(name=hostoscfg_name, configs=hostoscfg_configs, machines_type=machines_type)
    finally:
        configobject.delete(name=hostoscfg_name)


def test_070_host_os_configuration_bm_add_secret_values(configobject, kaas_manager):
    """Test applying 'secretValues' in a new HostOSConfiguration"""

    hostoscfg_name = f"test-070-add-secret-values-{configobject.cluster.name}".replace('_', '-')
    machines_type = 'control'

    # Remove existing HostOSConfiguration objects with such names
    configobject.delete(name=hostoscfg_name)

    hostoscfg_configs = []
    secret_names = []
    for module_data in modules_data:
        hostoscfg_module_name = module_data['spec']['name']
        if hostoscfg_module_name != 'tmpfile':
            msg = (f"Skip add_secret_values test for {hostoscfg_module_name} because it "
                   f"requires complex data structures as parameters (dict/list etc.), "
                   f"but we don't support base64 decoding for them")
            LOG.info(msg)
            continue
        hostoscfg_module_version = module_data['spec']['version']
        module_testobject = module_data['testobject']
        module_secret_name = f"{hostoscfg_name}-{module_data['key']}"
        # Remove existing Secret objects with such name
        configobject.ns.delete_secret(module_secret_name)
        secret_data = module_testobject.get_secret_values_to_create()
        configobject.ns.create_secret(name=module_secret_name, data=secret_data)
        secret_creation_timeout = 60
        secret_creation_interval = 5
        secret_creation_timeout_msg = (f'Secret {module_secret_name} in namespace {configobject.ns.name} '
                                       'has not been created')
        waiters.wait(lambda: bool(kaas_manager.api.secrets.present(name=module_secret_name,
                                                                   namespace=configobject.ns.name)),
                     timeout=secret_creation_timeout,
                     interval=secret_creation_interval,
                     timeout_msg=secret_creation_timeout_msg)
        secret_names.append(module_secret_name)

        hostoscfg_configs += [
            {
                "module": hostoscfg_module_name,
                "moduleVersion": hostoscfg_module_version,
                "secretValues": {
                    # Secret object name
                    "name": module_secret_name,
                    "namespace": configobject.ns.name,
                },
            },
        ]

    try:
        configobject.apply_new_config(name=hostoscfg_name, configs=hostoscfg_configs,
                                      machines_type=machines_type)
    finally:
        configobject.delete(name=hostoscfg_name)
        for secret in secret_names:
            configobject.ns.delete_secret(name=secret)


def test_090_host_os_configuration_bm_check_stateitemsoverwrites_after_hoc_deletion(kaas_manager):
    """Scenario:
        1. Create 2 HostOSConfiguration objects for the same machineSelector (apply on the same node)
        2. Delete one of HostOSConfiguration objects
        3. Check that no stateItemsOverwrites from deleted object are present in LCMMachine object
        4. Check that stateItemsOverwrites for preserved HostOSConfiguration object did not disappear
      """
    cluster_name = settings.TARGET_CLUSTER
    cluster_ns = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace=cluster_ns)
    cluster = ns.get_cluster(cluster_name)
    configobject1 = BaseModule(cluster, ns)
    configobject2 = BaseModule(cluster, ns)
    hostoscfg_name1 = f"test-090-check-stateitemsoverwrites-after-hoc-deletion-1-{configobject1.cluster.name}"
    hostoscfg_name1 = hostoscfg_name1.replace('_', '-')
    hostoscfg_name2 = f"test-090-check-stateitemsoverwrites-after-hoc-deletion-2-{configobject2.cluster.name}"
    hostoscfg_name2 = hostoscfg_name2.replace('_', '-')
    machines_type = 'control'
    # Remove existing HostOSConfiguration objects with such names
    configobject1.delete(name=hostoscfg_name1)
    configobject2.delete(name=hostoscfg_name2)

    # For this test please use DIFFERENT module names to avoid conflicts by applying
    # different configurations of the same module of the same version on the same node
    # NOTE: Do not use 'cpushield' module as it is not compatible with Ubuntu 20.04
    # which can be still deployed on childs
    module_name_hoc1 = 'tmpfile'
    module_name_hoc2 = 'sysctl'
    hostoscfg_configs1 = []
    hostoscfg_configs2 = []
    for module_data in modules_data:
        hostoscfg_module_name = module_data['spec']['name']
        hostoscfg_module_version = module_data['spec']['version']
        module_testobject = module_data['testobject']

        if hostoscfg_module_name == module_name_hoc1:
            hostoscfg_configs1 += [
                {
                    "module": hostoscfg_module_name,
                    "moduleVersion": hostoscfg_module_version,
                    "values": module_testobject.get_values_for_stateitemsoverwrites_after_hoc_deletion_test(),
                },
            ]
        if hostoscfg_module_name == module_name_hoc2:
            hostoscfg_configs2 += [
                {
                    "module": hostoscfg_module_name,
                    "moduleVersion": hostoscfg_module_version,
                    "values": module_testobject.get_values_for_stateitemsoverwrites_after_hoc_deletion_test(),
                },
            ]

    try:
        lcmmachines_timestamps_before1 = configobject1.cluster.get_cluster_lcmmachines_timestamps()
        lcmmachines_timestamps_before2 = configobject2.cluster.get_cluster_lcmmachines_timestamps()
        hostoscfg1 = configobject1.create(name=hostoscfg_name1, configs=hostoscfg_configs1, machines_type=machines_type)
        hostoscfg2 = configobject2.create(name=hostoscfg_name2, configs=hostoscfg_configs2, machines_type=machines_type)
        # Wait for the selected Machines in the hostosconfiguration status
        LOG.info("Check that machines from hostosconfiguration status field have labels used for machineSelector")
        configobject1.cluster.check.check_hostosconfig_machine_selector(hostoscfg1)
        configobject2.cluster.check.check_hostosconfig_machine_selector(hostoscfg2)
        LOG.info("Check that new items added into machineTypes in LCMCluster")
        configobject1.cluster.check.wait_lcmcluster_day2_machinetypes(hostoscfg1)
        configobject2.cluster.check.wait_lcmcluster_day2_machinetypes(hostoscfg2)
        LOG.info("Check that new items added into stateItems in LCMMachine")
        configobject1.cluster.check.wait_lcmmachine_day2_stateitems(hostoscfg1, lcmmachines_timestamps_before1)
        configobject2.cluster.check.wait_lcmmachine_day2_stateitems(hostoscfg2, lcmmachines_timestamps_before2)
        LOG.info("Check the result on the machines")
        configobject1.cluster.check.get_hostosconfig_machines_status(hostoscfg1)
        configobject2.cluster.check.get_hostosconfig_machines_status(hostoscfg2)
        hostoscfg1_data = hostoscfg1.data
        configobject1.delete(name=hostoscfg_name1)
        LOG.info("Check the result on the machines after first HOC object removal")
        cluster.check.wait_lcmmachine_day2_stateitemsoverwrites(hostoscfg1_data, absent=True)
        hostoscfg2_data = hostoscfg2.data
        cluster.check.wait_lcmmachine_day2_stateitemsoverwrites(hostoscfg2_data)
    finally:
        configobject1.delete(name=hostoscfg_name1)
        configobject2.delete(name=hostoscfg_name2)
