import functools
import os
import uuid
import tempfile
import untangle

from si_tests.deployments.utils import commons
from si_tests.deployments.utils import file_utils
from si_tests.deployments.utils import kubectl_utils
from si_tests.deployments.utils import namespace
from si_tests.deployments.utils import wait_utils
from si_tests import settings


def test_advanced_compute_instance_numa(openstack_client_manager, request):
    stack_name = f"advanced_compute_instance-{uuid.uuid4()}"
    target_template = f"/tmp/{stack_name}.yaml"
    target_parameters = f"/tmp/{stack_name}-parameters.yaml"
    source = file_utils.join(os.path.dirname(os.path.abspath(__file__)),
                             "advanced_compute_instance.yaml")
    client_name = openstack_client_manager.client.name
    kubectl = kubectl_utils.Kubectl()
    destination = "{}/{}:{}".format(
        namespace.NAMESPACE.openstack, client_name, target_template)
    parameters_dest = "{}/{}:{}".format(
        namespace.NAMESPACE.openstack, client_name, target_parameters)
    with tempfile.NamedTemporaryFile(mode="w") as tmp:
        az_name = settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME
        content = f"""
        parameters:
          server_availability_zone: {az_name}
        """
        tmp.write(content)
        tmp.flush()
        kubectl.cp(tmp.name,  parameters_dest)
    commons.LOG.info("Copy heat template")
    kubectl.cp(source, destination)

    commons.LOG.info("Create heat stack")
    openstack_client_manager.stack.create(
        [stack_name, "-t", target_template, "-e", target_parameters]
    )

    request.addfinalizer(
        functools.partial(
            openstack_client_manager.stack.delete,
            [stack_name, "-y", "--wait"]))

    commons.LOG.info("Wait for heat stack to create")
    wait = wait_utils.Waiter(openstack_client_manager.os_manager, 3600)

    def f():
        status = openstack_client_manager.stack.show(
            [stack_name])["stack_status"]
        if status == "CREATE_FAILED":
            resource_list = openstack_client_manager.stack.resource_list(
                ['-n', '10', stack_name])
            event_list = openstack_client_manager.stack.event_list(
                ['--nested-depth', '10', stack_name])
            commons.LOG.error("Resource info: %s\nHeat stack event list: %s",
                              resource_list, event_list)
            raise Exception("Failed to create stack")
        return status == "CREATE_COMPLETE"

    wait.wait(f)
    commons.LOG.info("Heat stack created successfully")

    server = openstack_client_manager.server.show([stack_name])
    hypervisor_hostname = server["OS-EXT-SRV-ATTR:hypervisor_hostname"].split(
        '.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name))
    # This instance should have 2 numa nodes
    assert 2 == len(domain_obj.domain.cpu.numa)
    host_capabilities = untangle.parse(
        openstack_client_manager.virsh_capabilities(hypervisor_hostname)
    ).capabilities.host
    host_cpus_by_cell = {}
    for cell in host_capabilities.topology.cells.cell:
        host_cpus_by_cell[cell.get_attribute("id")] = [
            cpu.get_attribute("id") for cpu in cell.cpus.children]
    instance_pcpus = []
    for vcpupin in domain_obj.domain.cputune.vcpupin:
        instance_pcpus.append(vcpupin.get_attribute("cpuset"))
    instance_pcpu_cells = []
    for cell, host_pcpus in host_cpus_by_cell.items():
        for instance_pcpu in instance_pcpus:
            if instance_pcpu in host_pcpus:
                instance_pcpu_cells.append(cell)
                break
    # Each instance CPU should come from different host NUMA node
    assert 2 == len(instance_pcpu_cells)
