#    Copyright 2019 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os
import pytest

from si_tests import settings, logger
from si_tests.managers import bootstrap_manager
from si_tests.managers import bootstrap_packet
from si_tests.managers.kaas_manager import Manager
from si_tests.managers import si_config_manager

LOG = logger.logger


class TestProvisionBmCluster():

    @pytest.mark.usefixtures("introspect_standalone_seed_management_deploy_objects")
    def test_bootstrap_kaas_bm_phase2(self, show_step):
        """Bootstrap KaaS_BM on OpenStack VM - remote case
        This is second part for remote case, first one:
        management-cluster/test_provision_seed_node.py

        Scenario:
            1. ...
            2. ...
            3. ...
            4. ...
            5. step_003_prepare_bm_seed_node | step_004_prepare_for_kaas_bm_cluster
            6. Generate Keycloak user credentials and store them in si-config.yaml before rendering templates
            7. Install kaas-bm mgmt cluster using ansible playbooks
            8. Create default MCC users, if requested
            9. Wait for all pods to be Running and Ready
            10. Check all expected pods exists
            11. Download bootstrap artifacts
            12. Check LDAP integration if enabled
            13. Check auditd if enabled
        """
        bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()
        show_step(5)
        bootstrap.step_003_prepare_bm_seed_node()
        bootstrap.step_004_prepare_for_kaas_bm_cluster()

        show_step(6)
        bootstrap.add_credentials_data_to_si_config()

        show_step(7)
        bootstrap.step_004_prepare_bm_on_seed()
        bootstrap.step_004_deploy_kaas_bm_mgmt_cluster()

        if settings.CREATE_DEFAULT_MCC_USERS_VIA_CLI:
            show_step(8)
            bootstrap.step_004_create_default_mcc_users(
                custom_bootstrap_path=settings.KAAS_BOOTSTRAP_TARGET_DIR)

        show_step(9)
        bootstrap.step_006_wait_for_pods()

        show_step(10)
        if settings.RUN_POD_CHECKS:
            bootstrap.step_006_postdeployment_checks(check_svc=True)

        show_step(11)
        bootstrap.step_007_download_bootstrap_artifacts()

        if settings.KEYCLOAK_LDAP_ENABLED:
            show_step(12)
            bootstrap.check_keystone_ldap_integration()

        show_step(13)
        bootstrap.check_audit()

    def test_bootstrap_kaas_bm_equinix_phase1(self, show_step):
        """Bootstrap seed node on equinix HS
        Scenario:
            1. Erase previous seed node by hostname,if exists.
               Create a equinix node like seed
        """
        bootstrapEq = bootstrap_packet.BootstrapManagerPacket()
        show_step(1)
        bootstrapEq.step_002_provision_seed_vm()

    @pytest.mark.usefixtures("introspect_openstack_seed_management_deploy_objects")
    def test_bootstrap_kaas_bm_equinix_phase2(self, openstack_client,
                                              show_step):
        """Bootstrap KaaS_BM on OpenStack VM

        Scenario:
            6. Install kaas-bm mgmt cluster using ansible playbooks
            7. Create default MCC users, if requested
            8. Wait for all pods to be Running and Ready
            9. Check all expected pods exists
            10. Download bootstrap artifacts
            11. Check LDAP integration if enabled
        """
        seed_ip = openstack_client.get_seed_ip()
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        show_step(3)
        bootstrap.step_003_prepare_bm_seed_node()

        # Uncomment only after implementing SI_CONFIG parameter on BM CI
        # show_step(4)
        # bootstrap.add_credentials_data_to_si_config()

        # step 005
        if settings.SKIP_BOOTSTRAP_TEMPLATES_CHECK:
            LOG.warning("Skip bootstrap templates check due "
                        "SKIP_BOOTSTRAP_TEMPLATES_CHECK flag is set")
            if not settings.KAAS_CUSTOM_LMA_LOGGING_ENABLED:
                bootstrap.disable_lma_logging()
            if settings.CORE_KEYCLOAK_LDAP_ENABLED:
                bootstrap.enable_keycloak_ldap()
        else:
            show_step(5)
            bootstrap.check_templates()

        show_step(6)
        bootstrap.step_004_prepare_for_kaas_bm_cluster()

        show_step(7)
        bootstrap.step_004_prepare_bm_on_seed()
        bootstrap.step_004_deploy_kaas_bm_mgmt_cluster()

        if settings.CREATE_DEFAULT_MCC_USERS_VIA_CLI:
            show_step(8)
            bootstrap.step_004_create_default_mcc_users(
                custom_bootstrap_path=settings.KAAS_BOOTSTRAP_TARGET_DIR,
                baremetal=True)

        if settings.MCC_SCHEDULED_UPGRADES:
            LOG.info("Update mcc-upgrade crd for feature upgrade scheduling")
            bootstrap.step_006_update_mcc_upgrade_crd()

        show_step(9)
        bootstrap.step_006_wait_for_pods()

        show_step(10)
        if settings.RUN_POD_CHECKS:
            bootstrap.step_006_postdeployment_checks(check_svc=True)

        show_step(11)
        bootstrap.step_007_download_bootstrap_artifacts()

        if settings.KEYCLOAK_LDAP_ENABLED:
            show_step(12)
            bootstrap.check_keystone_ldap_integration()

    @pytest.mark.usefixtures("introspect_openstack_seed_management_deploy_objects")
    def test_bootstrap_kaas_bm_physical(self, openstack_client, show_step):
        """Bootstrap KaaS_BM on OpenStack VM

        Scenario:
            1. Erase previous KaaS environment and BM node, if exists
            2. Create a stack with a node0
            3. Prepare a node0 and upload git repos
            4. Generate Keycloak user credentials and store them in si-config.yaml before rendering templates
            5. Check templates
            6. Install kaas-bm cluster using ansible playbooks
            7. Create default MCC users, if requested
            8. Wait for all pods to be Running and Ready
            9. Check that all expected pods exists
            10. Download bootstrap artifacts
            11. Store seed IP in si_config
            12. Check LDAP integration if enabled
        """
        if not settings.SEED_STANDALONE_EXTERNAL_IP:
            # step 001
            if settings.KEEP_ENV_BEFORE:
                LOG.info('Skip erase due KEEP_ENV_BEFORE flag is set')
            else:
                if openstack_client._current_stack:
                    # Erase the stack with the seed node
                    show_step(1)
                    seed_ip = openstack_client.get_seed_ip()
                    old_bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
                    old_bootstrap.step_001_erase_env_before()
                    LOG.info(f"Remove heat stack {settings.ENV_NAME}")
                    openstack_client.delete_stack()
                else:
                    LOG.warning("Can't erase environment due "
                                "seed node stack is absent")

            # Create a stack with a seed node
            show_step(2)
            openstack_client.create_seed_node()
            seed_ip = openstack_client.get_seed_ip()
        else:
            seed_ip = settings.SEED_STANDALONE_EXTERNAL_IP

        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        bootstrap.wait_for_seed_node_ssh()
        si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
        si_config.store_seed_ip(seed_ip)

        show_step(3)
        endpoints = openstack_client.get_os_cloud_endpoints()

        bootstrap.step_003_prepare_bm_seed_node(endpoints=endpoints)

        # step 004
        show_step(4)
        bootstrap.add_credentials_data_to_si_config()

        # step 005
        if settings.SKIP_BOOTSTRAP_TEMPLATES_CHECK:
            LOG.warning("Skip bootstrap templates check due "
                        "SKIP_BOOTSTRAP_TEMPLATES_CHECK flag is set")
            if not settings.KAAS_CUSTOM_LMA_LOGGING_ENABLED:
                bootstrap.disable_lma_logging()
            if settings.CORE_KEYCLOAK_LDAP_ENABLED:
                bootstrap.enable_keycloak_ldap()
        else:
            show_step(5)
            bootstrap.check_templates()

        show_step(6)
        bootstrap.step_004_deploy_kaas_bm_cluster_physical(settings.ENV_CONFIG_NAME or settings.ENV_NAME)

        if settings.CREATE_DEFAULT_MCC_USERS_VIA_CLI:
            show_step(7)
            bootstrap.step_004_create_default_mcc_users(
                custom_bootstrap_path=settings.KAAS_BOOTSTRAP_TARGET_DIR)

        show_step(8)
        bootstrap.step_006_wait_for_pods()

        show_step(9)
        if settings.RUN_POD_CHECKS:
            bootstrap.step_006_postdeployment_checks(check_svc=True)

        show_step(10)
        bootstrap.step_007_download_bootstrap_artifacts()

        show_step(11)
        kubeconfig_path = f"{settings.ARTIFACTS_DIR}/management_kubeconfig"
        if not os.path.isfile(kubeconfig_path):
            raise Exception(f"KUBECONFIG {kubeconfig_path} not found in"
                            f" artifacts!")
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        kaas_manager.si_config.store_seed_ip(seed_ip)
        cluster = kaas_manager.get_mgmt_cluster()
        cluster.check.check_actual_expected_distribution()

        if settings.KEYCLOAK_LDAP_ENABLED:
            show_step(12)
            bootstrap.check_keystone_ldap_integration()
