import pytest


from si_tests import settings
from si_tests import logger
from si_tests.utils import utils
from si_tests.managers.kaas_manager import Manager

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_regional_vsphere_child(kaas_manager: Manager, proxy_manager, show_step, _):
    """Deploy child cluster on top of management cluster.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating vSphere credential
        5. Creating RHEL license secret
        6. Set proxy
        7. Create child cluster and machines with SL label
        8. Check cluster readiness
        9. Pod checks (optional)
        10. Check offline isolation (optional)

    """

    # Collecting env data
    show_step(1)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    LOG.info("Available releases: {}".format(
        kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()
    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    slave_nodes_count = int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
    master_nodes_count = int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT)
    region = kaas_manager.get_mgmt_cluster().region_name

    rnd_string = utils.gen_random_string(4)

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_or_create_namespace(namespace_name)

    # Creating public key
    show_step(3)
    public_key_name = "{cluster_name}-{rnd_string}-key".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    ns.create_publickey(public_key_name, pub_key_content)

    # Creating vsphere secret
    show_step(4)
    credential_name = "{cluster_name}-{rnd_string}-cred".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))
    # clouds_yaml_str = templates.render_template(settings.CLOUDS_YAML_PATH)
    vsphere_creds = kaas_manager.get_vsphere_credential(
        "cloud-config", 'default', region)
    ns.create_vsphere_credential(
        credential_name,
        vsphere_creds,
        region=region)

    # Creating RHEL license secret
    show_step(5)
    rhel_license_name = "{cluster_name}-{rnd_string}-license".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    if 'mke-5-12' in release_name or 'mke-5-11' in release_name:
        LOG.info("Detected release mke-5-12/11. Use user/password")
        ns.create_rhel_license(
            rhel_license_name,
            rhel_creds=(
                settings.KAAS_CI_RHEL_LICENSES_USERNAME,
                settings.KAAS_CI_RHEL_LICENSES_PASSWORD
            ),
            region=region,
            is_license_key=False
        )
    else:
        LOG.info("Detected release greater then mke-5-12.")
        if settings.KAAS_CI_RHEL_LICENSES_IS_ACTIVATION_KEY:
            LOG.info("Use activation key for RHEL machines subscription")
            ns.create_rhel_license(
                rhel_license_name,
                rhel_creds=(
                    settings.KAAS_CI_RHEL_LICENSES_ACTIVATION_KEY,
                    settings.KAAS_CI_RHEL_LICENSES_ACTIVATION_ORG,
                    settings.KAAS_CI_RHEL_LICENSES_RPM_URL
                ),
                region=region,
                is_license_key=True)
        else:
            LOG.info("Use user/password for RHEL machines")
            ns.create_rhel_license(
                rhel_license_name,
                rhel_creds=(
                    settings.KAAS_CI_RHEL_LICENSES_USERNAME,
                    settings.KAAS_CI_RHEL_LICENSES_PASSWORD
                ),
                region=region,
                is_license_key=False
            )

    show_step(6)
    if ((settings.KAAS_EXTERNAL_PROXY_ACCESS_STR or
         settings.KAAS_INSTALL_FAKE_PROXY) and
            not settings.KAAS_OFFLINE_DEPLOYMENT):
        LOG.error("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT "
                  "is False.Using Proxy on online deployments makes no sense.")
        raise RuntimeError("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT"
                           " is False")

    if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
        LOG.info('Proxy mode is enabled. But external network will not '
                 'be restricted on child cluster nodes. Please consider limit '
                 'internet connection on your own. (Native internet isolation '
                 'is not available for VmWare inside si-tests now)')
        proxy_access_str = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    elif settings.KAAS_INSTALL_FAKE_PROXY:
        proxy_manager.deploy_proxy_sts()
        proxy_manager.deploy_proxy_svc()
        proxy_access_str = proxy_manager.generate_proxy_string()
        proxy_manager.check_proxy_conn()
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    else:
        proxy_object_name = None

    show_step(7)
    metallb_ip_range = settings.KAAS_CHILD_VSPHERE_METALLB_RANGE

    # All managed clusters with 2.25 MCC should be created with MetallbConfig
    ns.create_metallbconfig_substitutive_inlineconfig(cluster_name=cluster_name,
                                                      region=region,
                                                      ip_range=metallb_ip_range,
                                                      provider_name=settings.VSPHERE_PROVIDER_NAME)
    metallb_ip_range = None

    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        credentials_name=credential_name,
        region=region,
        provider="vsphere",
        loadbalancer_host=settings.KAAS_CHILD_VSPHERE_LB_HOST,
        services_cidr="10.96.0.0/16",
        pods_cidr="192.168.0.0/16",
        nodes_cidr="10.10.10.0/24",
        metallb_ip_range=metallb_ip_range,
        dns=child_dns,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        public_key_name=public_key_name,
        proxy_name=proxy_object_name,
        ipam_enabled=settings.KAAS_VSPHERE_IPAM_ENABLED,
        ipam_cidr=settings.KAAS_CHILD_VSPHERE_NETWORK_CIDR,
        ipam_gw=settings.KAAS_VSPHERE_NETWORK_GATEWAY,
        ipam_ns=settings.KAAS_CHILD_VSPHERE_NETWORK_NAMESERVERS,
        ipam_incl_range=settings.KAAS_CHILD_VSPHERE_NETWORK_INCLUDE_RANGES,
        secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED)

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    vm_templates = utils.vsphere_set_vm_template_list()
    if settings.VSPHERE_USE_VVMT_OBJECTS:
        # filter out templates which aren't related to the current region
        vvmt_objects = [vvmt for vvmt in kaas_manager.list_all_vvmt()
                        if vvmt.region == region]
        vm_templates = [vvmt.template_path for vvmt in vvmt_objects]
    if len(vm_templates) == 0:
        raise RuntimeError("No VM templates provided for machines")
    for i in range(master_nodes_count):
        idx = i % len(vm_templates)
        template = vm_templates[idx]
        rhel_license = ""
        if settings.VSPHERE_USE_VVMT_OBJECTS:
            rhel_license = rhel_license_name if vvmt_objects[idx].os_name == "rhel" else ""
        else:
            rhel_license = utils.vsphere_set_rhel_license(template, rhel_license_name)
        cluster.create_vsphere_machine(
            node_type="master",
            template=template,
            rhel_license=rhel_license,
            region=region)

    labels = \
        [{'key': k, 'value': v} for k, v in
         cluster.get_allowed_node_labels().items()]

    stacklight_nodes_count = \
        3 if slave_nodes_count >= 3 else slave_nodes_count
    for i in range(stacklight_nodes_count):
        idx = i % len(vm_templates)
        template = vm_templates[idx]
        rhel_license = ""
        if settings.VSPHERE_USE_VVMT_OBJECTS:
            rhel_license = rhel_license_name if vvmt_objects[idx].os_name == "rhel" else ""
        else:
            rhel_license = utils.vsphere_set_rhel_license(template, rhel_license_name)
        cluster.create_vsphere_machine(
            node_type="node",
            template=template,
            rhel_license=rhel_license,
            region=region,
            node_labels=labels)

    slave_nodes_count -= stacklight_nodes_count

    for i in range(slave_nodes_count):
        idx = i % len(vm_templates)
        template = vm_templates[idx]
        rhel_license = ""
        if settings.VSPHERE_USE_VVMT_OBJECTS:
            rhel_license = rhel_license_name if vvmt_objects[idx].os_name == "rhel" else ""
        else:
            rhel_license = utils.vsphere_set_rhel_license(template, rhel_license_name)
        cluster.create_vsphere_machine(
            node_type="node",
            template=template,
            rhel_license=rhel_license,
            region=region)

    # Waiting for machines are Ready and helmbundles are deployed
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()

    machines = [
        x for x in cluster.get_machines(machine_type="worker")
        if 'nodeLabels' in x.spec['providerSpec']['value'].keys() and
        labels == x.nodeLabels
    ]
    assert len(machines) == stacklight_nodes_count,\
        "Not all machines have expected nodeLabels"

    for machine in machines:
        machine.check_k8s_nodes_has_labels()

    show_step(8)
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()
    if settings.KAAS_VSPHERE_IPAM_ENABLED:
        cluster.check.check_deploy_stage_success()
    else:
        cluster.check.check_deploy_stage_success(skipped_stages_names='Network prepared')

    # Collecting artifacts
    cluster.store_k8s_artifacts()

    cluster.check.check_k8s_nodes()

    # Check that Machines hostnames are created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines()

    show_step(9)
    if settings.RUN_POD_CHECKS:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods(timeout=3200,
                                                 check_all_nss=True)
    cluster.check.check_overlay_encryption_functionality()

    cluster.provider_resources.save_artifact()

    show_step(10)
    if settings.KAAS_OFFLINE_DEPLOYMENT:
        cluster.check.check_offline_isolation(
            settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
