import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
def test_kaas_create_aws_child_negative(kaas_manager, show_step, _):
    """Deploy AWS child cluster on top of management cluster with
       wrong flavor and image for machines

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating aws credential
        5. Create child cluster
        6. Delete cluster/key/credentials/ns

    """

    # Collecting env data
    show_step(1)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    aws_region = settings.AWS_DEFAULT_REGION
    aws_az = settings.AWS_DEFAULT_AZ

    avail_aws_releases = kaas_manager.get_supported_clusterrelease_names(
        provider=settings.AWS_PROVIDER_NAME)
    release_name = sorted(avail_aws_releases)[-1]
    LOG.info(f"Using clusterrelease {release_name}")
    instance_type = settings.KAAS_CHILD_CLUSTER_MACHINE_FLAVOR_NAME
    master_nodes_count = 3
    rnd_string = utils.gen_random_string(6)

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_or_create_namespace(namespace_name)
    region = kaas_manager.get_mgmt_cluster().region_name

    # Creating public key
    show_step(3)
    public_key_name = "{cluster_name}-{rnd_string}-key".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    pubkey = ns.create_publickey(public_key_name, pub_key_content)

    # Get aws bootstrap credentials from management cluster
    aws_key_id, aws_secret_access_key = kaas_manager.get_aws_credential(
        "cloud-config", "default")

    # Creating aws credential
    show_step(4)
    credential_name = "{cluster_name}-{rnd_string}-cred".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))
    creds = ns.create_aws_credential(
        credential_name, aws_key_id, aws_secret_access_key, region)

    show_step(5)
    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        credential_name,
        region=region,
        provider="aws",
        aws_region=aws_region,
        aws_az=aws_az,
        aws_priv_subnet_cidr="10.0.0.0/24",
        aws_pub_subnet_cidr="10.0.1.0/24",
        services_cidr="10.96.0.0/16",
        pods_cidr="192.168.0.0/16",
        nodes_cidr="10.10.10.0/24",
        lma_enabled=False,
        public_key_name=public_key_name)

    try:
        for node in range(master_nodes_count):
            cluster.create_aws_machine(
                node_type="master",
                instance_type=instance_type,
                ami="nonexisting_image",
                root_device_size=120,
                region=region
            )
    except Exception as e:
        LOG.info(e)
    else:
        raise Exception("No error occurred during aws machine creation "
                        "with non-exisitng image")

    try:
        for node in range(master_nodes_count):
            cluster.create_aws_machine(
                node_type="master",
                instance_type="nonexisting_type",
                ami=settings.KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME,
                root_device_size=120,
                region=region
            )
    except Exception as e:
        LOG.info(e)
    else:
        raise Exception("No error occurred during aws machine creation "
                        "with non-exisitng instance type")

    show_step(6)
    cluster.delete()
    pubkey.delete()
    creds.delete()
    ns.delete()
