#    Copyright 2019 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import json
import os
import pkg_resources

from si_tests import feature_flags

_boolean_states = {'1': True, 'yes': True, 'true': True, 'on': True,
                   '0': False, 'no': False, 'false': False, 'off': False}


# WARNING: env variable name MUST not be started with "CI_"
# unless - it will be cleaned by kaas-bm CI during jenkins mangle!

# In case if environment variable is set but contains an empty string,
# use the default instead of empty value
def get_var(name, default=''):
    value = os.environ.get(name) or None
    return value or default


def get_var_as_bool(name, default):
    value = get_var(name, '')
    if isinstance(default, str):
        default = _boolean_states.get(default.lower(), default)
        if isinstance(default, str):
            raise Exception(f"default for '{name}' contains some non-binary trash: '{default}'")
    return _boolean_states.get(value.lower(), default)


def get_boolean_states_dict() -> dict:
    return _boolean_states.copy()


def get_var_from_json(name, default="{}"):
    values = json.loads(get_var(name, default))
    return values


# determine seed node os for future cmd preparations
def parse_seed_os_type(os_type):
    supported = ['linux', 'darwin']
    desired_os = os_type.strip().lower()
    assert desired_os in supported, ("Incompatible OS type: {0}, supported {1}"
                                     .format(desired_os, supported))

    return desired_os


# Feature flags
KNOWN_FLAGS = [
    "kaas-exporter",
    "relay-non-ha",
    "remove-tmp-patroni",
    "remove-squid",
    "mcc-metric-collector",
    "user-controller",
    "scope-controller",
    "lvp-in-kube-system-ns",
    "regional-telemetry",
    "minio",
    "minio25",
    "minio26",
    "minio27",
    "tekton",
    "testcontroller",
    "testcontroller24",
    "testcontroller25",
    "testcontroller26",
    "testcontroller27",
    "testcontroller-admissioncontroller",
    "proxy-controller",
    "sl-ucp-mcc-rename",
    "sl-metric-collector-rename",
    "iam-mariadb-controller",
    "squid-proxy",
    "nginx-cache",
    "mcc-cache",
    "rhellicense-controller",
    "vsphere-kaas-ipam",
    "ssh-key-common-naming",
    "alerta-ha",
    "equinixmetal",
    "patroni-13",
    "remove-netchecker",
    "remove-patroni-12",
    "azure",
    "ironic-prometheus-exporter",
    "tarball-update",
    "squid-proxy-in-vsphere-only",
    "disable-telegraf-arping",
    "sl-node-placement-enabled",
    "container-cloud-binary",
    "agent-controller",
    "rbac-controller",
    "new-public-ci",
    "add-api-server-to-noproxy",
    "remove-grafana-test",
    "ceph-split-controllers",
    "disable-pushgateway",
    "license-controller",
    "bm-enforce-distr",
    "event-controller",
    "certificate-rotation",
    "machinepool-controller",
    "aws-external-cloud-controller",
    "public-ci-azure",
    "es-os-renaming",
    "disable-iam-api",
    "ceph-maintenance-controller",
    "no-mgmt-ceph",
    "empty-noproxy-bootstrap",
    "no-mgmt-ceph-upgrade",
    "new-metallb-PRODX-25781",
    "bm-dhcp-relay",
    "refapp",
    "refapp-v2",
    "refapp-new-name",
    "configuration-collector",
    "aws-PRODX-27892",
    "upgrade-history",
    "sl-custom-certs",
    "vsphere-vm-template-controller",
    "ceph-2-mgrs",
    "ext-azure-ccm",
    "cephfs-ga",
    "cadvisor",
    "iam-proxy-ha",
    "kube-state-metrics-ha",
    "blackbox-exporter-ha",
    "ceph-disk-daemon",
    "drop_telegraf_openstack",
    "no-ceph-ns-regional",
    "alertmanager-ha-split",
    "si-guess-bootstrap-templates",
    "management-admin-oidc",
    "disabled-config-collector",
    "chown-daemonset",
    "metallbconfigs-child",
    "migrate-os-ccm",
    "force-plain-dashboard-auth",
    "enabled-config-collector",
    "upgrade-no-leftovers-check",
    "upgrade-no-leftovers-check-skip-kernel",
    "vsphere-out-of-tree",
    "kubernetes-audit",
    "byo",
    "host-os-modules",
    "policy-controller",
    "secret-controller",
    "diagnostic-controller",
    "credentials-controller",
    "update-groups",
    "update-auto-pause",
    "mariadb-iam-bump",
    "bm-netchecker",
    "machine-pauses",
    "nfd-bump",
    "license-controller-bootstrap"
]
FEATURE_FLAGS = feature_flags.parse(
    os.environ.get(
        "SI_TESTS_FEATURE_FLAGS", ""), KNOWN_FLAGS,
)

# CDN mapping
CDN_REGION = get_var('CDN_REGION', 'public')
CDN_REGION_DICT = {
    "internal-ci":
        "https://artifactory.mcp.mirantis.net/"
        "artifactory/binary-dev-kaas-virtual",
    "internal-ci-eu":
        "https://artifactory-eu.mcp.mirantis.net/"
        "artifactory/binary-dev-kaas-virtual",
    "internal-eu":
        "https://artifactory-eu.mcp.mirantis.net/"
        "artifactory/binary-dev-kaas-virtual",
    "public-ci":
        "https://binary-dev-kaas-virtual.mcp.mirantis.com",
    "public":
        "https://binary.mirantis.com",
    "kaas-2-12-0-internal-ci":
        "https://artifactory.mcp.mirantis.net/"
        "artifactory/binary-dev-kaas-virtual",
    "kaas-2-12-0-internal-eu":
        "https://artifactory-eu.mcp.mirantis.net/"
        "artifactory/binary-dev-kaas-virtual",
    "kaas-2-12-0-public-ci":
        "https://binary-dev-kaas-mirantis-com.s3.amazonaws.com",
    "kaas-2-12-0-public":
        "https://binary.mirantis.com",
}

ARTIFACTS_REGION_MAPPING = {
    "public": {
        "images_base_url": 'mirantis.azurecr.io',
        "binary_base_url": CDN_REGION_DICT[CDN_REGION]
    },
    "internal-ci": {
        "images_base_url": 'docker-dev-kaas-virtual.docker.mirantis.net',
        "binary_base_url": CDN_REGION_DICT[CDN_REGION]
    },
    "internal-ci-eu": {
        "images_base_url": 'docker-dev-kaas-virtual.artifactory-eu.mcp.mirantis.net',
        "binary_base_url": CDN_REGION_DICT[CDN_REGION]
    },
    "internal-eu": {
        "images_base_url": 'docker-dev-kaas-virtual.artifactory-eu.mcp.mirantis.net',
        "binary_base_url": CDN_REGION_DICT[CDN_REGION]
    },
    "public-ci": {
        "images_base_url":
        "mirantisdevkaas.azurecr.io" if
        FEATURE_FLAGS.enabled("public-ci-azure") else
        "docker-dev-kaas-virtual.mcp.mirantis.com",
        "binary_base_url": CDN_REGION_DICT[CDN_REGION]
    }
}


# Helper function to get image path based on CDN
def get_full_image_path(path):
    images_base_url = ARTIFACTS_REGION_MAPPING[CDN_REGION]["images_base_url"]
    return f"{images_base_url}/{path}"


# Exact root of https://gerrit.mcp.mirantis.com/kaas/si-tests repo
# By default, local dir
SI_TESTS_REPO_ROOT = get_var('SI_TESTS_REPO_ROOT', './')
SI_OPENID_AUTH = get_var_as_bool('SI_OPENID_AUTH', False)
SI_OPENID_USERNAME = get_var('SI_OPENID_USERNAME', 'writer')
SI_OPENID_PASSWORD = get_var('SI_OPENID_PASSWORD', None)
SI_OPENID_IDP_CA = get_var('SI_OPENID_IDP_CA', None)

# Jenkins build url, empty by default
JENKINS_BUILD_URL = get_var('BUILD_URL', '')
JENKINS_BUILD_NUMBER = get_var('BUILD_NUMBER', '')

ARTIFACTS_DIR = get_var(
    'ARTIFACTS_DIR',
    os.path.join(os.getcwd(), 'artifacts/'))
if not os.path.exists(ARTIFACTS_DIR):
    os.makedirs(ARTIFACTS_DIR)

LOGS_DIR = get_var('LOGS_DIR', ARTIFACTS_DIR)
LOG_NAME = get_var('LOG_NAME', 'tests.log')
LOG_REST_DEBUG = get_var("LOG_REST_DEBUG", True)

ENV_NAME = get_var("ENV_NAME")
ENV_CONFIG_NAME = get_var("ENV_CONFIG_NAME", "")
ENV_CONFIG_BASEDIR = os.path.expanduser(get_var(
    "ENV_CONFIG_BASEDIR", pkg_resources.resource_filename(
        __name__, "kaas-bm-env/labs")))
ENV_PARAMETERS_NAME = get_var("ENV_PARAMETERS_NAME", "defaults.yaml")
# Must be used only for kaas-mgmt scope!
CLUSTER_NAME = get_var("CLUSTER_NAME", 'si-kaas-management')
CLUSTER_NAMESPACE = get_var("CLUSTER_NAMESPACE", "default")

KEEP_ENV_BEFORE = get_var_as_bool("KEEP_ENV_BEFORE", False)
KEEP_ENV_AFTER = get_var_as_bool("KEEP_ENV_AFTER", True)

KAAS_BASTION_SSH_LOGIN = get_var('KAAS_BASTION_SSH_LOGIN', 'mcc-user')

SEED_SSH_LOGIN = get_var('SEED_SSH_LOGIN', 'ubuntu')
SEED_SSH_PRIV_KEY_FILE = os.path.expanduser(
    get_var('SEED_SSH_PRIV_KEY_FILE', '~/.ssh/id_rsa'))

KAAS_MGMT_CLUSTER_BOOTSTRAP_USERSPACE = get_var(
    "MGMT_CLUSTER_BOOTSTRAP_USERSPACE", "ubuntu")
SEED_NODE_MCC_ARTIFACTS_BASE_DIR = "/home/{}/cluster_artifacts".format(
    KAAS_MGMT_CLUSTER_BOOTSTRAP_USERSPACE)

# Scenario, when seed node does not managed by SI tests
SEED_STANDALONE_EXTERNAL_IP = get_var('SEED_STANDALONE_EXTERNAL_IP', '')
SEED_STANDALONE_OS_TYPE = parse_seed_os_type(
    get_var('SEED_STANDALONE_OS_TYPE', 'linux')
)

# Inject MCC metadata into seed node instance properties
CHAIN_SEED_WITH_MCC = get_var_as_bool("CHAIN_SEED_WITH_MCC", True)

# List of deb pkgs, to be installed during step_003_prepare_seed_node_base.
# Space separated.
SEED_NODE_EXTRA_PKG = get_var('SEED_NODE_EXTRA_PKG', '')
SEED_NODE_EXTRA_DO_UPGRADE = get_var('SEED_NODE_EXTRA_DO_UPGRADE', False)
# Shoule we reboot node during step_003_prepare_seed_node_base ?
SEED_NODE_EXTRA_PKG_REBOOT_AFTER = get_var_as_bool(
    'SEED_NODE_EXTRA_PKG_REBOOT_AFTER', False)
SEED_CERTS_DIR = get_var(
    'SEED_CERTS_DIR', '/usr/local/share/ca-certificates/custom_proxy.crt')

KAAS_SKIP_BOOTSTRAP_CLEANUP = get_var_as_bool("KAAS_SKIP_BOOTSTRAP_CLEANUP",
                                              False)
BM_COLLECT_LOGS = get_var_as_bool("BM_COLLECT_LOGS", True)

KAAS_FORCE_FAIL_ON_BOOTSTRAP_CLEANUP = get_var_as_bool(
    "KAAS_FORCE_FAIL_ON_BOOTSTRAP_CLEANUP", False)
KAAS_MGMT_CLUSTER_PRIVATE_KEY_FILE = get_var(
    "KAAS_MGMT_CLUSTER_PRIVATE_KEY_FILE", "~/.ssh/id_rsa")
KAAS_MGMT_CLUSTER_SSH_LOGIN = get_var(
    "KAAS_MGMT_CLUSTER_SSH_LOGIN", "mcc-user")

KAAS_PRIVATE_KEY_FILE_PATH = get_var("KAAS_PRIVATE_KEY_FILE_PATH", "")
# quick hot-fix to be able to pass exact path at seed node, for kubeconfig
# Must be used only under si_tests/tests/deployment/test_collect_logs.py
# Example = kaas-bootstrap/kubeconfig-mcc-mgmt
KAAS_MGMT_KUBECONFIG_ON_SEED_PATH = get_var("KAAS_MGMT_KUBECONFIG_ON_SEED_PATH", "")

# START: KAAS-BM internal CI opts
# File with bash-env variables in a yaml dict,to be passed to ansible.
# Usually, file located at target seed node0
ANSIBLE_ENV_OVERRIDE_FILE = get_var(
    "ANSIBLE_ENV_OVERRIDE_REMOTE_FILE",
    '.ansible_state/ansible_env_override.yaml')

# We(kaas-bm per commit)
# need to support both cases - ['kaas_core','kaas_releases']
# in ansible as trigger way
KAAS_BM_DEPLOY_KAAS_BASED_ON_SOURCE = \
    get_var("KAAS_BM_DEPLOY_KAAS_BASED_ON_SOURCE", "")

KAAS_BM_CHARTS_REPO = get_var('KAAS_BM_CHARTS_REPO',
                              'https://artifactory.mcp.mirantis.net/'
                              'artifactory/binary-dev-kaas-virtual/bm/helm')
# baremetal-public-api
KAAS_BM_PUBLIC_API_CHART_URL = \
    get_var('KAAS_BM_PUBLIC_API_CHART_URL', '')
KAAS_BM_PUBLIC_API_CHART_VERSION = \
    get_var('KAAS_BM_PUBLIC_API_CHART_VERSION', '')
# baremetal-operator
KAAS_BM_OPERATOR_CHART_URL = \
    get_var('KAAS_BM_OPERATOR_CHART_URL', '')
KAAS_BM_OPERATOR_CHART_VERSION = \
    get_var('KAAS_BM_OPERATOR_CHART_VERSION', '')
KAAS_BM_OPERATOR_DOCKER_IMAGE = \
    get_var('KAAS_BM_OPERATOR_DOCKER_IMAGE', '')
# kaas-ipam
KAAS_BM_IPAM_CHART_URL = get_var('KAAS_BM_IPAM_CHART_URL', '')
KAAS_BM_IPAM_CHART_VERSION = get_var('KAAS_BM_IPAM_CHART_VERSION', '')
KAAS_BM_IPAM_DOCKER_IMAGE = get_var('KAAS_BM_IPAM_DOCKER_IMAGE', '')
KAAS_BM_KAAS_IPAM_DOCKER_IMAGE = get_var(
    'KAAS_BM_KAAS_IPAM_DOCKER_IMAGE', '')
KAAS_BM_KAAS_ANSIBLE_TAR_GZ = get_var('KAAS_BM_KAAS_ANSIBLE_TAR_GZ', '')
KAAS_BM_KAAS_ANSIBLE_TAR_GZ_MD5 = get_var(
    'KAAS_BM_KAAS_ANSIBLE_TAR_GZ_MD5', '')
# ironic-operator
KAAS_BM_IRONIC_OPERATOR_DOCKER_IMAGE = \
    get_var('KAAS_BM_IRONIC_OPERATOR_DOCKER_IMAGE', '')

KAAS_BM_UPGRADE_SPEC_FILE = get_var('KAAS_BM_UPGRADE_SPEC_FILE')
KAAS_BM_CI_SCALEDOWNRESOURCES = \
    get_var_as_bool('KAAS_BM_CI_SCALEDOWNRESOURCES', False)
KAAS_BM_BOOT_UEFI = \
    get_var_as_bool('KAAS_BM_BOOT_UEFI', True)
KAAS_BM_MGMT_KUBECONFIG_REMOTE_FILE = \
    get_var('KAAS_BM_MGMT_KUBECONFIG_REMOTE_FILE', 'management_kubeconfig')
ASYNC_BMH_CONTROL_NODES_WAIT = get_var_as_bool('ASYNC_BMH_CONTROL_NODES_WAIT',
                                               False)
KAAS_BM_IGNORE_CEPH_FAILURE = get_var_as_bool('KAAS_BM_IGNORE_CEPH_FAILURE',
                                              False)
KAAS_BM_WA_PRODX_6630 = get_var_as_bool('KAAS_BM_WA_PRODX_6630', True)
KAAS_BM_AIO_ON_MGMT = get_var_as_bool('KAAS_BM_AIO_ON_MGMT', False)
KAAS_BM_AIO_ON_CHILD = get_var_as_bool('KAAS_BM_AIO_ON_CHILD', False)
KAAS_AIO_CLUSTER = get_var_as_bool('AIO_CLUSTER', False)

# https://binary.mirantis.com/bm/bin/efi/ubuntu/tgz-bionic-20210426111442
TEST_CUSTOM_ROOTFSURL = get_var('TEST_CUSTOM_ROOTFSURL', '')

KAAS_BM_CI_ON_EQUINIX = get_var_as_bool('KAAS_BM_CI_ON_EQUINIX', False)
KAAS_BM_CI_EQUINIX_API_TOKEN = get_var('KAAS_BM_CI_EQUINIX_API_TOKEN', None)
KAAS_BM_CI_EQUINIX_PROJECT_ID = get_var('KAAS_BM_CI_EQUINIX_PROJECT_ID', None)
KAAS_BM_CI_EQUINIX_METRO = get_var('KAAS_BM_CI_EQUINIX_METRO', 'sv')
KAAS_BM_CI_EQUINIX_FACILITY = get_var('KAAS_BM_CI_EQUINIX_FACILITY', 'sv15')
KAAS_BM_CI_EQUINIX_PLAN = get_var('KAAS_BM_CI_EQUINIX_PLAN', 's3.xlarge.x86')
KAAS_BM_CI_EQUINIX_OS = get_var('KAAS_BM_CI_EQUINIX_OS', 'ubuntu_20_04')
KAAS_BM_CI_EQUINIX_USERDATA = get_var(
    'KAAS_BM_CI_EQUINIX_USERDATA',
    pkg_resources.resource_filename(__name__,
                                    'templates/kaas-bm/userdata-equnix-seed-kaas-bm.yaml'))  # noqa
KAAS_BM_CI_ON_EQUINIX = get_var_as_bool('KAAS_BM_CI_ON_EQUINIX', False)
# END: KAAS-BM internal CI opts

KAAS_BM_TEMPLATES_DIR = os.path.expanduser(get_var(
    "KAAS_BM_TEMPLATES_DIR", pkg_resources.resource_filename(__name__, "templates/kaas-bm")))

CLOUDS_YAML_PATH = os.path.expanduser(get_var(
    "CLOUDS_YAML_PATH",
    pkg_resources.resource_filename(__name__,
                                    'templates/bootstrap/clouds_si_eu.yaml')))

REGIONAL_CLOUDS_YAML_PATH = get_var(
    "REGIONAL_CLOUDS_YAML_PATH", CLOUDS_YAML_PATH)

CLOUDS_FAIL_YAML_PATH = os.path.expanduser(get_var(
    "CLOUDS_FAIL_YAML_PATH",
    pkg_resources.resource_filename(
        __name__,
        'templates/bootstrap/failed_clouds.yaml')))

CLOUD_NAME = get_var("CLOUD_NAME", "openstack")
OS_AZ = get_var("OS_AZ", None)
OS_AUTH_URL = get_var("OS_AUTH_URL", 'https://keystone.ic-eu.ssl.mirantis.net/v3')
CLOUD_KEY_NAME = get_var("CLOUD_KEY_NAME", None)
PHYS_NETWORK = get_var("PHYS_NETWORK", None)
PHYS_SUBNET = get_var("PHYS_SUBNET", None)
PHYS_SEED_FIXED_IP = get_var("PHYS_SEED_FIXED_IP", None)
PHYS_NETMASK = get_var("PHYS_NETMASK", None)
PHYS_GATEWAY = get_var("PHYS_GATEWAY", None)
DNS_NAMESERVER = get_var("DNS_NAMESERVER", None)

# openstack provider related
KAAS_OPENSTACK_ENABLED = get_var_as_bool("KAAS_OPENSTACK_ENABLED", False)

# Cleanup OS cloud resources reletaed
FILTERED_STACKS_PREFIX = list(
    filter(None, get_var('FILTERED_STACKS_PREFIX', 'system-phys-').split(',')))

# If STACKS_TO_CLEANUP_PREFIX is set, FILTERED_STACKS_PREFIX will be ignored
STACKS_TO_CLEANUP_PREFIX = list(
    filter(None, get_var('STACKS_TO_CLEANUP_PREFIX', '').split(',')))
TRIGGER_RESOURCES_CLEANUP = get_var_as_bool('TRIGGER_RESOURCES_CLEANUP', False)
TRIGGER_STACKS_CLEANUP = get_var_as_bool('TRIGGER_STACKS_CLEANUP', False)
RESOURCE_TO_CLEANUP_PREFIX = tuple(
    filter(None, get_var('RESOURCE_TO_CLEANUP_PREFIX', 'kaas').split(',')))

HEAT_TEMPLATE_PATH = get_var(
    "HEAT_TEMPLATE_PATH",
    pkg_resources.resource_filename(__name__,
                                    'templates/heat/kaas-vm-seed-node.hot'))

HEAT_ENV_PATH = get_var(
    "HEAT_ENV_PATH",
    pkg_resources.resource_filename(
        __name__, 'templates/heat/envs/eu-cloud-vm-seed.env'))
# automatically get .hot\.env files from SI_HEAT_AUTO_GUESS_PATH
SI_HEAT_AUTO_GUESS_CONFIG = get_var_as_bool('SI_HEAT_AUTO_GUESS_CONFIG', False)
SI_HEAT_AUTO_GUESS_PATH = os.path.expanduser(get_var(
    "SI_HEAT_AUTO_GUESS_PATH", pkg_resources.resource_filename(
        __name__,
        'kaas-bm-env/labs/{}/heat_data/'.format(
            ENV_CONFIG_NAME or ENV_NAME))))
# used only in vbmc.template bm + shipmi
SI_SHIPMI_PROJECT_NAME = get_var('SI_SHIPMI_PROJECT_NAME', '')
SI_SHIPMI_PASSWORD = get_var('SI_SHIPMI_PASSWORD', '')
SI_SHIPMI_USERNAME = get_var('SI_SHIPMI_USERNAME', '')
SI_SHIPMI_AUTH_URL = get_var('SI_SHIPMI_AUTH_URL', '')
OS_HEAT_VERSION = get_var('OS_HEAT_VERSION', 1)

KAAS_CLUSTER_NAME = get_var("KAAS_CLUSTER_NAME", "kaas")
KAAS_PLATFORM_RELEASE_REF = get_var("KAAS_PLATFORM_RELEASE_REF", "")
# Deprecated?
KAAS_BOOTSTRAP_URL = get_var("KAAS_BOOTSTRAP_URL", '')
KAAS_BOOTSTRAP_TEMPLATES_DIR = os.path.expanduser(get_var(
    "KAAS_BOOTSTRAP_TEMPLATES_DIR",
    pkg_resources.resource_filename(
        __name__,
        'templates/bootstrap/')))

KAAS_BOOTSTRAP_TIMEOUT = int(get_var("KAAS_BOOTSTRAP_TIMEOUT", 7200))
KAAS_UPDATE_MGMT_WAIT_TIMEOUT = int(get_var("KAAS_UPDATE_MGMT_WAIT_TIMEOUT",
                                            40 * 60))
# Since kaas-2-29-5 active label not set while charts are not ready, keep wait timeout higher
KAAS_UPDATE_MGMT_ACTIVE_WAIT_TIMEOUT = int(get_var("KAAS_UPDATE_MGMT_ACTIVE_WAIT_TIMEOUT", 120 * 60))
SKIP_BOOTSTRAP_TEMPLATES_CHECK = get_var_as_bool(
    "SKIP_BOOTSTRAP_TEMPLATES_CHECK", False)
SKIP_CHILD_EXPECTED_POD_CHECK = get_var_as_bool("SKIP_CHILD_EXPECTED_POD_CHECK", False)
CHECK_ACTUAL_EXPECTED_PODS_TIMEOUT = int(get_var("CHECK_ACTUAL_EXPECTED_PODS_TIMEOUT", 5 * 60))

# Used to disable Stacklight in custom templates.
# SKIP_BOOTSTRAP_TEMPLATES_CHECK variable should be set to True to use that.
# Useful on core-ci scenarios
KAAS_CUSTOM_LMA_LOGGING_ENABLED = get_var_as_bool(
    "KAAS_CUSTOM_LMA_LOGGING_ENABLED", True)

WAIT_PODS_READY_TIMEOUT = int(get_var("WAIT_PODS_READY_TIMEOUT", 4200))

KAAS_UI_TEST_POD_YAML = os.path.expanduser(get_var(
    'KAAS_UI_TEST_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/kaas_ui_pod.yaml')))

KAAS_UI_TEST_NAMESPACE = get_var('KAAS_UI_TEST_NAMESPACE', 'kaas')
KAAS_UI_CHILD_CLUSTER_NAME = get_var('KAAS_UI_CHILD_CLUSTER_NAME', 'kaas-ui-child')
KAAS_UI_TEST_POD_NAME = get_var('KAAS_UI_TEST_POD_NAME', 'ui-e2e-test')
KAAS_UI_TEST_REPORTS_DIR = get_var('KAAS_UI_TEST_REPORTS_DIR', '/report')
KEEP_KAAS_UI_TEST_POD_AFTER = get_var_as_bool("KEEP_KAAS_UI_TEST_POD_AFTER",
                                              True)
KAAS_UI_TEST_TYPE = get_var('KAAS_UI_TEST_TYPE', 'e2e')
KAAS_UI_E2E_TEST_NS_PREFIX = get_var('KAAS_UI_E2E_TEST_CLUSTER_NAME_PREFIX', 'e2e-test-ns')

KAAS_UI_TEST_IMAGE = get_var('KAAS_UI_TEST_IMAGE',
                             'core/test/frontend')
KAAS_UI_TEST_IMAGE_TAG = get_var('KAAS_UI_TEST_IMAGE_TAG', '')

IAM_BDD_TEST_POD_NAME = get_var('IAM_BDD_TEST_POD_NAME', 'iam-bdd')
IAM_BDD_TEST_POD_YAML = os.path.expanduser(get_var(
    'IAM_BDD_TEST_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/iam_bdd_pod.yaml')))

IAM_BDD_TEST_REPORTS_DIR = get_var('IAM_BDD_TEST_REPORTS_DIR',
                                   '/report')

IAM_BDD_TEST_IMAGE = get_var('IAM_BDD_TEST_IMAGE', 'iam/e2e')
IAM_VERSION = get_var('IAM_VERSION')
# Max approximate time to reconfigure iam statefulset to schedule future mariadb keycloak backup
IAM_APPROX_RECONFIGURE_TIME = int(get_var('IAM_APPROX_RECONFIGURE_TIME', 300))
MCC_MARIADB_RESTORE_JOB_TIMEOUT = int(get_var("MCC_MARIADB_RESTORE_JOB_TIMEOUT", 1500))

KEYCLOAK_LDAP_ENABLED = get_var_as_bool('KEYCLOAK_LDAP_ENABLED', False)
# LDAP bind DN and bind credentials that has rights to access the ldap server
# Optional. If empty - will not be used
KEYCLOAK_LDAP_BIND_USERNAME = get_var('KEYCLOAK_LDAP_BIND_USERNAME', '')
KEYCLOAK_LDAP_BIND_PASSWORD = get_var('KEYCLOAK_LDAP_BIND_PASSWORD', '')

# Used to enable Keycloak+LDAP integration for core templates (e.g for templates from kaas/core repo).
# Use only with skipping bootstrap templates check and only for core repo - assuming that iam does not configured
# in cluster templates
# If used - you may additionally want to set KEYCLOAK_LDAP_ENABLED for LDAP verification.
CORE_KEYCLOAK_LDAP_ENABLED = get_var_as_bool('CORE_KEYCLOAK_LDAP_ENABLED', False)

KEYCLOAK_USER = get_var('KEYCLOAK_USER', "keycloak")
KEYCLOAK_PASSWORD = get_var('KEYCLOAK_PASSWORD', "")
KEYCLOAK_URL = get_var('KEYCLOAK_URL')

# If generate password is set to False, then the default 'password' string will be used
# as password for the users 'writer', 'reader' and 'operator'. If True - the password string will be generated.
KEYCLOAK_USERS_GENERATE_PASSWORD = get_var_as_bool('KEYCLOAK_USERS_GENERATE_PASSWORD', True)

MCC_LDAP_UI_USER = get_var('MCC_LDAP_UI_USER')
MCC_LDAP_UI_PASSWORD = get_var('MCC_LDAP_UI_PASSWORD')

CREATE_DEFAULT_MCC_USERS_VIA_CLI = get_var_as_bool(
    'CREATE_DEFAULT_MCC_USERS_VIA_CLI', True)

UCP_UI_URL = get_var('UCP_UI_URL', "")
UCP_UI_USER = get_var('UCP_UI_USER', "admin")
UCP_UI_PASSWORD = get_var('UCP_UI_PASSWORD', "topsecret")

# usually tls/ca.pem in bootstrap folder
IAM_CACERT = get_var('IAM_CACERT', "")

IAM_BDD_TEST_TAGS = get_var('IAM_BDD_TEST_TAGS', "~@k8s")

SVC_CHECK_POD_YAML = os.path.expanduser(get_var(
    'SVC_CHECK_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/svc_check_pod.yaml')))

MACHINE_PRIVELEGED_POD_YAML = os.path.expanduser(get_var(
    'MACHINE_PRIVELEGED_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/machine_priveleged_pod.yaml')))

CALICOCTL_PROVELEGED_POD_YAML = os.path.expanduser(get_var(
    'CALICOCTL_PROVELEGED_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/machine_priveleged_calicoctl.yaml')))

K8S_CONFORMANCE_POD_YAML = os.path.expanduser(get_var(
    'K8S_CONFORMANCE_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/k8s_conformance_test_pod.yaml')))

K8S_CONFORMANCE_NAMESPACE = get_var('K8S_CONFORMANCE_NAMESPACE',
                                    'conformance')
K8S_CONFORMANCE_POD_NAME = get_var('K8S_CONFORMANCE_POD_NAME',
                                   'conformance')
K8S_CONFORMANCE_REPORTS_DIR = get_var('K8S_CONFORMANCE_REPORTS_DIR',
                                      '/report')

K8S_CONFORMANCE_RESULTS_WAIT_TIMEOUT = int(get_var(
    "K8S_CONFORMANCE_RESULTS_WAIT_TIMEOUT", 5400
))

K8S_CONFORMANCE_IMAGE = get_var(
    'K8S_CONFORMANCE_IMAGE',
    'lcm/kubernetes/k8s-conformance'
)

# TODO: remove when PRODX-4679 is Done
K8S_CONFORMANCE_IMAGE_VERSION_CHECK_PATH = \
    get_var('K8S_CONFORMANCE_IMAGE_VERSION_CHECK_PATH',
            'https://docker-dev-kaas-virtual.docker.mirantis.net/artifactory/docker-dev-kaas-virtual/' +
            K8S_CONFORMANCE_IMAGE)

K8S_CONFORMANCE_IMAGE_VERSION = get_var(
    'K8S_CONFORMANCE_IMAGE_VERSION',
    ''
)
K8S_CONFORMANCE_CLUSTER_DOMAIN = get_var(
    'K8S_CONFORMANCE_CLUSTER_DOMAIN',
    ''
)
K8S_CONFORMANCE_USER_DEFINED_SKIP_REGEX = get_var(
    'K8S_CONFORMANCE_USER_DEFINED_SKIP_REGEX',
    ''
)
K8S_CONFORMANCE_CHECK_RESULTS = get_var_as_bool(
    "K8S_CONFORMANCE_CHECK_RESULTS", True)

K8S_CONFORMANCE_NON_BLOCKING_TAINTS = get_var_as_bool(
    "K8S_CONFORMANCE_NON_BLOCKING_TAINTS",
    False
)

K8S_CONFORMANCE_DISABLE_OFFLINE_LOGIC = get_var_as_bool(
    "K8S_CONFORMANCE_DISABLE_OFFLINE_LOGIC", False)

# FIO:
FIO_NAMESPACE = get_var("FIO_NAMESPACE", "fio")
FIO_NODE_LABEL = get_var("FIO_NODE_LABEL", "")
FIO_SERVER_COUNT = int(get_var("FIO_SERVER_COUNT", 2))
FIO_STORAGE_CLASS_NAME = get_var("FIO_STORAGE_CLASS_NAME", "kubernetes-ssd")
FIO_STORAGE_SIZE = get_var("FIO_STORAGE_SIZE", "7G")
FIO_LOOPS = int(get_var("FIO_LOOPS", 1))
FIO_WITH_WARMUP = get_var_as_bool("FIO_WITH_WARMUP", True)
FIO_JOBS_CONFIG_PATH = os.path.expanduser(get_var(
    'FIO_JOBS_CONFIG_PATH',
    pkg_resources.resource_filename(
        __name__,
        'tests/scale-performance/fio_scenarios/fio_jobs_with_warmup.yaml')))
FIO_CHART_URL = get_var('FIO_CHART_URL',
                        ('https://artifactory.mcp.mirantis.net/artifactory/'
                         'binary-dev-kaas-local/scale/helm/fio-0.1.0-mcp-2.tgz'))
FIO_TEST_TIMEOUT = int(get_var("FIO_TEST_TIMEOUT", 6000))
FIO_JOBS_TO_COMPARE = int(get_var("FIO_JOBS_TO_COMPARE", 10))
FIO_NIGHTLY_DEVIATION_PERCENT = int(get_var("FIO_NIGHTLY_DEVIATION_PERCENT", 10))
FIO_NIGHTLY_WORKFLOW_NAME = get_var("FIO_NIGHTLY_WORKFLOW_NAME", "")
FIO_JOB_SCENARIO_NAME = get_var("FIO_JOB_SCENARIO_NAME", "rand_write")

# Rally:
RALLY_DEPLOY_CLUSTER_NAME = get_var('RALLY_DEPLOY_CLUSTER_NAME', 'rally')
RALLY_DEPLOY_CLUSTER_NAMESPACE = get_var('RALLY_DEPLOY_CLUSTER_NAMESPACE', 'rally')
RALLY_NAMESPACE = get_var('RALLY_NAMESPACE', 'rally')
RALLY_ON_DEDICATED_CLUSTER = get_var_as_bool(
    'RALLY_ON_DEDICATED_CLUSTER', False)
RALLY_NODE_NAME = get_var('RALLY_NODE_NAME', None)
INTERNAL_K8S_ENDPOINT_ENABLED = get_var_as_bool(
    'INTERNAL_K8S_ENDPOINT_ENABLED', False)
RALLY_LABEL_KEY = get_var('RALLY_LABEL_KEY', 'rally')
RALLY_LABEL_VALUE = get_var('RALLY_LABEL_VALUE', 'runner')
RALLY_TOLERATION_KEY = get_var('RALLY_TOLERATION_KEY', 'rally')
RALLY_TOLERATION_VALUE = get_var('RALLY_TOLERATION_VALUE', 'runner')
RALLY_TOLERATION_EFFECT = get_var('RALLY_TOLERATION_EFFECT', 'NoSchedule')
RALLY_WAIT_TIMEOUT = int(get_var("RALLY_WAIT_TIMEOUT", 7200))
RALLY_GET_STEAL_TIME = get_var_as_bool('RALLY_GET_STEAL_TIME', False)
RALLY_DEBUG_MODE = get_var_as_bool('RALLY_DEBUG_MODE', False)
RALLY_PLATFORM = get_var('RALLY_PLATFORM', 'kubernetes')
RALLY_PARALLEL_TASKS = int(get_var("RALLY_PARALLEL_TASKS", 1))
RALLY_IMAGE = get_var(
    'RALLY_IMAGE',
    'cr.skyalex.net/rally/rally-openstack:latest'
)
RALLY_SCENARIOS_REPO = get_var(
    'RALLY_SCENARIOS_REPO',
    'https://github.com/Mirantis/scale-scenarios'
)
RALLY_SCENARIOS_BRANCH = get_var(
    'RALLY_SCENARIOS_BRANCH',
    'master'
)

RALLY_CONFIGMAP_NAME = get_var('RALLY_CONFIGMAP_NAME', 'rally')
RALLY_CONFIGMAP_YAML = os.path.expanduser(
    get_var(
        'RALLY_CONFIGMAP_YAML',
        pkg_resources.resource_filename(__name__, 'templates/pods/rally_test_configmap.yaml')
    )
)
RALLY_POD_NAME = get_var('RALLY_POD_NAME', 'rally')
RALLY_POD_YAML = os.path.expanduser(get_var(
    'RALLY_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/rally_test_pod.yaml')))
RALLY_POD_NGINX_IMAGE = get_full_image_path(
    get_var("RALLY_POD_NGINX_IMAGE", "general/external/docker.io/library/nginx:1.19.5"))

RALLY_TASK_ARGS = get_var('RALLY_TASK_ARGS', '')
RALLY_ENABLE_SCENARIOS = get_var('RALLY_ENABLE_SCENARIOS', '')
RALLY_SKIP_SCENARIOS = get_var('RALLY_SKIP_SCENARIOS', '')

RALLY_SAVE_TREND_REPORTS = get_var_as_bool("RALLY_SAVE_TREND_REPORTS", False)
RALLY_UPLOAD_SCALE_LAB_TRENDS = get_var_as_bool("RALLY_UPLOAD_SCALE_LAB_TRENDS", False)
RALLY_UPLOAD_SCALE_LAB_TRENDS_URL = get_var("RALLY_UPLOAD_SCALE_LAB_TRENDS_URL", '')
RALLY_ENV_TYPE = get_var('RALLY_ENV_TYPE', '')
RALLY_ENV_NAME = get_var('RALLY_ENV_NAME', '')

RALLY_JOBS_TO_COMPARE = int(get_var('RALLY_JOBS_TO_COMPARE', 10))
RALLY_NIGHTLY_DEVIATION_PERCENT = int(get_var('RALLY_NIGHTLY_DEVIATION_PERCENT', 10))
RALLY_NIGHTLY_WORKFLOW_NAME = get_var('RALLY_NIGHTLY_WORKFLOW_NAME', 'si-swarm-wf-aws-scale-performance-nightly')

OPENSTACK_PASSWORD_ROTATION_TIMEOUT = int(get_var("OPENSTACK_PASSWORD_ROTATION_TIMEOUT", 3600))
OPENSTACK_OVN_MIGRATION_TIMEOUT = int(get_var("OPENSTACK_OVN_MIGRATION_TIMEOUT", 7200))

# Density:
DENSITY_CHART_PER_NS_LIMIT = int(get_var('DENSITY_CHART_PER_NS_LIMIT', 800))
DENSITY_CONCURRENCY = int(get_var('DENSITY_CONCURRENCY', 10))
DENSITY_CHART_COUNT = int(get_var('DENSITY_CHART_COUNT', 0))
DENSITY_PREFIX = get_var('DENSITY_PREFIX', '')
DENSITY_NAMESPACE = get_var('DENSITY_NAMESPACE', 'density')
DENSITY_VALUES = get_var('DENSITY_VALUES', '')
DENSITY_WARMUP = get_var_as_bool("DENSITY_WARMUP", False)

# Longevity:
LONGEVITY_ENV_NAME = get_var('LONGEVITY_ENV_NAME', '')
LONGEVITY_METRICS = get_var('LONGEVITY_METRICS', '')
LONGEVITY_METRICS_STEP = int(get_var('LONGEVITY_METRICS_STEP', 60))
LONGEVITY_START_TIMESTAMP = int(get_var('LONGEVITY_START_TIMESTAMP', 0))
LONGEVITY_END_TIMESTAMP = int(get_var('LONGEVITY_END_TIMESTAMP', 0))
LONGEVITY_HOST = get_var('LONGEVITY_HOST', '')
LONGEVITY_SSH_LOGIN = get_var('LONGEVITY_SSH_LOGIN', 'prometheus')
LONGEVITY_SSH_PRIV_KEY_FILE = os.path.expanduser(
    get_var('LONGEVITY_SSH_PRIV_KEY_FILE', '~/.ssh/id_rsa'))

# RefApp:
REFAPP_NODE_LABEL = get_var('REFAPP_NODE_LABEL', 'refapp-db=enabled')
REFAPP_DB_SIZE = get_var('REFAPP_DB_SIZE', '450Mi')
REFAPP_DB_CHART_URL = get_var('REFAPP_DB_CHART_URL',
                              ('https://artifactory.mcp.mirantis.net/artifactory/binary-dev-kaas-local/openstack/helm/'
                               'openstack-helm-infra/mariadb-0.1.0-mcp-2787.tgz'))
REFAPP_CHART_URL = get_var('REFAPP_CHART_URL',
                           ('https://artifactory.mcp.mirantis.net/artifactory/'
                            'master-refapp-20220222170605/refapp-0.1.0-mcp-2.tgz'))
REFAPP_IMAGE = get_var('REFAPP_IMAGE', 'zhdkirill/refapp:0.1')
REFAPP_API_REPLICAS = get_var('REFAPP_API_REPLICAS', 3)

BYO_PROVIDER_NAME = "byo"
UCP_BYO_DOCKER_URL = get_var("UCP_BYO_DOCKER_URL", "")
UCP_BYO_CACERT = get_var("UCP_BYO_CACERT", "")
UCP_BYO_CLIENTCERT = get_var("UCP_BYO_CLIENTCERT", "")
UCP_BYO_CLIENTKEY = get_var("UCP_BYO_CLIENTKEY", "")
UCP_BYO_KUBECONFIG = get_var("UCP_BYO_KUBECONFIG", "")
UCP_BYO_CLOUD_PROVIDER = get_var("UCP_BYO_CLOUD_PROVIDER", "aws")

OPENSTACK_PROVIDER_E2E_POD_YAML = os.path.expanduser(get_var(
    'OPENSTACK_PROVIDER_E2E_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/openstack_provider_e2e_test_pod.yaml')))
OPENSTACK_PROVIDER_E2E_NAMESPACE = get_var(
    'OPENSTACK_PROVIDER_E2E_NAMESPACE', 'kaas-os-provider-e2e-test')
OPENSTACK_PROVIDER_E2E_POD_NAME = get_var(
    'OPENSTACK_PROVIDER_E2E_POD_NAME', 'kaas-os-provider-e2e')
OPENSTACK_PROVIDER_E2E_REPORTS_DIR = get_var(
    'OPENSTACK_PROVIDER_E2E_REPORTS_DIR', '/reports')
OPENSTACK_PROVIDER_E2E_IMAGE = get_var(
    'OPENSTACK_PROVIDER_E2E_IMAGE',
    'mirantis/kaas/cluster-api-provider-openstack-e2e'
)

# IrqBalance
IRQBALANCE_HOC_FILE_PATH = os.path.expanduser(get_var(
    "IRQBALANCE_HOC_FILE_PATH",
    pkg_resources.resource_filename(__name__, 'templates/extra-objects/irqbalance_hoc.yaml')))
IRQBALANCE_ENABLE_DAEMON = get_var_as_bool('IRQBALANCE_ENABLE_DAEMON', True)
IRQBALANCE_MODULE_VERSION = get_var('IRQBALANCE_MODULE_VERSION', '1.0.0')
IRQBALANCE_BANNED_CPULIST = get_var('IRQBALANCE_BANNED_CPULIST', '1,3,5')
IRQBALANCE_NODE_LABELS = list(filter(None, get_var(
    'IRQBALANCE_NODE_LABELS').split(',')))

# Tmeporary pin image
OPENSTACK_PROVIDER_E2E_IMAGE_VERSION = get_var(
    'OPENSTACK_PROVIDER_E2E_IMAGE_VERSION', '0.2.36'
)
TUNGSTEN_RESULT_POD_YAML = os.path.expanduser(get_var(
    'TUNGSTEN_RESULT_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/result_from_pv_pod.yaml')))

HPA_TEST_POD_YAML = os.path.expanduser(get_var(
    'HPA_TEST_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/hpa_pod.yaml')))

STACKLIGHT_TEST_POD_YAML = os.path.expanduser(get_var(
    'STACKLIGHT_TEST_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/sl_pod.yaml')))
STACKLIGHT_TEST_NAMESPACE = get_var('STACKLIGHT_TEST_NAMESPACE',
                                    'slns')
STACKLIGHT_TEST_POD_NAME = get_var('STACKLIGHT_TEST_POD_NAME',
                                   'test-stacklight')
STACKLIGHT_TEST_REPORTS_DIR = get_var('STACKLIGHT_TEST_REPORTS_DIR',
                                      '/report')
STACKLIGHT_TEST_DEST_REPORT = get_var('STACKLIGHT_TEST_DEST_REPORT',
                                      './stacklight_test_report.tar')
STACKLIGHT_TEST_POD_RESOURCES = get_var_from_json('STACKLIGHT_TEST_POD_RESOURCES')

STACKLIGHT_TEST_IMAGE = get_var(
    'STACKLIGHT_TEST_IMAGE',
    '')

STACKLIGHT_TEST_IMAGE_DICT = {
    "0.1.2-mcp-566": "0.1.2-mcp-563",
    "0.2.0-mcp-36": "0.1-xenial-20211110125339",
    "0.3.1": "0.1-xenial-20211228091411",
    "0.6.0": "0.1-xenial-20220505132833",
    "0.9.1": "0.1-xenial-20220930054418",
    "0.9.2": "0.1-xenial-20221027132100",
    "0.9.3": "0.1-xenial-20221027132100",
    "0.10.4": "0.1-xenial-20221219115850",
    "0.10.5": "0.1-xenial-20221219115850",
    "0.10.6": "0.1-xenial-20221219115850",
    "0.11.2": "0.11.1",
    "0.11.3": "0.1-20230712135236",
    "0.11.9": "0.1-20230712135236",
    "0.12.6": "0.1-20230620085836",
    "0.13.5": "0.1-20231101105608",
    "0.15.11": "0.1-20241004131846",
}

STACKLIGHT_ENABLE_LOGGING = get_var_as_bool("STACKLIGHT_ENABLE_LOGGING", False)
STACKLIGHT_ENABLE_HA = get_var_as_bool("STACKLIGHT_ENABLE_HA", False)
STACKLIGHT_ON_COMPUTE_NODES = get_var_as_bool("STACKLIGHT_ON_COMPUTE_NODES", False)

NODE_LABEL_SELECTOR = get_var("NODE_LABEL_SELECTOR", "")
NODE_TAINT_KEY = get_var("NODE_TAINT_KEY", "")
NODE_TAINT_VALUE = get_var("NODE_TAINT_VALUE", "")
NODE_TAINT_EFFECT = get_var("NODE_TAINT_EFFECT", "")

COMPONENT_TEST_RUN_ON_TARGET_CLUSTER = get_var_as_bool(
    "COMPONENT_TEST_RUN_ON_TARGET_CLUSTER", False)

TARGET_CLUSTER = get_var('TARGET_CLUSTER', 'si-kaas-management')
TARGET_NAMESPACE = get_var('TARGET_NAMESPACE', 'default')
NO_KAASCEPHCLUSTER = get_var_as_bool('NO_KAASCEPHCLUSTER', False)

# comma separated list of clusters to collect logs from
# format "namespace1/cluster1,namespace2/cluster2"
TARGET_CLUSTERS = get_var('TARGET_CLUSTERS', '').split(",")

# Must be used different for 2.3+ releases
TARGET_REGION = get_var('TARGET_REGION', 'region-one')
# force deleting child namespace after child cluster deletion
CHILD_NAMESPACE_FORCE_CLEANUP = get_var_as_bool(
    "KAAS_CHILD_NAMESPACE_FORCE_CLEANUP", False)
# Check resources are deleted after cluster deletion
CHECK_RESOURCES_DELETION = get_var_as_bool(
    "CHECK_RESOURCES_DELETION", False)

KUBECONFIG_PATH = get_var('KUBECONFIG', None)
SI_CONFIG_PATH = get_var('SI_CONFIG', None)

SI_SCENARIO = get_var('SI_SCENARIO', '')
SI_SCENARIOS_PATH = os.path.expanduser(get_var(
    "SI_SCENARIOS_PATH", pkg_resources.resource_filename(__name__, "templates/scenarios")))

# please use KUBECONFIG_MGMT_PATH only for openstack special case
# when KUBECONFIG_PATH don't contain management config
KUBECONFIG_MGMT_PATH = get_var('MGMT_KUBECONFIG', None)
# default path for KinD cluster kubeconfig
KUBECONFIG_KIND_PATH = get_var('KUBECONFIG_KIND_PATH',
                               '.kube/kind-config-clusterapi')

SI_BINARIES_DIR = get_var('SI_BINARIES_DIR', get_var("WORKSPACE", None))

KAAS_EXTERNAL_NETWORK_ID = get_var(
    'KAAS_EXTERNAL_NETWORK_ID', 'bf6b85a1-39db-4582-b0d1-f4291dddb9cf')
KAAS_NAMESPACE = get_var('KAAS_NAMESPACE', 'testspace1')
KAAS_CHILD_CLUSTER_NAME = get_var('KAAS_CHILD_CLUSTER_NAME', 'testcluster1')
KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE = os.path.expanduser(
    get_var('KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE', '~/.ssh/id_rsa.pub'))
KAAS_CHILD_CLUSTER_SSH_LOGIN = get_var('KAAS_CHILD_CLUSTER_SSH_LOGIN', 'mcc-user')
KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE = os.path.expanduser(
    get_var('KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE', '~/.ssh/id_rsa'))
KAAS_CHILD_CLUSTER_RELEASE_NAME = get_var('KAAS_CHILD_CLUSTER_RELEASE_NAME', '')
KAAS_CHILD_CLUSTER_UPDATE_RELEASE_NAME = get_var('KAAS_CHILD_CLUSTER_UPDATE_RELEASE_NAME', '')
# Update plan 'name' or 'id' of a step which should be latest in the current upgrade session
KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME = get_var('KAAS_CHILD_CLUSTER_UPDATE_PLAN_TILL_STEP_NAME', '')
KAAS_CHILD_CLUSTER_UPDATE_CONTINUATION = get_var_as_bool('KAAS_CHILD_CLUSTER_UPDATE_CONTINUATION', False)
KAAS_CHILD_CLUSTER_RELEASE_MUST_NOT_BE_SAME = get_var_as_bool('KAAS_CHILD_CLUSTER_RELEASE_MUST_NOT_BE_SAME', False)
KAAS_CHILD_CLUSTER_DNS = get_var('KAAS_CHILD_CLUSTER_DNS', '172.18.176.6,172.19.80.70')
KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME = get_var('KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME',
                                                'bionic-server-cloudimg-amd64-20200724')
KAAS_CHILD_CLUSTER_MACHINE_FLAVOR_NAME = get_var('KAAS_CHILD_CLUSTER_MACHINE_FLAVOR_NAME', 'kaas.minimal')
KAAS_CHILD_CLUSTER_MACHINE_AWS_DISK_TYPE = get_var('KAAS_CHILD_CLUSTER_MACHINE_AWS_DISK_TYPE', 'gp3')
KAAS_CHILD_CLUSTER_CEPH_MACHINE_FLAVOR_NAME = get_var('KAAS_CHILD_CLUSTER_CEPH_MACHINE_FLAVOR_NAME', '')
KAAS_CHILD_CLUSTER_CEPH_AWS_DISK_TYPE = get_var('KAAS_CHILD_CLUSTER_CEPH_AWS_DISK_TYPE', 'gp3')
KAAS_CHILD_CLUSTER_CEPH_AWS_DISK_SIZE = int(get_var('KAAS_CHILD_CLUSTER_CEPH_AWS_DISK_SIZE', 120))
KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_FLAVOR_NAME = get_var('KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_FLAVOR_NAME', '')
KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_DISK_TYPE = get_var('KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_DISK_TYPE', 'gp3')
KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_DISK_SIZE = int(get_var('KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_DISK_SIZE', 120))
KAAS_CHILD_CLUSTER_CONTROL_MACHINE_FLAVOR_NAME = get_var('KAAS_CHILD_CLUSTER_CONTROL_MACHINE_FLAVOR_NAME', '')
KAAS_CHILD_CLUSTER_CONTROL_MACHINE_AWS_DISK_TYPE = get_var('KAAS_CHILD_CLUSTER_CONTROL_MACHINE_AWS_DISK_TYPE', 'gp3')
KAAS_CHILD_CLUSTER_CONTROL_MACHINE_AWS_DISK_SIZE = int(get_var('KAAS_CHILD_CLUSTER_CONTROL_MACHINE_AWS_DISK_SIZE', 120))
KAAS_CHILD_CLUSTER_MASTER_FLAVOR_NAME = get_var('KAAS_CHILD_CLUSTER_MASTER_FLAVOR_NAME', '')
KAAS_CHILD_CLUSTER_MASTER_AWS_DISK_TYPE = get_var('KAAS_CHILD_CLUSTER_MASTER_AWS_DISK_TYPE', 'gp3')
KAAS_CHILD_CLUSTER_MASTER_AWS_DISK_SIZE = int(get_var('KAAS_CHILD_CLUSTER_MASTER_AWS_DISK_SIZE', 120))
KAAS_CHILD_CLUSTER_SLAVE_SL_FLAVOR_NAME = get_var('KAAS_CHILD_CLUSTER_SLAVE_SL_FLAVOR_NAME', '')
KAAS_CHILD_CLUSTER_SLAVE_SL_AWS_DISK_TYPE = get_var('KAAS_CHILD_CLUSTER_SLAVE_SL_AWS_DISK_TYPE', 'gp3')
KAAS_MANAGEMENT_CLUSTER_DNS1 = get_var('KAAS_MANAGEMENT_CLUSTER_DNS1', '172.18.176.6')
KAAS_MANAGEMENT_CLUSTER_DNS2 = get_var('KAAS_MANAGEMENT_CLUSTER_DNS2', '172.19.80.70')
KAAS_CHILD_CLUSTER_AZ_NAME = get_var('KAAS_CHILD_CLUSTER_AZ_NAME', 'nova')
KAAS_MANAGEMENT_CLUSTER_NODES = get_var('KAAS_MANAGEMENT_CLUSTER_NODES', '3')
KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT = get_var('KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT', '3')
WORKER_COUNT = get_var('WORKER_COUNT', '3')
KAAS_CHILD_CLUSTER_CEPH_NODES_COUNT = get_var('KAAS_CHILD_CLUSTER_CEPH_NODES_COUNT', '3')
KAAS_CHILD_CLUSTER_COMPUTE_NODES_COUNT = get_var('KAAS_CHILD_CLUSTER_COMPUTE_NODES_COUNT', '2')
KAAS_CHILD_CLUSTER_CONTROL_NODES_COUNT = get_var('KAAS_CHILD_CLUSTER_CONTROL_NODES_COUNT', '3')
KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT = get_var('KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT', '3')
KAAS_CHILD_CLUSTER_SLAVE_SL_NODES_COUNT = get_var('KAAS_CHILD_CLUSTER_SLAVE_SL_NODES_COUNT', '0')
KAAS_CHILD_WORKER_PREPARE_COUNT = int(get_var('KAAS_CHILD_WORKER_PREPARE_COUNT', 2))
KAAS_CHILD_WORKER_UPGRADE_COUNT = int(get_var('KAAS_CHILD_WORKER_UPGRADE_COUNT', 2))
PARALLEL_UPGRADE_ENABLED = get_var_as_bool('PARALLEL_UPGRADE_ENABLED', False)
KAAS_CHILD_CLUSTER_SET_CUSTOM_INDEXES = get_var_as_bool("KAAS_CHILD_CLUSTER_SET_CUSTOM_INDEXES", False)
MACHINES_TIMESTAMPS_YAML_FILE = get_var('MACHINES_TIMESTAMPS_YAML_FILE', '')
KAAS_OS_CHILD_DEDICATED_CONTROLPLANE = get_var_as_bool("KAAS_OS_CHILD_DEDICATED_CONTROLPLANE", True)
CHECK_MACHINE_STATUS_TIMEOUT = int(get_var('CHECK_MACHINE_STATUS_TIMEOUT', 6300))
CHECK_LCMMACHINE_DAY2_STATEITEMS_TIMEOUT = int(get_var('CHECK_LCMMACHINE_DAY2_STATEITEMS_TIMEOUT', 900))
# Semicolon-separated list of comma-separated lists of kaas cluster releases
KAAS_CHILD_CLUSTER_UPGRADE_SEQUENCE = get_var('KAAS_CHILD_CLUSTER_UPGRADE_SEQUENCE', '')

# Directory where yaml files with predefined upgrade paths are placed
UPGRADE_PATHS_YAML_FILE_DIR = os.path.expanduser(get_var(
    "UPGRADE_PATHS_YAML_FILE_DIR",
    pkg_resources.resource_filename(__name__, 'templates/release-upgrade-paths/')))
# Use upgrade path from yaml or not.
# If True then UPGRADE_PATHS_YAML_FILE_DIR will be used for searching file with upgrade paths
USE_UPGRADE_PATH_FROM_YAML = get_var_as_bool('USE_UPGRADE_PATH_FROM_YAML', False)
# We can predefine upgrade path for upgrade test
KAAS_CHILD_CLUSTER_UPGRADE_PATH = list(filter(None, get_var(
    'KAAS_CHILD_CLUSTER_UPGRADE_PATH').split(',')))
# Info collected from child cluster before update, which can be re-used to complete ClusterUpdatePlan steps
KAAS_CHILD_CLUSTER_UPDATE_INFO_PATH = get_var('KAAS_CHILD_CLUSTER_UPDATE_INFO_PATH', '')

# Comma-separated lists of extra helm releases for the child cluster
KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES = list(filter(None, get_var(
    'KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES',
    '').split(',')))

KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES_JSON = get_var_from_json("KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES_JSON")
# Deploy stacklight LMA helm release on the top of child cluster
KAAS_CHILD_CLUSTER_DEPLOY_LMA = get_var_as_bool(
    "KAAS_CHILD_CLUSTER_DEPLOY_LMA", True
)
# Dict with LMA settings for child cluster
KAAS_CHILD_LMA_EXTRA_OPTIONS = get_var(
    'KAAS_CHILD_LMA_EXTRA_OPTIONS', '')
KAAS_CHILD_CLUSTER_MACHINES_PREFIX = get_var(
    'KAAS_CHILD_CLUSTER_MACHINES_PREFIX', None)

# Managed by BM team, bumps machines distribution to latest allowed, used in kaas-bm-cr-core-upgrade-upgradetest
KAAS_CHILD_CLUSTER_DISTRO_UPGRADE = get_var_as_bool('KAAS_CHILD_CLUSTER_DISTRO_UPGRADE', False)

# Managed by BM team, allows to update to Noble distribution in some BM CI scenarios
KAAS_BM_CI_ENABLE_NOBLE_DISTRIBUTION = get_var_as_bool('KAAS_BM_CI_ENABLE_NOBLE_DISTRIBUTION', False)

KAAS_REGIONAL_CLUSTER_SSH_LOGIN = get_var('KAAS_REGIONAL_CLUSTER_SSH_LOGIN',
                                          'mcc-user')
KAAS_REGIONAL_CLUSTER_PRIVATE_KEY_FILE = os.path.expanduser(
    get_var('KAAS_REGIONAL_CLUSTER_PRIVATE_KEY_FILE', '~/.ssh/id_rsa'))

EXPECTED_PODS_TEMPLATES_DIR = get_var(
    "EXPECTED_PODS_TEMPLATES_DIR",
    pkg_resources.resource_filename(__name__,
                                    'templates/expected-pods/'))
EXPECTED_ROLEBINDINGS_TEMPLATES_DIR = get_var(
    "EXPECTED_ROLEBINDINGS_TEMPLATES_DIR",
    pkg_resources.resource_filename(__name__,
                                    'templates/expected-rolebindings/'))
EXPECTED_KERNEL_VERSIONS_DIR = get_var(
    "EXPECTED_KERNEL_VERSIONS_DIR",
    pkg_resources.resource_filename(__name__,
                                    'templates/expected-kernel-versions/'))
SKIP_EXPECTED_KERNEL_VERSIONS_CHECK = get_var_as_bool('SKIP_EXPECTED_KERNEL_VERSIONS_CHECK', False)
SKIP_EXPECTED_DISTRIB_CHECK = get_var_as_bool('SKIP_EXPECTED_DISTRIB_CHECK', False)
SKIP_CHECK_ACTUAL_EXPECTED_DOCKER_SERVICES_CHECK = get_var_as_bool('SKIP_CHECK_ACTUAL_EXPECTED_DOCKER_SERVICES_CHECK',
                                                                   False)
SKIP_CHECK_CHECK_PERSISTENT_VOLUMES_MOUNTS = get_var_as_bool('SKIP_CHECK_CHECK_PERSISTENT_VOLUMES_MOUNTS', False)
UPDATE_LICENSE = get_var_as_bool("UPDATE_LICENSE", True)

DUMMY_TEST_POD_YAML = os.path.expanduser(get_var(
    'DUMMY_TEST_POD_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/dummy_pod.yaml')))

HA_TEST_PRIVATE_KEY_FILE = os.path.expanduser(
    get_var('HA_TEST_PRIVATE_KEY_FILE', '~/.ssh/id_rsa'))

HA_TEST_NAMESPACE_LIST = list(filter(None, get_var(
    'HA_TEST_NAMESPACE_LIST').split(',')))

HA_NODE_REBOOT_LABELS = list(filter(None, get_var(
    'HA_NODE_REBOOT_LABELS').split(',')))

HA_SVC_TIMEOUT = int(get_var("HA_SVC_TIMEOUT", 180))

HA_SL_SVC_TIMEOUT = [("iam-proxy-alerta", "stacklight", 20),
                     ("iam-proxy-alertmanager", "stacklight", 20),
                     ("iam-proxy-grafana", "stacklight", 20),
                     ("iam-proxy-kibana", "stacklight", 20),
                     ("iam-proxy-prometheus", "stacklight", 20)]

HA_SL_SVC_HA_MODE_DOWNTIME = int(get_var("HA_SL_SVC_HA_MODE_DOWNTIME", 240))
HA_FILL_DISK_PERCENT = int(get_var("HA_FILL_DISK_PERCENT", 95))

# etcd custom options
# ETCD_STORAGE_QUOTA examples: 2GB, 8GB, 500MB
ETCD_STORAGE_QOUTA = get_var('ETCD_STORAGE_QOUTA', '')

# openvswitchd pattern name for checking reboot policy
OPENVSWITCHD_PATTERN_NAME = get_var('OPENVSWITCHD_PATTERN_NAME',
                                    'openvswitch-openvswitch-vswitchd-default')
NEUTRON_L3_PATTERN_NAME = get_var('NEUTRON_L3_PATTERN_NAME',
                                  'neutron-l3-agent-default')
# openvswitch-ovn-controller pattern name
OVN_CONTROLLER_PATTERN_NAME = get_var('OVN_CONTROLLER_PATTERN_NAME', 'openvswitch-ovn-controller')

# Nodes info yaml file with access information to kubernetes nodes in the
# following format:
# ip:
#   address: <ip address>
# ssh:
#   username: <user name>
#   private_key: |
#       <private key>
NODES_INFO = get_var('NODES_INFO')

# Means cloned locally
# https://gerrit.mcp.mirantis.com/admin/repos/kaas/releases
KAAS_RELEASES_FOLDER = get_var('KAAS_RELEASES_FOLDER')
KAAS_RELEASES_REMOTE_FOLDER = get_var(
    'KAAS_RELEASES_REMOTE_FOLDER', 'kaas_releases')
KAAS_RELEASES_GENERATED_FOLDER = get_var(
    'KAAS_RELEASES_GENERATED_FOLDER', 'releases')
KAAS_BOOTSTRAP_TARGET_DIR = get_var('KAAS_BOOTSTRAP_TARGET_DIR',
                                    'kaas-bootstrap')
KAAS_GET_SCRIPT_NAME = get_var('KAAS_GET_SCRIPT_NAME', 'get_kaas.sh')
# KAAS_RELEASE_FILE can have the following values
# - a name of existing file of kaas release - this file will be used
# - empty (None) - latest available file will be used (in most cases this
#   will be release candidate)
# - 'stable' - latest available stable file will be used
KAAS_RELEASE_FILE = get_var('KAAS_RELEASE_FILE')

# We can override chart versions before mgmt deploy
# Use next format '{"release-controller": "1.41.0", "admission-controller": "1.40.38"}' etc
KAAS_RELEASE_CHARTS_VERSIONS_OVERRIDES = get_var('KAAS_RELEASE_CHARTS_VERSIONS_OVERRIDES', '')

KAAS_CHILD_CLUSTER_UPGRADE_RUN_CONFORMANCE = get_var_as_bool(
    "KAAS_CHILD_CLUSTER_UPGRADE_RUN_CONFORMANCE", True)

KAAS_CHILD_CLUSTER_DELETE_TIMEOUT = int(get_var("KAAS_CHILD_CLUSTER_DELETE_TIMEOUT",
                                                30 * 60))
KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT = int(get_var("KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT",
                                                "18000"))

KAAS_ROLLOUT_ATTEMPTS = get_var('KAAS_ROLLOUT_ATTEMPTS', '120')

# OpenStack Controller
OPENSTACK_PROVIDER_NAME = "openstack"
OSH_OPERATOR_PATH = get_var('OSH_OPERATOR_PATH', None)
OSH_OPERATOR_UPDATE_TAG = get_var('OSH_OPERATOR_UPDATE_TAG', None)
OSH_OPERATOR_UPDATE_REPO = get_var(
    'OSH_OPERATOR_UPDATE_REPO',
    'docker-dev-kaas-local.docker.mirantis.net/mcp/openstack-controller')
OSH_OPERATOR_CHART_URL = get_var('OSH_OPERATOR_CHART_URL', None)
OSH_OPERATOR_HB_NAME = get_var('OSH_OPERATOR_HB_NAME', 'openstack-operator')
OSH_NAMESPACE = get_var('OSH_NAMESPACE', 'openstack')
OSH_DEPLOYMENT_NAME = get_var('OSH_DEPLOYMENT_NAME', 'osh-dev')
OPENSTACK_BOOT_FROM_VOLUME = get_var_as_bool('OPENSTACK_BOOT_FROM_VOLUME', False)
OPENSTACK_BOOT_VOLUME_SIZE = get_var('OPENSTACK_BOOT_VOLUME_SIZE', 120)
OPENSTACK_UPGRADE_VERSION = get_var('OPENSTACK_UPGRADE_VERSION', 'stein')
OPENSTACK_UPGRADE_TIMEOUT = int(get_var(
    'OPENSTACK_UPGRADE_TIMEOUT', 6000))
# often lcm operations include images precaching,
# setting timeout to 1 h by default
OPENSTACK_LCM_OPERATIONS_TIMEOUT = int(get_var(
    'OPENSTACK_LCM_OPERATIONS_TIMEOUT', 3600))

CHECK_CLUSTER_READINESS_TIMEOUT = int(get_var(
    'CHECK_CLUSTER_READINESS_TIMEOUT', 3600))

OPENSTACK_K8S_OBJECT_REPLICAS = int(get_var(
    "OPENSTACK_K8S_OBJECT_REPLICAS", 3))
OPENSTACK_K8S_OBJECT_NUM_DELETE_PODS = int(get_var(
    "OPENSTACK_K8S_OBJECT_NUM_DELETE_PODS", 1))

SKIP_SECRET_SECTIONS_LIST = list(filter(None, get_var(
    'SKIP_SECRET_SECTIONS_LIST',
    'ironic').split(',')))
SKIP_SECRET_KEYS_LIST = list(filter(None, get_var(
    'SKIP_SECRET_KEYS_LIST',
    'auth_type').split(',')))
SKIP_CONFIGMAP_LIST = list(filter(None, get_var(
    'SKIP_CONFIGMAP_LIST',
    'ingress-openstack,'
    'ingress-openstack-nginx,'
    'ingress-openstack-openstack-ingress-nginx,'
    'openstack-mariadb-openstack-mariadb-mariadb-ingress,'
    'openstack-mariadb-mariadb-state').split(',')))
SKIP_SECRET_LIST = list(filter(None, get_var(
    'SKIP_SECRET_LIST',
    'keystone-fernet-data').split(',')))
TEMPEST_CUSTOM_FLAVOR = get_var('TEMPEST_CUSTOM_FLAVOR', '')
TEMPEST_CUSTOM_IMAGE = get_var('TEMPEST_CUSTOM_IMAGE', '')
TEMPEST_CUSTOM_IMAGE_ALT = get_var('TEMPEST_CUSTOM_IMAGE_ALT', '')
TEMPEST_CUSTOM_PUBLIC_NET = get_var('TEMPEST_CUSTOM_PUBLIC_NET', '')
TEMPEST_CUSTOM_PARAMETERS = get_var('TEMPEST_CUSTOM_PARAMETERS', '')
IRONIC_DEPLOY_IMAGE_URL = get_var('IRONIC_DEPLOY_IMAGE_URL', '')
IRONIC_IMAGE_NAME = get_var('IRONIC_IMAGE_NAME', 'ubuntu-manual')
IRONIC_FLAVOR_NAME = get_var('IRONIC_FLAVOR_NAME', 'baremetal-ram64000-cpus16-disk500')

STEPLER_DOCKER_IMAGE = get_var('STEPLER_DOCKER_IMAGE', None)
STEPLER_EXTRA_CONFIG_OPTIONS = get_var('STEPLER_EXTRA_CONFIG_OPTIONS', None)

MCP_PIPELINE_PATH = get_var('MCP_PIPELINE_PATH', None)
KAAS_BOOTSTRAP_LICENSE_FILE = get_var(
    'KAAS_BOOTSTRAP_LICENSE_FILE', None)
KAAS_BOOTSTRAP_LICENSE_REMOTE_FILE = get_var(
    'KAAS_BOOTSTRAP_LICENSE_REMOTE_FILE', 'mirantis.lic')

KAAS_MANAGEMENT_CLUSTER_IMAGE = get_var(
    'KAAS_MANAGEMENT_CLUSTER_IMAGE', 'bionic-server-cloudimg-amd64-20200724')

# TungstenFabric Operator
TF_OPERATOR_CR_NAME = get_var('TF_OPERATOR_CR_NAME', 'openstack-tf')
TF_OPERATOR_API_V2 = get_var_as_bool('TF_OPERATOR_API_V2', False)
TF_CASSANDRA_NODES_CLENAUP = get_var_as_bool('TF_CASSANDRA_NODES_CLENAUP', False)

# Baremetal specific
BAREMETAL_PROVIDER_NAME = "baremetal"
KAAS_BM_ENABLED = get_var_as_bool("KAAS_BM_ENABLED", False)
BM_DEFAULT_NS = get_var(
    'BM_DEFAULT_NS', 'kaas')
BM_METALLB_NS = get_var(
    'BM_METALLB_NS', 'metallb-system')
BM_TEST_NS = get_var(
    'BM_TEST_NS', 'bm-test-ns')
BM_SEC_TEMPLATE = get_var(
    'BM_SEC_TEMPLATE', 'si-tests/si_tests/templates/kaas-bm/bm-secret.yaml')
BM_BMH_TEMPLATE = get_var(
    'BM_BMH_TEMPLATE', None)
BM_DEPLOY_CHILD = get_var_as_bool("DEPLOY_CHILD_CLUSTER", False)
SI_BM_UPGRADE_CHILD = get_var_as_bool("SI_BM_UPGRADE_CHILD", False)
BM_CHILD_SETTINGS_YAML_PATH = os.path.expanduser(get_var(
    "BM_CHILD_SETTINGS_YAML_PATH", pkg_resources.resource_filename(
        __name__,
        'kaas-bm-env/labs/{}/child/child_data/main.yaml.j2'.format(
            ENV_CONFIG_NAME or ENV_NAME))))
BM_DEPLOY_REGIONAL = get_var_as_bool("DEPLOY_REGIONAL_CLUSTER", False)
BM_DEPLOY_REGIONAL_SPLITTED_SEED = get_var_as_bool("BM_DEPLOY_REGIONAL_SPLITTED_SEED", False)

BM_CHECK_BMC_STATUS_AFTER_DELETE = get_var_as_bool(
    "BM_CHECK_BMC_STATUS_AFTER_DELETE", False
)
# Switch local env VMs(non product one) to internal mirrors
USE_MIRA_APT_MIRRORS = get_var_as_bool('USE_MIRA_APT_MIRRORS', False)
# Use half-virtual lab, i.e. no nested virtualization only libvirt VMs
BM_HALF_VIRTUAL_ENV = get_var_as_bool("DEPLOY_HALF_VIRTUAL_LAB", False)
BM_HALF_VIRTUAL_NODE = get_var("HALF_VIRTUAL_NODE", None)
BM_HALF_VIRTUAL_NODE_PREFIX = "kaas-bm-team-"

USE_REDUCE_ENV = get_var("USE_REDUCE_ENV", False)

# RHEL License credential
KAAS_CI_RHEL_LICENSES_USERNAME = get_var(
    'KAAS_CI_RHEL_LICENSES_USERNAME', None)
KAAS_CI_RHEL_LICENSES_PASSWORD = get_var(
    'KAAS_CI_RHEL_LICENSES_PASSWORD', None)
KAAS_CI_RHEL_LICENSES_ACTIVATION_KEY = get_var(
    'KAAS_CI_RHEL_LICENSES_ACTIVATION_KEY', None)
KAAS_CI_RHEL_LICENSES_ACTIVATION_ORG = get_var(
    'KAAS_CI_RHEL_LICENSES_ACTIVATION_ORG', None)
KAAS_CI_RHEL_LICENSES_IS_ACTIVATION_KEY = get_var_as_bool(
    'KAAS_CI_RHEL_LICENSES_IS_ACTIVATION_KEY', False)
KAAS_CI_RHEL_LICENSES_RPM_URL = get_var(
    'KAAS_CI_RHEL_LICENSES_RPM_URL', None)

# Vsphere specific
VSPHERE_PROVIDER_NAME = "vsphere"
KAAS_VSPHERE_ENABLED = get_var_as_bool(
    'KAAS_VSPHERE_ENABLED', False)
KAAS_CHILD_VSPHERE_LB_HOST = get_var(
    'KAAS_CHILD_VSPHERE_LB_HOST', None)
KAAS_CHILD_VSPHERE_METALLB_RANGE = get_var(
    'KAAS_CHILD_VSPHERE_METALLB_RANGE', None)
KAAS_VSPHERE_LB_HOST = get_var(
    'KAAS_VSPHERE_LB_HOST', None)
KAAS_VSPHERE_METALLB_RANGE = get_var(
    'KAAS_VSPHERE_METALLB_RANGE', None)
KAAS_VSPHERE_CAPI_PROVIDER_USERNAME = get_var(
    "KAAS_VSPHERE_CAPI_PROVIDER_USERNAME", None)
KAAS_VSPHERE_CAPI_PROVIDER_PASSWORD = get_var(
    "KAAS_VSPHERE_CAPI_PROVIDER_PASSWORD", None)
KAAS_VSPHERE_CLOUD_PROVIDER_USERNAME = get_var(
    "KAAS_VSPHERE_CLOUD_PROVIDER_USERNAME", None)
KAAS_VSPHERE_CLOUD_PROVIDER_PASSWORD = get_var(
    "KAAS_VSPHERE_CLOUD_PROVIDER_PASSWORD", None)
KAAS_VSPHERE_DATACENTER_NAME = get_var("KAAS_VSPHERE_DATACENTER_NAME", None)
KAAS_VSPHERE_DATACENTER_PATH = get_var("KAAS_VSPHERE_DATACENTER_PATH", None)
KAAS_VSPHERE_DATASTORECLUSTER_PATH = get_var(
    "KAAS_VSPHERE_DATASTORECLUSTER_PATH", None)
KAAS_VSPHERE_DATASTORE_NAME = get_var("KAAS_VSPHERE_DATASTORE_NAME", None)
KAAS_VSPHERE_NETWORK_PATH = get_var("KAAS_VSPHERE_NETWORK_PATH", None)
KAAS_VSPHERE_RESOURCE_POOL_PATH = get_var(
    "KAAS_VSPHERE_RESOURCE_POOL_PATH", None)
KAAS_VSPHERE_MACHINES_FOLDER = get_var("KAAS_VSPHERE_MACHINES_FOLDER", None)
KAAS_VSPHERE_SCSI_CONTROLLER_TYPE = get_var(
    "KAAS_VSPHERE_SCSI_CONTROLLER_TYPE", "pvscsi")
KAAS_VSPHERE_SERVER_INSECURE = get_var("KAAS_VSPHERE_SERVER_INSECURE", True)
KAAS_VSPHERE_RHEL8_TEMPLATE_PATH = get_var(
    'KAAS_VSPHERE_RHEL8_TEMPLATE_PATH', None)
KAAS_VSPHERE_UBUNTU_TEMPLATE_PATH = get_var(
    'KAAS_VSPHERE_UBUNTU_TEMPLATE_PATH', None)
KAAS_VSPHERE_IPAM_ENABLED = get_var_as_bool('KAAS_VSPHERE_IPAM_ENABLED', False)
KAAS_VSPHERE_NETWORK_CIDR = get_var(
    'KAAS_VSPHERE_NETWORK_CIDR', None)
KAAS_VSPHERE_CIDR_INCLUDE_RANGES = get_var(
    'KAAS_VSPHERE_CIDR_INCLUDE_RANGES', None)
KAAS_VSPHERE_CIDR_EXCLUDE_RANGES = get_var(
    'KAAS_VSPHERE_CIDR_EXCLUDE_RANGES', None)
KAAS_VSPHERE_NETWORK_GATEWAY = get_var(
    'KAAS_VSPHERE_NETWORK_GATEWAY', None)
KAAS_VSPHERE_NETWORK_NAMESERVERS = get_var(
    'KAAS_VSPHERE_NETWORK_NAMESERVERS', None)
KAAS_CHILD_VSPHERE_NETWORK_CIDR = get_var(
    'KAAS_CHILD_VSPHERE_NETWORK_CIDR', None)
KAAS_CHILD_VSPHERE_NETWORK_INCLUDE_RANGES = get_var(
    'KAAS_CHILD_VSPHERE_NETWORK_INCLUDE_RANGES', None)
KAAS_CHILD_VSPHERE_CIDR_EXCLUDE_RANGES = get_var(
    'KAAS_VSPHERE_CIDR_EXCLUDE_RANGES', None)
KAAS_CHILD_VSPHERE_NETWORK_GATEWAY = get_var(
    'KAAS_CHILD_VSPHERE_NETWORK_GATEWAY', None)
KAAS_CHILD_VSPHERE_NETWORK_NAMESERVERS = get_var(
    'KAAS_CHILD_VSPHERE_NETWORK_NAMESERVERS', None)
KAAS_VSPHERE_RO_USER = get_var('KAAS_VSPHERE_RO_USER', None)
KAAS_VSPHERE_RO_PASSWORD = get_var('KAAS_VSPHERE_RO_PASSWORD', None)
KAAS_VSPHERE_TEMPLATE_OS_NAME = get_var('KAAS_VSPHERE_TEMPLATE_OS_NAME', None)
KAAS_VSPHERE_TEMPLATE_OS_VERSION = get_var(
    'KAAS_VSPHERE_TEMPLATE_OS_VERSION', None)
KAAS_VSPHERE_TEMPLATE_ISO_URL = get_var('KAAS_VSPHERE_TEMPLATE_ISO_URL', None)
VSPHERE_USE_VVMT_OBJECTS = get_var_as_bool('VSPHERE_USE_VVMT_OBJECTS', False)
KAAS_VSPHERE_VVMT_CREATION_TIMEOUT = int(get_var("KAAS_VSPHERE_VVMT_CREATION_TIMEOUT", 2700))  # 45m
KAAS_VSPHERE_MACHINE_CPU = int(get_var("KAAS_VSPHERE_MACHINE_CPU", 8))
KAAS_VSPHERE_MACHINE_RAM = int(get_var("KAAS_VSPHERE_MACHINE_RAM", 16384))

# Equinix Metal specific
EQUINIX_PROVIDER_NAME = "equinixmetal"
EQUINIXMETALV2_PROVIDER_NAME = "equinixmetalv2"
KAAS_EQUINIX_ENABLED = get_var_as_bool("KAAS_EQUINIX_ENABLED", False)
KAAS_EQUINIXMETALV2_ENABLED = get_var_as_bool("KAAS_EQUINIXMETALV2_ENABLED", False)
KAAS_EQUINIX_PROJECT_ID = get_var("KAAS_EQUINIX_PROJECT_ID", None)
KAAS_EQUINIX_OS = get_var("KAAS_EQUINIX_OS", 'ubuntu_20_04')
KAAS_EQUINIX_BILLING_CYCLE = get_var("KAAS_EQUINIX_BILLING_CYCLE", 'hourly')
KAAS_EQUINIX_API_TOKEN = get_var("KAAS_EQUINIX_API_TOKEN", None)
KAAS_EQUINIX_USER_API_TOKEN = get_var("KAAS_EQUINIX_USER_API_TOKEN", None)
KAAS_EQUINIX_FACILITY_REGION = get_var("KAAS_EQUINIX_FACILITY_REGION", "AM6")
KAAS_EQUINIX_CEPH_MANUAL_CONFIGURATION = get_var_as_bool("KAAS_EQUINIX_CEPH_MANUAL_CONFIGURATION", False)
KAAS_EQUINIX_CHILD_PUBLIC_KEY_NAME = get_var("KAAS_EQUINIX_CHILD_PUBLIC_KEY_NAME", None)
KAAS_EQUINIX_CHILD_CREDENTIAL_NAME = get_var("KAAS_EQUINIX_CHILD_CREDENTIAL_NAME", None)
KAAS_EQUINIX_CHILD_DEDICATED_CONTROLPLANE = get_var_as_bool("KAAS_EQUINIX_CHILD_DEDICATED_CONTROLPLANE", False)
KAAS_EQUINIX_CHILD_METRO = get_var("KAAS_EQUINIX_CHILD_METRO", "sv")
# Equinix private setup specific
# path to the terraform original file with configuration for private MCC installation
KAAS_EQUINIX_PRIVATE_NETWORK_CONFIG = get_var("KAAS_EQUINIX_PRIVATE_NETWORK_CONFIG", None)
# sane-default parameters for Equinix MCC bootstrap in private mode
KAAS_EQUINIX_PRIVATE_PXE_BRIDGE = get_var("KAAS_EQUINIX_PRIVATE_PXE_BRIDGE", "br0")
# port of squid proxy, based at the edge router to allow internet access for mgmt/regionals
KAAS_EQUINIX_PRIVATE_PROXY_PORT = get_var("KAAS_EQUINIX_PRIVATE_PROXY_PORT", 3128)
# Machine types for equinixmetal/equinixmetalv2 providers
KAAS_EQUINIX_MACHINE_TYPE = get_var("KAAS_EQUINIX_MACHINE_TYPE", None)
_machine_types = get_var_from_json("KAAS_EQUINIX_CHILD_MACHINE_TYPES")
# MKE 'master' nodes
KAAS_EQUINIX_CHILD_MACHINE_TYPE_MASTER = _machine_types.get("master", KAAS_EQUINIX_MACHINE_TYPE)
# MKE 'worker' nodes
KAAS_EQUINIX_CHILD_MACHINE_TYPE_WORKER = _machine_types.get("worker", KAAS_EQUINIX_MACHINE_TYPE)
# MKE 'worker' nodes for Stacklight
KAAS_EQUINIX_CHILD_MACHINE_TYPE_STACKLIGHT = _machine_types.get("stacklight", KAAS_EQUINIX_MACHINE_TYPE)
# MKE 'worker' nodes for Openstack controllers
KAAS_EQUINIX_CHILD_MACHINE_TYPE_OS_CONTROL = _machine_types.get("os_control", KAAS_EQUINIX_MACHINE_TYPE)
# Comma-separated SSH key names that configured in the Equinixmetal for the current project
# To access equinixmetal instances with serial console
KAAS_EQUINIX_PROJECT_SSH_KEYS = get_var("KAAS_EQUINIX_PROJECT_SSH_KEYS", '')

# Azure specific
AZURE_PROVIDER_NAME = "azure"
KAAS_AZURE_ENABLED = get_var_as_bool("KAAS_AZURE_ENABLED", False)
KAAS_AZURE_SUBSCRIPTION_ID = get_var("KAAS_AZURE_SUBSCRIPTION_ID", None)
KAAS_AZURE_TENANT_ID = get_var("KAAS_AZURE_TENANT_ID", None)
KAAS_AZURE_CLIENT_ID = get_var("KAAS_AZURE_CLIENT_ID", None)
KAAS_AZURE_CLIENT_SECRET = get_var("KAAS_AZURE_CLIENT_SECRET", None)
KAAS_AZURE_CLUSTER_LOCATION = get_var("KAAS_AZURE_CLUSTER_LOCATION", "eastus")
KAAS_AZURE_OS_TYPE = get_var("KAAS_AZURE_OS_TYPE", "Linux")
KAAS_AZURE_VM_SIZE = get_var("KAAS_AZURE_VM_SIZE", "Standard_F16s_v2")
KAAS_AZURE_VM_SIZE_CHILD = get_var("KAAS_AZURE_VM_SIZE_CHILD", "Standard_F8s_v2")
KAAS_AZURE_VM_DISK_SIZE = int(get_var("KAAS_AZURE_VM_DISK_SIZE", 128))

# Aws specific
AWS_PROVIDER_NAME = "aws"
KAAS_AWS_ENABLED = get_var_as_bool('KAAS_AWS_ENABLED', False)
AWS_SECRET_ACCESS_KEY = get_var('AWS_SECRET_ACCESS_KEY', None)
AWS_ACCESS_KEY_ID = get_var('AWS_ACCESS_KEY_ID', None)
AWS_DEFAULT_REGION = get_var('AWS_DEFAULT_REGION', 'us-east-2')
AWS_DEFAULT_AZ = get_var('AWS_DEFAULT_AZ', 'us-east-2a')
AWS_CLOUDFORMATION_STACK_NAME = get_var(
    'AWS_CLOUDFORMATION_STACK_NAME',
    'cluster-api-provider-aws-kaas-mirantis-com')

# ====
# Offline section
# ====

# Can be used only in core-ci with unstable templates
CORE_KAAS_NTP_ENABLED = get_var_as_bool(
    'CORE_KAAS_NTP_ENABLED', False)
# comma-separated ntp list to deploy with. Can be applied ONLY BEFORE deployment and ONLY for unstable templates
# EU related: 172.19.124.2, US: 172.16.243.0
CORE_KAAS_NTP_LIST = get_var('CORE_KAAS_NTP_LIST', '172.19.124.2,172.16.243.0')

# Enable or disable network isolation (for OS provider / MGMT/Reg/Child)
# And perform checks (only variables for now) for MGMT (VmWare/AWS/BM)
KAAS_OFFLINE_DEPLOYMENT = get_var_as_bool(
    'KAAS_OFFLINE_DEPLOYMENT', False)

# Install or not proxy on management cluster in case if you have no external
# proxy for checking child cases. Requires internet connection for mgmt to
# work as expected.
KAAS_INSTALL_FAKE_PROXY = get_var_as_bool('KAAS_INSTALL_FAKE_PROXY', False)

# Used in 'update/set NTP test' as NTP list source.
# EU related: 172.19.124.2, US: 172.16.243.0
KAAS_NTP_LIST = get_var('KAAS_NTP_LIST', '172.19.124.2,172.16.243.0')

# Install ntp on seed node parameters
SEED_NTP_SET_DATETIME = get_var('SEED_NTP_SET_DATETIME', '')
SEED_NTP_SET_TIMEZONE = get_var('SEED_NTP_SET_TIMEZONE', '')
SEED_NTP_POOLS = list(filter(None, get_var('SEED_NTP_POOLS').split(',')))

# Used to create a squid proxy on mgmt for usage by child
KAAS_ADDITIONAL_PROXY_YAML = os.path.expanduser(get_var(
    'KAAS_ADDITIONAL_PROXY_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/minimal_squid_proxy.yaml')))

KAAS_ADDITIONAL_PROXY_SVC_YAML = os.path.expanduser(get_var(
    'KAAS_ADDITIONAL_PROXY_SVC_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/minimal_squid_proxy_svc.yaml')))
KAAS_ADDITIONAL_PROXY_NS = get_var("KAAS_ADDITIONAL_PROXY_NS", 'default')
KAAS_ADDITIONAL_PROXY_USER = get_var('KAAS_ADDITIONAL_PROXY_USER', 'user')
KAAS_ADDITIONAL_PROXY_PASS = get_var('KAAS_ADDITIONAL_PROXY_PASS', 't0psecret')
KAAS_ADDITIONAL_PROXY_NAME = get_var(
    'KAAS_ADDITIONAL_PROXY_PASS', 'test-offline-proxy')
KAAS_ADDITIONAL_PROXY_IMAGE = get_var(
    'KAAS_ADDITIONAL_PROXY_IMAGE',
    'mirantis.azurecr.io/general/external/yegor256/squid-proxy:0.1')

# airgap variables
MCC_AIRGAP = get_var_as_bool("MCC_AIRGAP", False)
MCC_CDN_DOCKER = get_var("MCC_CDN_DOCKER", f"{SEED_STANDALONE_EXTERNAL_IP}")
MCC_CDN_BINARY = get_var("MCC_CDN_BINARY", f"https://{SEED_STANDALONE_EXTERNAL_IP}:8081")
MCC_CDN_DEBIAN = get_var("MCC_CDN_DEBIAN", f"https://{SEED_STANDALONE_EXTERNAL_IP}:8082")
MCC_CDN_MCR_REPO = get_var("MCC_CDN_MCR_REPO", f"https://{SEED_STANDALONE_EXTERNAL_IP}:8083")
MCC_CDN_CA_BUNDLE_PATH = get_var("MCC_CDN_CA_BUNDLE_PATH", f"/home/{SEED_SSH_LOGIN}/demo-ca.crt")
KAAS_AIRGAP_TOOLS_IMAGE = get_var("KAAS_AIRGAP_TOOLS_IMAGE", '')
KAAS_RELEASE_ARTIFACTS_TAR_URL = get_var("KAAS_RELEASE_ARTIFACTS_TAR_URL", '')
KAAS_RELEASES_TAR = get_var("KAAS_RELEASES_TAR", '')

KAAS_BOOTSTRAP_VERSION = get_var("KAAS_BOOTSTRAP_VERSION", None)

# Used in case if proxy already deployed outside of cluster
# Format: http://<user>:<password>@<ip_addr>:<port>
KAAS_EXTERNAL_PROXY_ACCESS_STR = get_var(
    'KAAS_EXTERNAL_PROXY_ACCESS_STR', None)
KAAS_NO_PROXY_ACCESS_STR = get_var('KAAS_NO_PROXY_ACCESS_STR', '')
KAAS_EXTERNAL_PROXY_ACCESS_STR_FOR_UPDATE = get_var(
    'KAAS_EXTERNAL_PROXY_ACCESS_STR_FOR_UPDATE', None)
KAAS_SSL_PROXY_CERTIFICATE_FILE = get_var(
    'KAAS_SSL_PROXY_CERTIFICATE_FILE',
    None)
KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE = get_var(
    'KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE',
    'custom_proxy.crt')
# Used to create a proxyobject to use in cluster deployment
KAAS_PROXYOBJECT_YAML = os.path.expanduser(get_var(
    'KAAS_PROXYOBJECT_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/extra-objects/proxy.yaml')))
KAAS_PROXYOBJECT_REGION = get_var('KAAS_PROXYOBJECT_REGION', 'region-one')
KAAS_PROXYOBJECT_NAME = get_var('KAAS_PROXYOBJECT_NAME', 'test-proxy')
KAAS_PROXYOBJECT_NO_PROXY = get_var('KAAS_PROXYOBJECT_NO_PROXY', '')
KAAS_PROXYOBJECT_NO_PROXY_AWS = get_var('KAAS_PROXYOBJECT_NO_PROXY_AWS', '10.96.0.1,169.254.169.254')

KAAS_FIPS_ENABLED = get_var('KAAS_FIPS_ENABLED', False)

KAAS_REGISTRYOBJECT_YAML = os.path.expanduser(get_var(
    'KAAS_REGISTRYOBJECT_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/extra-objects/containerregistry.yaml')))
KAAS_REGISTRYOBJECT_NAME = get_var('KAAS_REGISTRYOBJECT_NAME', 'test-container-registry')
KAAS_REGISTRYOBJECT_DOMAIN = get_var('KAAS_REGISTRYOBJECT_DOMAIN', '')
KAAS_REGISTRYOBJECT_CERT = get_var('KAAS_REGISTRYOBJECT_CERT', '')

KAAS_GRACEFULREBOOTREQUEST_YAML = os.path.expanduser(get_var(
    'KAAS_GRACEFULREBOOTREQUEST_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/extra-objects/gracefulrebootrequest.yaml')))
KAAS_GRACEFULREBOOTREQUEST_NAME = get_var('KAAS_GRACEFULREBOOTREQUEST_NAME', 'test-grr')
KAAS_GRACEFULREBOOTREQUEST_NAMESPACE = get_var('KAAS_GRACEFULREBOOTREQUEST_NAMESPACE', '')
KAAS_GRACEFULREBOOTREQUEST_MACHINES = get_var('KAAS_GRACEFULREBOOTREQUEST_MACHINES', '')
APPROX_NODE_GRACEFUL_REBOOT_TIME = int(get_var('APPROX_NODE_GRACEFUL_REBOOT_TIME', 600))

KAAS_RESTART_DS_YAML = os.path.expanduser(get_var(
    'KAAS_RESTART_DS_YAML',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/restart_ds.yaml')))
KAAS_RESTART_CHECKER_ENABLED = get_var_as_bool('KAAS_RESTART_CHECKER_ENABLED', True)

# Netchecker parameters
NETCHECKER_FILE_PATH = os.path.expanduser(get_var(
    "NETCHECKER_FILE_PATH",
    pkg_resources.resource_filename(__name__, 'templates/netchecker/netchecker.yaml')))
NETCHECKER_OBJECT_NAME = get_var('NETCHECKER_OBJECT_NAME', 'si-tests-infraconnectivitymonitor')
NETCHECKER_INVENTORY_CONFIG_NAME = get_var('NETCHECKER_INVENTORY_CONFIG_NAME', 'mcc-netchecker-inventory-config')
NETCHECKER_TARGETS_CONFIG_NAME = get_var('NETCHECKER_TARGETS_CONFIG_NAME', 'mcc-netchecker-targets-config')
NETCHECKER_NAMESPACE = get_var('NETCHECKER_NAMESPACE', 'netchecker')
NETCHECKER_OBJECTS_SPECS_FILE = get_var('NETCHECKER_OBJECTS_SPECS_FILE', '')
NETCHECKER_CLOUDPROBER_OVERRIDES_FILE = get_var('NETCHECKER_CLOUDPROBER_OVERRIDES_FILE', '')
NETCHECKER_SI_LABEL = get_var_from_json("NETCHECKER_SI_LABEL", '{"si-netchecker-label": "enabled"}')

# If parameter below is set to True, then it is possible to change targets and inventory configs manually
# and changes will not be overwritten by controller
NETCHECKER_SKIP_CONFIG_UPDATES = get_var_as_bool('NETCHECKER_SKIP_CONFIG_UPDATES', False)

VSPHERE_ENDPOINTS = "rhel-satellite-server.bud.mirantis.net,172.16.38.230"
# vSphere offline networks
FIP_RANGES = f"172.16.35.0/24,172.16.37.0/24,172.16.178.0/23,172.16.41.27/32,172.16.37.56/32,{VSPHERE_ENDPOINTS}"  # noqa
# Mirantis EU and US cloud fip ranges + vSphere offline networks
OS_FIP_RANGES = f"172.16.240.0/20,172.19.112.0/20,172.19.110.0/23,{FIP_RANGES}"  # noqa

# Additional network which will be accessible in offline mode
# format: 10.10.10.0/24,10.11.0.0/16
OFFLINE_CUSTOM_NETWORK_FORWARD = get_var(
    "OFFLINE_CUSTOM_NETWORK_FORWARD", None)
OFFLINE_SG_NAME = get_var('OFFLINE_SG_NAME', None)

# json describes pods that should not be restarted during tests.
# Where keys are namespaces names, and values are list of pod name patterns to check.
# format: {"openstack": ["openvswitch-openvswitch"]}
PODS_SHOULD_NOT_BE_RESTARTED_NAME_PATTERNS = get_var_from_json("PODS_SHOULD_NOT_BE_RESTARTED_NAME_PATTERNS", "{}")

OFFLINE_SG_RULES = [
    # Calico
    {"direction": "egress", "proto": "4", "cidr": "10.10.10.0/24"},

    {"direction": "egress", "proto": "icmp", "cidr": "10.10.10.0/24"},

    # OS EU
    {"direction": "egress", "proto": "tcp", "cidr": "172.16.37.56/32"},
    # OS US
    {"direction": "egress", "proto": "tcp", "cidr": "172.16.237.4/32"},
    # OS EU2
    {"direction": "egress", "proto": "tcp", "cidr": "172.19.108.56/32"},

    {"direction": "egress", "proto": "tcp", "cidr": "10.10.10.0/24"},

    {"direction": "egress", "proto": "udp", "cidr": "10.10.10.0/24"},

    # NTP
    {"direction": "egress", "proto": "tcp", "port": "123",
     "cidr": "0.0.0.0/0"},

    # Calico
    {"direction": "ingress", "proto": "4", "cidr": "10.10.10.0/24"},

    {"direction": "ingress", "proto": "icmp", "cidr": "0.0.0.0/0"},
    {"direction": "ingress", "proto": "tcp", "cidr": "10.10.10.0/24"},
    {"direction": "ingress", "proto": "tcp", "port": "22",
     "cidr": "0.0.0.0/0"},
    {"direction": "ingress", "proto": "udp", "cidr": "10.10.10.0/24"},

    # DNS
    {"direction": "egress", "proto": "tcp", "port": "53",
     "cidr": "{}/32".format(KAAS_CHILD_CLUSTER_DNS.split(',')[0])},

    # PRODX-1375 Grafana
    {"direction": "egress", "proto": "tcp", "port": "443",
     "cidr": "192.30.255.120/32"},
    {"direction": "egress", "proto": "tcp", "port": "443",
     "cidr": "35.241.23.245/32"},
    {"direction": "egress", "proto": "tcp", "port": "443",
     "cidr": "140.82.113.10/32"},
    {"direction": "egress", "proto": "tcp", "port": "443",
     "cidr": "140.82.114.9/32"},
    {"direction": "egress", "proto": "tcp", "port": "443",
     "cidr": "140.82.113.9/32"},

    # PRODX-1338
    {"direction": "egress", "proto": "tcp", "port": "443",
     "cidr": "5.43.224.93/32"},

    # DNS
    {"direction": "egress", "proto": "udp", "port": "53",
     "cidr": "{}/32".format(KAAS_CHILD_CLUSTER_DNS.split(',')[0])},

    # vSphere server
    {"direction": "egress", "proto": "tcp", "port": "443",
     "cidr": "172.16.41.27/32"},
]

# Update mgmt setting
KAAS_RELEASE_REPO_URL = get_var("KAAS_RELEASE_REPO_URL", "")
KAAS_CORE_REPO_URL = get_var("KAAS_CORE_REPO_URL", "")  # Used to check that we are using a correct CDN

# DEBUG_DEV_MODE can be used to get additional previlegies on cluster, like using
# images from internal docker repo and so on
KAAS_DEPLOY_DEBUG_DEV_MODE = get_var_as_bool('KAAS_DEPLOY_DEBUG_DEV_MODE', False)

RUN_POD_CHECKS = get_var_as_bool('RUN_POD_CHECKS', True)

# For default checks PRODX-3613
BOOTSTRAP_WITH_DEFAULTS = get_var_as_bool('BOOTSTRAP_WITH_DEFAULTS', False)
KAAS_RELEASES_BASE_URL = get_var("KAAS_RELEASES_BASE_URL", "")
# manage bootstrap logging verbosity level (4 - is preferred debug, 10 max)
KAAS_BOOTSTRAP_LOG_LVL = get_var("KAAS_BOOTSTRAP_LOG_LVL", "0")
KAAS_DELETE_COLLECTED_LOGS = get_var_as_bool('KAAS_DELETE_COLLECTED_LOGS', True)
KAAS_BOOTSTRAP_INFINITE_TIMEOUT = get_var_as_bool('KAAS_BOOTSTRAP_INFINITE_TIMEOUT', False)

SI_CHECK_MACHINES_STATUS_TIMEOUT = int(get_var("SI_CHECK_MACHINES_STATUS_TIMEOUT", "6300"))

# Deploy OpenStack on UCP env settings
OPENSTACK_DEPLOY_SSH_KEY_PATH = get_var(
    "OPENSTACK_DEPLOY_SSH_KEY_PATH",
    os.path.join(ARTIFACTS_DIR, "id_rsa_ucp"))

OPENSTACK_DEPLOY_SSH_USERNAME = get_var(
    "OPENSTACK_DEPLOY_SSH_USERNAME", "mcc-user")

OPENSTACK_DEPLOY_HOSTNAME = get_var(
    "OPENSTACK_DEPLOY_HOSTNAME")

OPENSTACK_DEPLOY_RELEASE_DIR = os.path.abspath(get_var(
    "OPENSTACK_DEPLOY_RELEASE_DIR",
    "release-openstack-k8s"))

OPENSTACK_DEPLOY_CONTEXT_DIR = get_var(
    "OPENSTACK_DEPLOY_CONTEXT_DIR", "mcp-pipelines")

OPENSTACK_DEPLOY_TIMEOUT = json.loads(
    get_var("OPENSTACK_DEPLOY_TIMEOUT", '{"1": 1200, "2": 8000, "3": 180}'))

OPENSTACK_DEPLOY_SINGLE_TEST_TIMEOUT = int(get_var(
    "OPENSTACK_DEPLOY_SINGLE_TEST_TIMEOUT", "1200"))

OPENSTACK_DEPLOY_CLUSTER_INFO_YAML = get_var(
    "OPENSTACK_DEPLOY_CLUSTER_INFO_YAML",
    os.path.join(ARTIFACTS_DIR, "cluster_info.yaml"))

OPENSTACK_DEPLOY_NODES_METADATA_YAML = get_var(
    "OPENSTACK_DEPLOY_NODES_METADATA_YAML",
    os.path.join(ARTIFACTS_DIR, "nodes_metadata.yaml"))

OPENSTACK_DEPLOY_CONTEXT_NAME = get_var(
    "OPENSTACK_DEPLOY_CONTEXT_NAME")

OPENSTACK_DEPLOY_FAKE_DEPLOYMENT = get_var_as_bool(
    "OPENSTACK_DEPLOY_FAKE_DEPLOYMENT", False)

OPENSTACK_DEPLOY_OPENSTACK_VERSION = get_var(
    "OPENSTACK_DEPLOY_OPENSTACK_VERSION")

# Leave empty to deploy the default version provided by TFOperator.
OPENSTACK_DEPLOY_TF_VERSION = get_var(
    "OPENSTACK_DEPLOY_TF_VERSION", "")

OPENSTACK_UPDATE_TF_VERSION = get_var(
    "OPENSTACK_UPDATE_TF_VERSION", "24.1")

OPENSTACK_DEPLOY_VAULT = get_var_as_bool(
    "OPENSTACK_DEPLOY_VAULT", True)

OPENSTACK_DEPLOY_IAM = get_var_as_bool(
    "OPENSTACK_DEPLOY_IAM", False)

OPENSTACK_USE_MGMT_IAM = get_var_as_bool(
    "OPENSTACK_USE_MGMT_IAM", True)

OPENSTACK_DEPLOY_STACKLIGHT = get_var_as_bool(
    "OPENSTACK_DEPLOY_STACKLIGHT", False)

OPENSTACK_DEPLOY_EXTRA_CONTEXT_FILE = get_var(
    "OPENSTACK_DEPLOY_EXTRA_CONTEXT_FILE",
    os.path.join(ARTIFACTS_DIR, "extra_context.yaml"))

OPENSTACK_DEPLOY_TELEMETRY = get_var_as_bool(
    "OPENSTACK_DEPLOY_TELEMETRY", False)

OPENSTACK_DEPLOY_MANILA = get_var_as_bool(
    "OPENSTACK_DEPLOY_MANILA", False)

OPENSTACK_DEPLOY_MASAKARI = get_var_as_bool(
    "OPENSTACK_DEPLOY_MASAKARI", False)

OPENSTACK_DEPLOY_REFAPP_DIR = get_var(
    "OPENSTACK_DEPLOY_REFAPP_DIR", "openstack-refapp")

OPENSTACK_REFAPP_RECORDS_FILE = get_var(
    "OPENSTACK_REFAPP_RECORDS_FILE",
    os.path.join(ARTIFACTS_DIR, "refapp_records.yaml"))

OPENSTACK_REFAPP_DEPLOYER = get_var(
    "OPENSTACK_REFAPP_DEPLOYER", "heat")

OPENSTACK_REFAPP_IMAGE = get_var("OPENSTACK_REFAPP_IMAGE")

# valid values for next param: on_error, always
OPENSTACK_COLLECT_REFAPP_MODE = get_var("OPENSTACK_COLLECT_REFAPP_MODE", "on_error")

COLLECT_REFAPP_STATISTIC_RETRIES = int(get_var(
    "COLLECT_REFAPP_STATISTIC_RETRIES", 1000))

OPENSTACK_TEST_SCHEME_PATH = get_var(
    "OPENSTACK_TEST_SCHEME_PATH",
    os.path.join(ARTIFACTS_DIR, "test_scheme.yaml"))

OPENSTACK_NODES_WITH_HUGEPAGES_LABEL = get_var(
    "OPENSTACK_NODES_WITH_HUGEPAGES_LABEL",
    "openstack-compute-node-dpdk=enabled")

OPENSTACK_TF_VSRX_ROUTER = get_var(
    "OPENSTACK_TF_VSRX_ROUTER", "")

OPENSTACK_DEPLOY_USE_HUGEPAGES = get_var_as_bool(
    "OPENSTACK_DEPLOY_USE_HUGEPAGES", False)

TF_NODES_WITH_DPDK_LABEL = get_var(
    "TF_NODES_WITH_DPDK_LABEL",
    "tfvrouter-dpdk=enabled")

OPENSTACK_DEPLOY_DPDK = get_var_as_bool(
    "OPENSTACK_DEPLOY_DPDK", False)

OPENSTACK_DEPLOY_SRIOV = get_var_as_bool(
    "OPENSTACK_DEPLOY_SRIOV", False)

OPENSTACK_DEPLOY_NEUTRON_PORTPROBER = get_var_as_bool(
    "OPENSTACK_DEPLOY_NEUTRON_PORTPROBER", False)

OPENSTACK_AVAILABILITY_ZONE_NAME = get_var(
    "OPENSTACK_AVAILABILITY_ZONE_NAME", "nova")

OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME = get_var(
    "OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME", "nova")

OPENSTACK_DPDK_DRIVER_PACKAGE = get_var(
    "OPENSTACK_DPDK_DRIVER_PACKAGE", "")

DEFAULT_OPENSTACK_DPDK_DRIVER_PACKAGE_BY_DISTRO = {
    "focal": "dpdk-igb-uio-dkms",
    "jammy": "dpdk-kmods-dkms",
}

OPENSTACK_ENCRYPTED_VOLUME = get_var_as_bool(
    "OPENSTACK_ENCRYPTED_VOLUME", False)

OPENSTACK_ENCRYPTED_VOLUME_NAME = get_var(
    "OPENSTACK_ENCRYPTED_VOLUME_NAME", "volumes_luks")

OPENSTACK_RMQ_EXTERNAL_NOTIFICATIONS = get_var_as_bool(
    "OPENSTACK_RMQ_EXTERNAL_NOTIFICATIONS", False)

MOSK_FLOATING_NETWORKS = get_var("MOSK_FLOATING_NETWORKS", "")
MOSK_CHILD_CLUSTER_TEMPLATES_BASE_DIR = "/home/{}/{}/{}".format(
    KAAS_MGMT_CLUSTER_BOOTSTRAP_USERSPACE, KAAS_NAMESPACE, KAAS_CHILD_CLUSTER_NAME)

# Flag to trigger ceph deployment on child cluster, controlls only ceph deployment.
# Ceph controller deployment is not controlled by this flag.
MOSK_CHILD_DEPLOY_CEPH = get_var_as_bool("MOSK_CHILD_DEPLOY_CEPH", True)

SCALE_NODE_NAME_PATTERN = get_var(
    "SCALE_NODE_NAME_PATTERN", "scale-node")

ROOK_CEPH_NS = get_var(
    "ROOK_CEPH_NS", "rook-ceph")

CEPH_CLUSTER_NAME_PREFIX = get_var(
    "CEPH_CLUSTER_NAME_PREFIX", "ceph-")

CEPH_CLUSTER_NAME = CEPH_CLUSTER_NAME_PREFIX + \
                    TARGET_CLUSTER

CEPH_CWL_NAME = get_var(
    "CEPH_CWL_NAME", "ceph-clusterworkloadlock")

CEPH_DEVICE_ROLE = get_var(
    "CEPH_DEVICE_ROLE", "hdd")

CEPH_LCM_MIRANTIS_NS = get_var(
    "CEPH_LCM_MIRANTIS_NS", "ceph-lcm-mirantis")

CEPH_DEPLOY_RGW = get_var_as_bool("CEPH_DEPLOY_RGW", False)
CEPH_RGW_CUSTOM_SSL_SECRET_PATH = os.path.expanduser(get_var(
    "CEPH_RGW_CUSTOM_SSL_SECRET_PATH",
    pkg_resources.resource_filename(__name__, 'templates/ceph/rgw-custom-ssl.yaml')))

CEPH_DEPLOY_MDS = get_var_as_bool("CEPH_DEPLOY_MDS", False)

CEPH_EXEC_CRASH_ARCHIVE_ALL = get_var_as_bool("CEPH_EXEC_CRASH_ARCHIVE_ALL", False)

CEPH_HC_DATA_PATH = get_var(
    "CEPH_HC_DATA_PATH",
    "si_tests/templates/ceph/ceph_hyperconverge_data.yaml")

CEPH_OPERATIONREQUEST_YAML_PATH = get_var(
    "CEPH_OPERATIONREQUEST_YAML_PATH",
    "si_tests/templates/ceph/ceph-operations-request.yaml")
CEPH_OSDREMOVEREQUEST_YAML_PATH = get_var(
    "CEPH_OSDREMOVEREQUEST_YAML_PATH",
    "si_tests/templates/ceph/ceph-osdremove-request.yaml")

# Ceph performance test related parameters
CEPH_PERF_TEST_REQUEST_YAML_PATH = get_var(
    "CEPH_PERF_TEST_REQUEST_YAML_PATH",
    "si_tests/templates/ceph/ceph-performance-test-request.yaml")
CEPH_PERFTEST_CHILD_REQUEST_YAML_PATH = get_var(
    "CEPH_PERF_TEST_REQUEST_YAML_PATH",
    "si_tests/templates/ceph/ceph-perftest-request.yaml")
CEPH_PERF_TEST_OPTIONS = get_var_from_json("CEPH_PERF_TEST_OPTIONS")
CEPH_PERF_TEST_RBD_IMAGE_SIZE = get_var("CEPH_PERF_TEST_RBD_IMAGE_SIZE", '10G')
CEPH_PERF_TEST_PERIODIC = get_var_as_bool("CEPH_PERF_TEST_PERIODIC", False)
CEPH_PERF_TEST_SCHEDULE = get_var("CEPH_PERF_TEST_SCHEDULE", '0 0 * * *')
CEPH_PERF_TEST_RUNS_TO_KEEP = int(get_var("CEPH_PERF_TEST_RUNS_TO_KEEP", 3))
CEPH_PERF_TEST_IMAGE = get_var("CEPH_PERF_TEST_IMAGE", '')
CEPH_PERF_TEST_ENTRYPOINT_COMMAND = get_var("CEPH_PERF_TEST_ENTRYPOINT_COMMAND", '')

# Ceph PG rebalance timeout for requests. The default is 30 minutes
CEPH_PG_REBALANCE_TIMEOUT = int(get_var("CEPH_PG_REBALANCE_TIMEOUT", 30))

FAILED_DISK_NODE_NAME_PATTERN = get_var(
    "FAILED_DISK_NODE_NAME_PATTERN", "")

FAILED_DISK_NAME = get_var(
    "FAILED_DISK_NAME", "sdc")

MOSK_PRIVATE_KEY_PATH = get_var(
    "MOSK_PRIVATE_KEY_PATH", "")

REGION_NAMESPACE = get_var("REGION_NAMESPACE", "default")

REGIONAL_CLUSTER_NAME = get_var("REGIONAL_CLUSTER_NAME",
                                "si-tests-region-cluster")

OS_REGION = get_var("OS_REGION", "region-one")

AWS_REGION = get_var("AWS_REGION", "region-one")

# Name for deployment regional cluster, also required after 2.3 even for
# kaas-mgmt, in case we use custom bmhp.
BM_REGION = get_var("BM_REGION", "region-one")

VSPHERE_REGION = get_var("VSPHERE_REGION", "region-one")

EQUINIX_REGION = get_var("EQUINIX_REGION", "region-one")

EQUINIXMETALV2_REGION = get_var("EQUINIXMETALV2_REGION", "region-one")

AZURE_REGION = get_var("AZURE_REGION", "region-one")

DISABLE_IP_CONSISTENCY_CHECK = get_var_as_bool("DISABLE_IP_CONSISTENCY_CHECK",
                                               False)
DISABLE_INTROSPECT_PRODX_9696 = get_var_as_bool("DISABLE_INTROSPECT_PRODX_9696",
                                                False)

# sleep time for bm child bmh ops
KAAS_BM_CHILD_SLEEP_PRODX_7751 = int(get_var("KAAS_BM_CHILD_SLEEP_PRODX_7751", 15))
# kaas-bm hack, dont even try to use it.
KAAS_BM_DUMMY_PROVISION = get_var_as_bool('KAAS_BM_DUMMY_PROVISION', False)
KAAS_BM_DUMMY_PROVISION_SIZE = int(get_var("KAAS_BM_DUMMY_PROVISION_SIZE", 8))

KAAS_BM_EXTRA_BMC_ENVS = ['phys-429', 'phys-429-vsphere', 'phys-1490', 'system-phys-410',
                          'phys-429-multirack', 'phys-1481', 'phys-1481-mosk']
# reflected option from bm-operator
KAAS_BM_PROVISIONING_LIMIT = int(get_var("KAAS_BM_PROVISIONING_LIMIT", 5))
# see kaas_manager.wait_baremetalhosts_statuses func for more info
KAAS_BM_WAIT_STATUS_TRANSITION = int(get_var("KAAS_BM_WAIT_STATUS_TRANSITION", 50))
# wait for bmh machine deleted
KAAS_BM_WAIT_BMH_MACHINE_DELETE_RETRY = int(get_var("KAAS_BM_WAIT_BMH_MACHINE_DELETE_RETRY", 60))
# vbmc internal WA, related to migration which we dont have, for vmbc
# should be removed after 2.25 release
KAAS_BM_PRODX_34279 = get_var_as_bool('KAAS_BM_PRODX_34279', False)
# where to configure 'kaasbm' user in BMH post deployment
KAAS_BM_DEBUG_USER_ENABLED = get_var_as_bool('KAAS_BM_DEBUG_USER_ENABLED', True)

# use user-provided/self-signed certificates for user-facing endpoints MCC,Keycloak UIs
CUSTOM_CERTIFICATES = get_var_as_bool("CUSTOM_CERTIFICATES", False)

CHILD_MKE_CUSTOM_CERTIFICATES = get_var_as_bool("CHILD_MKE_CUSTOM_CERTIFICATES", False)

CHECK_READINESS = get_var_as_bool("CHECK_READINESS", False)

BM_DEPLOY_MD_RAID = get_var_as_bool("BM_DEPLOY_MD_RAID", False)

BM_DEPLOY_LVM_RAID = get_var_as_bool("BM_DEPLOY_LVM_RAID", False)

# internal -bm hack, PRODX-42152
BM_DEPLOY_MGMT_ON_VSPHERE = get_var_as_bool('BM_DEPLOY_MGMT_ON_VSPHERE', False)

DISTRIBUTION_RELEVANCE = get_var('DISTRIBUTION_RELEVANCE', 'default')

# https://mirantis.jira.com/browse/PRODX-20891
CUSTOM_HOSTNAMES = get_var_as_bool("CUSTOM_HOSTNAMES", False)

# If auto-delay enabled, then mgmt cluster not upgraded automaticaly untill 20 days pass from release date
MCC_UPGRADE_AUTO_DELAY_ENABLED = get_var_as_bool("MCC_UPGRADE_AUTO_DELAY_ENABLED", False)
MCC_UPGRADE_OBJECT_NAME = get_var('MCC_UPGRADE_OBJECT_NAME', 'mcc-upgrade')

CEPH_PV_CHECK_POD_YAML = os.path.expanduser(get_var(
    'POD-NGINX-CEPH-PVC',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/pod-nginx-ceph-volume.yaml')))

CEPH_S3_CONFIGMAP_YAML = os.path.expanduser(
    get_var(
        'CONFIGMAP-CEPH-S3',
        pkg_resources.resource_filename(__name__, 'templates/configmaps/ceph_s3_config.yaml')
    )
)
CEPH_S3_CLIENT_YAML = os.path.expanduser(get_var(
    'POD-CEPH-S3-CLIENT',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/pod-ceph-s3-client.yaml')))

CEPH_PVC_YAML = os.path.expanduser(get_var(
    'CEPH-PVC',
    pkg_resources.resource_filename(
        __name__,
        'templates/pvc/pvc-ceph.yaml')))

CONTAINERREGISTRY_CHECK_POD_YAML = os.path.expanduser(get_var(
    'POD-CONTAINERREGISTRY',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/pod-containerregistry.yaml')))

CONTAINERREGISTRY_BASE_IMAGE = get_var("CONTAINERREGISTRY_BASE_IMAGE", 'msr.infra.mirantis.net/repo/nginx:1.19.6')

CLUSTER_TEMPLATES = get_var(
    "CLUSTER_TEMPLATES",
    pkg_resources.resource_filename(__name__,
                                    'templates/cluster-configs'))

# Bastion node configuration
KAAS_BASTION_CONFIG = get_var("KAAS_BASTION_CONFIG", None)

KAAS_AWS_BASTION_CONFIG = {
    "amiId": get_var("KAAS_BASTION_AMI_ID", ""),
    "instanceType": get_var("KAAS_BASTION_INSTANCE_TYPE", "")
}

KAAS_BASTION_OS_CONFIG = {
    "serverMetadata": get_var_from_json("KAAS_BASTION_SERVER_METADATA", "{}"),
    "flavor": get_var("KAAS_BASTION_FLAVOR", KAAS_CHILD_CLUSTER_MACHINE_FLAVOR_NAME),
    "image": get_var("KAAS_BASTION_IMAGE", KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME),
    "availabilityZone": get_var("KAAS_BASTION_AVAILABILITY_ZONE", KAAS_CHILD_CLUSTER_AZ_NAME),
    "bootFromVolume": {
        "enabled": get_var_as_bool("KAAS_BASTION_BOOT_FROM_VOLUME", False),
        "volumeSize": int(get_var("KAAS_BASTION_BOOT_FROM_VOLUME_SIZE", 0))
    }
}


def get_machine_updating_status():
    return "Deploy"


TERRAFORM_NETWORK_METADATA = get_var("TERRAFORM_NETWORK_METADATA", "output.json")

KAAS_CUSTOM_DEPLOYMENT_BOOTSTRAP_TEMPLATES_TARBALL = get_var("KAAS_CUSTOM_DEPLOYMENT_BOOTSTRAP_TEMPLATES_TARBALL")

SYSTEM_ADMINS = ['root', 'administrator']

KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED = \
    get_var_as_bool("KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED", False)

MOSK_RELEASE_PREFIX = "mosk"

MARIADB_BACKUP_JOB_TIMEOUT = int(get_var("MARIADB_BACKUP_JOB_TIMEOUT", "600"))
MARIADB_RESTORE_JOB_TIMEOUT = int(get_var("MARIADB_RESTORE_JOB_TIMEOUT", "1200"))
MARIADB_FILL_DATA_POD_YAML = pod_template_path = os.path.expanduser(
    pkg_resources.resource_filename(__name__,
                                    'templates/pods/mariadb-fill-data.yaml'))
MARIADB_FILL_DATA_TIMEOUT = int(get_var("MARIADB_FILL_DATA_TIMEOUT", "5400"))
MARIADB_FILL_DATA_SIZE_MB = int(get_var("MARIADB_FILL_DATA_SIZE_MB", "15360"))
MARIADB_FILL_DATA_IMAGE = get_full_image_path(
    get_var("MARIADB_FILL_DATA_IMAGE", "openstack/extra/mariadb-test:focal-20220723145954"))

MCC_BOOTSTRAP_VERSION = get_var("MCC_BOOTSTRAP_VERSION", "v1")
MCC_BOOTSTRAPV2_INIT_TIMEOUT = int(get_var("MCC_BOOTSTRAPV2_INIT_TIMEOUT", 1200))
MCC_BV2_MACHINES_DEPLOY_TIMEOUT = int(get_var(
    "MCC_BV2_MACHINES_DEPLOY_TIMEOUT", 3000))
MCC_BV2_CLUSTER_BOOTSTRAP_READINESS_TIMEOUT = int(get_var(
    "MCC_BV2_CLUSTER_BOOTSTRAP_READINESS_TIMEOUT", 3000))
MCC_BV2_BM_MACHINES_DEPLOY_TIMEOUT = int(get_var(
    "MCC_BV2_MACHINES_DEPLOY_TIMEOUT", 4800))
MCC_BV2_CLUSTER_READINESS_TIMEOUT = int(get_var(
    "MCC_BV2_CLUSTER_READINESS_TIMEOUT", 2400))
MCC_BV2_BOOTSTRAP_REGION_READINESS_TIMEOUT = int(get_var(
    "MCC_BV2_BOOTSTRAP_REGION_READINESS_TIMEOUT", 1800))

RUN_ON_REMOTE = get_var_as_bool('RUN_ON_REMOTE', False)

EQUINIX_SKIP_FACILITY_SELECTION = get_var_as_bool("EQUINIX_SKIP_FACILITY_SELECTION", False)
EQUINIX_FACILITY = get_var("EQUINIX_FACILITY", "")

# Enable FRR installation on gtw node
METALLB_BGP_ENABLE = get_var_as_bool("METALLB_BGP_ENABLE", False)

MACHINE_DELETION_POLICY_ENABLED = get_var_as_bool("MACHINE_DELETION_POLICY_ENABLED", True)
CHECK_WORKLOAD_LOCKS_ON_MACHINE_DELETE = get_var_as_bool("CHECK_WORKLOAD_LOCKS_ON_MACHINE_DELETE", False)

# Enable auditd
AUDITD_ENABLE = get_var_as_bool("AUDITD_ENABLE", False)

# Enable k8s api profiling
K8S_PROFILING_ENABLE = get_var_as_bool("K8S_PROFILING_ENABLE", False)

# Parameters for workload generation and downtimes analysis
ALLOW_WORKLOAD = get_var_as_bool('ALLOW_WORKLOAD', False)
SI_LOADTOOLS_DOCKER_IMAGE_URL = get_var('SI_LOADTOOLS_DOCKER_IMAGE_URL',
                                        'mirantis.azurecr.io/si/si-loadtools:1.0.1-e58cdf7')
SI_LOADTOOLS_DEFAULT_LOCUST_TAGS = list(filter(None, get_var(
    'SI_LOADTOOLS_DEFAULT_LOCUST_TAGS', 'refapp').split(',')))
SI_LOADTOOLS_CHECK_KEYSTONE_WORKLOAD = get_var_as_bool('SI_LOADTOOLS_CHECK_KEYSTONE_WORKLOAD', False)
SI_LOADTOOLS_CHECK_TF_API_WORKLOAD = get_var_as_bool('SI_LOADTOOLS_CHECK_TF_API_WORKLOAD', False)
SI_LOADTOOLS_OS_ADMIN_USERNAME = get_var('SI_LOADTOOLS_OS_ADMIN_USERNAME', 'si-openstack-admin-user')
SI_LOADTOOLS_OS_ADMIN_PASSWORD = get_var('SI_LOADTOOLS_OS_ADMIN_PASSWORD', 'si-openstack-admin-password')

DOWNTIME_STATISTIC_FILE_PATH = get_var("DOWNTIME_STATISTIC_FILE_PATH",
                                       os.path.join(ARTIFACTS_DIR, "downtime-statistic.yaml"))

# We can use custom config where we can pass chunk size and other parameters to more precise calculations
DOWNTIME_STATISTIC_USE_CUSTOM_CONFIG = get_var_as_bool('DOWNTIME_STATISTIC_USE_CUSTOM_CONFIG', False)
DOWNTIME_STATISTIC_CUSTOM_CONFIG = os.path.expanduser(get_var(
    'DOWNTIME_STATISTIC_CUSTOM_CONFIG', pkg_resources.resource_filename(
        __name__, 'templates/downtime-statistic-config/downtime-statistic-custom-config.yaml')))

REFAPP_EXPECTED_DOWNTIME = int(get_var('REFAPP_EXPECTED_DOWNTIME', 5))
MOSK_WORKLOAD_DOWNTIME_REPORT = get_var_as_bool('MOSK_WORKLOAD_DOWNTIME_REPORT', ALLOW_WORKLOAD)
# TODO(mkarpin): implement separate project/domain for si-tests
MOSK_WORKLOAD_PROJECT_NAME = "admin"

# Cache Warmup
KAAS_CACHE_WARMUP_CLUSTER_RELEASES = get_var("KAAS_CACHE_WARMUP_CLUSTER_RELEASES", '')
KAAS_CACHE_WARMUP_OPENSTACK_RELEASES = get_var("KAAS_CACHE_WARMUP_OPENSTACK_RELEASES", '')
KAAS_CACHE_WARMUP_OPENSTACK_ONLY = get_var_as_bool("KAAS_CACHE_WARMUP_OPENSTACK_ONLY", False)
KAAS_CACHE_WARMUP_CLIENTS_PER_EP = int(get_var("KAAS_CACHE_WARMUP_CLIENTS_PER_EP", 2))
KAAS_CACHE_WARMUP_READINESS_TIMEOUT = int(get_var("KAAS_CACHE_WARMUP_READINESS_TIMEOUT", 4500))

# Support mgmt/child clusters on BM and OS
ENFORCE_ENABLED = get_var_as_bool("ENFORCE_ENABLED", False)

# TLSConfig test
SEED_PDNS_LOCAL_ZONE = get_var("SEED_PDNS_LOCAL_ZONE", 'sitest')
SEED_PDNS_USE_OFFICIAL_REPOS = get_var_as_bool("SEED_PDNS_USE_OFFICIAL_REPOS", False)

# https://mirantis.jira.com/wiki/spaces/NG/pages/4627007770/Patch+releases+for+MCC+MOSK
# child + mgmt
MAJOR_RELEASES_FROM_WIKI = [
    '15.0.1'
]

# OpenStack Controller functional
OPENSTACK_CONTROLLER_TESTS_POD_TEMPLATE = os.path.expanduser(get_var(
    'OPENSTACK_CONTROLLER_TESTS_POD_TEMPLATE',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/openstack_controller_tests.yaml')))
OPENSTACK_CONTROLLER_TESTS_NETWORK_POLICY = os.path.expanduser(get_var(
    'OPENSTACK_CONTROLLER_TESTS_NETWORK_POLICY',
    pkg_resources.resource_filename(
        __name__,
        'templates/networkpolicies/openstack_controller_tests.yaml')))
OPENSTACK_CONTROLLER_TESTS_POD_NAME = get_var("OPENSTACK_CONTROLLER_TESTS_POD_NAME", "openstack-controller-tests")
OPENSTACK_CONTROLLER_TESTS_POD_NAMESPACE = get_var("OPENSTACK_CONTROLLER_TESTS_POD_NAMESPACE", "osh-system")
OPENSTACK_CONTROLLER_TESTS_TIMEOUT = int(get_var("OPENSTACK_CONTROLLER_TESTS_TIMEOUT", "3600"))

# OpenStack Controller sos report
OPENSTACK_CONTROLLER_SOS_REPORT_POD_TEMPLATE = os.path.expanduser(get_var(
    'OPENSTACK_CONTROLLER_SOS_REPORT_POD_TEMPLATE',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/openstack_controller_sos_report.yaml')))
OPENSTACK_CONTROLLER_SOS_REPORT_TIMEOUT = int(get_var("OPENSTACK_CONTROLLER_SOS_REPORT_TIMEOUT", "1800"))

# Performance Analyzer settings
PERF_ANALYZER_DIR = os.path.abspath(get_var(
    "PERF_ANALYZER_DIR",
    "perf-analyzer"))

PERF_ANALYZER_SCENARIO_FILE = os.path.abspath(get_var("PERF_ANALYZER_SCENARIO_FILE", "setMe"))
PERF_ANALYZER_DNS_WAIT_TIMEOUT = int(get_var("PERF_ANALYZER_DNS_WAIT_TIMEOUT", 14400))
PERF_ANALYZER_ETCD_WAIT_TIMEOUT = int(get_var("PERF_ANALYZER_ETCD_WAIT_TIMEOUT", 300))
PERF_ANALYZER_MARIADB_WAIT_TIMEOUT = int(get_var("PERF_ANALYZER_MARIADB_WAIT_TIMEOUT", 4500))
PERF_ANALYZER_RABBITMQ_WAIT_TIMEOUT = int(get_var("PERF_ANALYZER_RABBITMQ_WAIT_TIMEOUT", 4500))

# Rulexam tests
RULEXAM_TESTS_POD_TEMPLATE = os.path.expanduser(get_var(
    'RULEXAM_TESTS_POD_TEMPLATE',
    pkg_resources.resource_filename(
        __name__,
        'templates/pods/rulexam_tests.yaml')))
RULEXAM_TESTS_POD_NAME = get_var("RULEXAM_TESTS_POD_NAME", "rulexam-tests")
RULEXAM_TESTS_POD_NAMESPACE = get_var("RULEXAM_TESTS_POD_NAMESPACE", "openstack")
RULEXAM_TESTS_TIMEOUT = int(get_var("RULEXAM_TESTS_TIMEOUT", "3600"))

# Host OS Configuration settings
HOCM_ENABLE_DAY2_DEV_MODULES = get_var_as_bool('HOCM_ENABLE_DAY2_DEV_MODULES', False)
HOCM_DAY2_INDEX_DEV_URL = get_var('HOCM_DAY2_INDEX_DEV_URL',
                                  'https://artifactory.mcp.mirantis.net/artifactory/binary-dev-kaas-virtual/'
                                  'bm/bin/host-os-modules/index-dev.yaml')

CHECK_SKIP_DAY2_HOCM_CREATE_CONFLICT = get_var_as_bool('CHECK_SKIP_DAY2_HOCM_CREATE_CONFLICT', False)
HOC_USE_CUSTOM_TMPFILE_MODULE = get_var_as_bool('HOC_USE_CUSTOM_TMPFILE_MODULE', True)
# HOC_USE_BUILTIN_MCC_MODULE_VERSIONS default to False as BM team uses a fixed modules list
HOC_USE_BUILTIN_MCC_MODULE_VERSIONS = get_var_as_bool('HOC_USE_BUILTIN_MCC_MODULE_VERSIONS', False)
# If HOC_USE_BUILTIN_MCC_MODULE_VERSIONS is False, then test HOC with the following modules:
HOC_FIXED_SUPPORTED_MODULE_VERSIONS = get_var_from_json(
    "HOC_FIXED_SUPPORTED_MODULE_VERSIONS",
    '{"sysctl": ["1.1.0"], "package": ["1.2.0"], "tmpfile": ["1.0.0"], "irqbalance": ["1.0.0"]}')
HOC_USE_SKIP_MODULE_VERSIONS = get_var_as_bool('HOC_USE_SKIP_MODULE_VERSIONS', True)
# If HOC_USE_SKIP_MODULE_VERSIONS is True, then skip the follwing module versions:
HOC_SKIP_MODULE_VERSIONS = get_var_from_json(
    "HOC_SKIP_MODULE_VERSIONS",
    '{"cpushield": ["1.0.0"], "irqbalance": ["1.0.0"], "grub_settings": ["1.0.0"],'
    '"sysctl": ["1.0.0", "1.1.0"], "package": ["1.0.0", "1.1.0"]}')
# package
HOC_PACKAGE_MODULE_VERSION = get_var('HOC_PACKAGE_MODULE_VERSION', '1.2.0')
HOC_PACKAGE_MODULE_SI_TEST_PKG_NAME = get_var('HOC_PACKAGE_MODULE_SI_TEST_PKG_NAME', 'bubblewrap')
HOC_CPUSHIELD_MODULE_VERSION = get_var('HOC_CPUSHIELD_MODULE_VERSION', '1.0.0')

HOC_CPUSHIELD_TEST_HOC_NAME = get_var('HOC_CPUSHIELD_TEST_HOC_NAME', 'test-cpushield-module')
HOC_CPUSHIELD_REBOOT_LABEL = get_var_from_json(
    "HOC_CPUSHIELD_REBOOT_LABEL",
    '{"day2-cpushield-test": "enabled"}')
ROLLING_REBOOT_MACHINES_LABELS = list(filter(None, get_var(
    'ROLLING_REBOOT_MACHINES_LABELS').split(',')))
ROLLING_REBOOT_MACHINES_KERNEL_CHECK = get_var_as_bool('ROLLING_REBOOT_MACHINES_KERNEL_CHECK', True)

# linux_hard_limits module
HOC_LHL_PACKAGE_MODULE_VERSION = get_var('HOC_LHL_PACKAGE_MODULE_VERSION', '1.0.0')
HOC_LHL_TEST_HOC_NAME = get_var('HOC_LHL_TEST_HOC_NAME', 'linux-hard-limits')

# Dignostic settings
DIAGNOSTIC_FINISHING_TIMEOUT = int(get_var("DIAGNOSTIC_FINISHING_TIMEOUT", 60))
# TODO (tleontovich) Enable to True after 2.30 PRODX-48972
MOSK_CHILD_RUN_DIAGNOSTIC_CHECK = get_var_as_bool('MOSK_CHILD_RUN_DIAGNOSTIC_CHECK', False)

# Desired runtime, temporary parameter for checkin migration to containerd during 2.27 -> 2.28 -> 2.29
# TODO: gvlasov_cleanup_runtime
DESIRED_RUNTIME = get_var("DESIRED_RUNTIME", "")
# si_tests/managers/runtime_manager.py::partial_migrate timeout parameter
WAIT_RUNTIME_MIGRATION_STARTED_TIMEOUT = int(get_var("WAIT_RUNTIME_MIGRATION_STARTED_TIMEOUT", 600))

LMA_DOWNTIME_USER = get_var("LMA_DOWNTIME_USER", 'writer')
PROMETHEUS_DOWNTIME_CHECK = get_var_as_bool('PROMETHEUS_DOWNTIME_CHECK', False)
ALERTA_DOWNTIME_CHECK = get_var_as_bool('ALERTA_DOWNTIME_CHECK', False)
GRAFANA_DOWNTIME_CHECK = get_var_as_bool('GRAFANA_DOWNTIME_CHECK', False)
ALERTMANAGER_DOWNTIME_CHECK = get_var_as_bool('ALERTMANAGER_DOWNTIME_CHECK', False)
KIBANA_DOWNTIME_CHECK = get_var_as_bool('KIBANA_DOWNTIME_CHECK', False)
KEYCLOAK_DOWNTIME_CHECK = get_var_as_bool('KEYCLOAK_DOWNTIME_CHECK', False)
FORCE_LOAD_TOOLS_CONTAINER_CLEANUP = get_var_as_bool('FORCE_LOAD_TOOLS_CONTAINER_CLEANUP', True)

# Create BMH objects using BareMetalHostInventory instead of BareMetalHost
USE_BMH_INVENTORY = get_var_as_bool("USE_BMH_INVENTORY", False)
# Check that all related BMH Inventory objects are present.
# True: fail the test if some BMHI is missing
# False: print warning into logs, don't fail the test. Default till MCC 2.29.0
CHECK_BMH_INVENTORY = get_var_as_bool("CHECK_BMH_INVENTORY", False)

# Enable create_hoc_before_lcm_and_delete_after() fixture which will be executed for all LCM tests
CREATE_HOC_BEFORE_LCM_AND_DELETE_AFTER = get_var_as_bool("CREATE_HOC_BEFORE_LCM_AND_DELETE_AFTER", False)

# If set to False, machine pauses must be disabled during creation without any additional checks
MACHINE_PAUSE_DURING_CREATION_ENABLED = get_var_as_bool('MACHINE_PAUSE_DURING_CREATION_ENABLED', False)
SKIP_TO_CHECK_PRODX_51933 = get_var_as_bool('SKIP_TO_CHECK_PRODX_51933', False)

SI_IGNORE_UPDATE_PLAN_ESTIMATED_STEP_TIMEOUT_FAIL = get_var_as_bool(
    'SI_IGNORE_UPDATE_PLAN_ESTIMATED_STEP_TIMEOUT_FAIL',
    False)
