#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
import csv
import io

from exec_helpers import exec_result, ExecHelperTimeoutError
from retry import retry
from urllib.parse import urlparse

from si_tests import logger, settings
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.managers.bootstrap_manager import BootstrapManager
from si_tests.utils import waiters, utils

LOG = logger.logger

# workaround to use curl on offline mode
CURL_CMD = "curl --noproxy '*' "


class LoadToolsException(Exception):

    def __init__(self, message: str, result: exec_result.ExecResult):
        msg = f"\n{message}\nCommand: {result.cmd}\nFailed with error:\n{result.stderr_str}"
        super().__init__(msg)


class LoadToolsManager:

    def __init__(self, bootstrap_manager: BootstrapManager, os_manager: OpenStackManager,
                 refapp_address: str = '', verbose: bool = False):
        self.__verbose = verbose
        self.__loadtools_url = None
        self.__container_name = None
        self.bootstrap_manager = bootstrap_manager
        self.os_manager = os_manager
        self.remote = bootstrap_manager.remote_seed()
        self.remote.sudo_mode = True
        self.refapp_url = refapp_address

    def check_loadtools_available(self, run_refapp_smoke=False, loadtools_url=None):
        if not loadtools_url:
            loadtools_url = self.loadtools_url
        cmd_healthcheck = CURL_CMD + "--request GET '{}'".format(loadtools_url)
        healthcheck_result = self.remote.execute(verbose=self.__verbose, command=cmd_healthcheck, timeout=30)

        if healthcheck_result.exit_code == 0:
            if run_refapp_smoke:
                self.run_smoke()
            return True
        else:
            LOG.info(f"Loadtools is not ready yet. CMD: "
                     f"{cmd_healthcheck}:\nResult:\n{healthcheck_result.stderr_brief}")
            return False

    def wait_loadtools_available(self, run_refapp_smoke=False, loadtools_url=None, interval=10, timeout=300):
        LOG.info("Waiting loadtools ready")
        waiters.wait(lambda: self.check_loadtools_available(
            run_refapp_smoke=run_refapp_smoke,
            loadtools_url=loadtools_url), interval=interval, timeout=timeout)

    def run_loadtools_base(self, env_vars=None, locust_tags=None, docker_run_additional_option=None,
                           container_name_prefix=None, run_class=None):

        self.__container_name = 'kaas-si-loadtools-' + utils.gen_random_string(4)
        si_loadtools_image_url = settings.SI_LOADTOOLS_DOCKER_IMAGE_URL
        if container_name_prefix:
            self.__container_name += container_name_prefix
        base_docker_cmd = "docker run -d "
        if not env_vars:
            env_vars = []
        if not locust_tags:
            locust_tags = []
        if docker_run_additional_option:
            base_docker_cmd += docker_run_additional_option
        if locust_tags:
            # To get expected format like '[dnsquery,refapp,openstack]'
            locust_tags_str = f"\'[{','.join(locust_tags)}]\'"
            env_vars.append(f"LOCUST_TAGS={locust_tags_str}")
        if run_class:
            env_vars.append(f"RUN_CLASS={run_class}")
        env_var = ' -e '.join(env_vars)
        if env_var:
            env_var = f"-e {env_var}"
        cmd = base_docker_cmd + "--name {} {} {} ".format(self.__container_name, env_var, si_loadtools_image_url)
        docker_result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=300)
        if docker_result.exit_code != 0:
            raise LoadToolsException("Error while starting docker container on seed node", docker_result)

        docker_host_cmd = "docker inspect -f '{{range.NetworkSettings.Networks}}{{.IPAddress}}{{end}}' " \
                          f"{self.__container_name}"
        docker_host = self.remote.execute(verbose=self.__verbose, command=docker_host_cmd, timeout=180)
        if docker_host.exit_code != 0:
            raise LoadToolsException("Error while docker inspecting", docker_host)

        self.loadtools_url = docker_host.stdout_str

        is_refapp = 'refapp' in locust_tags
        self.wait_loadtools_available(run_refapp_smoke=is_refapp)

        LOG.info(f"Loadtools container host: {self.loadtools_url}")
        return self.__container_name, self.loadtools_url

    @property
    def refapp_url(self):
        return self.__refapp_url

    @refapp_url.setter
    def refapp_url(self, address: str):
        self.__refapp_url = "http://" + address if not address.startswith("http") else address

    @property
    def default_user_count(self):
        users = 1
        if self.os_manager.is_dns_test_record_present:
            users += 1
        if settings.SI_LOADTOOLS_CHECK_KEYSTONE_WORKLOAD:
            users += 1
        if settings.SI_LOADTOOLS_CHECK_TF_API_WORKLOAD:
            users += 1
        return users

    def get_dns_config(self):
        if not self.os_manager.is_dns_test_record_present:
            return (None, None)
        nameserver = self.os_manager.get_powerdns_svc_ip()
        test_record = "test-record.test-zone.test"
        return (nameserver, test_record)

    @property
    def loadtools_url(self):
        return self.__loadtools_url

    @loadtools_url.setter
    def loadtools_url(self, address: str, port="8089"):
        address = address if not address.endswith("/") else address[:-1]
        url_scheme = "http://" + address if not address.startswith("http") else address
        port = f":{port}" if ":" not in address else ""
        self.__loadtools_url = url_scheme + port

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def run_loadtools_on_seed_node(self, run_class=None):
        """
        Run loadtools docker container on seed node
        Args:
            run_class: loadtools app test class to execute
                       Details: https://gerrit.mcp.mirantis.com/plugins/gitiles/kaas/si-loadtools#writing-a-locustfile

        Returns: None

        """
        LOG.info("Start loadtools docker container on seed node")
        additional_options = None
        dns_server, dns_record = self.get_dns_config()
        env_vars = []
        locust_tags = settings.SI_LOADTOOLS_DEFAULT_LOCUST_TAGS
        if run_class:
            env_vars.append(f"RUN_CLASS={run_class}")
        if dns_server:
            if 'dnsquery' not in locust_tags:
                locust_tags.append('dnsquery')
            env_vars.append(f"DNSQUERY_NAMESERVER={dns_server}")
            env_vars.append(f"DNSQUERY_TEST_RECORD={dns_record}")

        if settings.SI_LOADTOOLS_CHECK_KEYSTONE_WORKLOAD:
            if 'keystone' not in locust_tags:
                locust_tags.append('keystone')

        # Keystone endpoint entry in etc/hosts is required for both workloads (Keystone/TF)
        if settings.SI_LOADTOOLS_CHECK_KEYSTONE_WORKLOAD or settings.SI_LOADTOOLS_CHECK_TF_API_WORKLOAD:
            endpoint_url = self.os_manager.get_service_endpoint(
                service_name='keystone', interface='public').get('URL', '')
            openstack_ingress = self.os_manager.get_ingress_svc_external_ip()
            hostname = urlparse(endpoint_url).hostname
            additional_options = f"--add-host {hostname}:{openstack_ingress} "

        if settings.SI_LOADTOOLS_CHECK_TF_API_WORKLOAD:
            if 'tungstenfabric' not in locust_tags:
                locust_tags.append('tungstenfabric')
            endpoint_url = self.os_manager.get_service_endpoint(
                service_name='tungstenfabric', interface='public').get('URL', '')
            openstack_ingress = self.os_manager.get_ingress_svc_external_ip()
            hostname = urlparse(endpoint_url).hostname
            additional_options += f" --add-host {hostname}:{openstack_ingress} "

        self.run_loadtools_base(env_vars=env_vars, locust_tags=locust_tags,
                                docker_run_additional_option=additional_options,
                                run_class=run_class)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def run_loadtools_on_seed_node_mcc_only(self, tag, run_class=None):
        """
                Run loadtools docker container on seed node for MCC only services
                Args:
                    tag: tag of loadtools test to be run on seed node
                    run_class: Test class to execute

                Returns: None

                """
        LOG.info("Start loadtools docker container on seed node for mcc-only")
        env_vars = []
        locust_tags = []
        if run_class:
            env_vars.append(f"RUN_CLASS={run_class}")
        locust_tags.append(tag)
        self.run_loadtools_base(env_vars=env_vars, locust_tags=locust_tags, run_class=run_class)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def run_loadtools_for_keystone_check(self, cluster_name, cluster_namespace):
        """
        Run loadtools docker container on seed node to check mosk keystone service
        Args:
            cluster_name: name of the cluster to check keystone
            cluster_namespace: cluster namespace
        Returns: (spawned container name, loadtools url)
        """

        LOG.info("Start loadtools docker container on seed node to check keystone")
        endpoint_url = self.os_manager.get_service_endpoint(
            service_name='keystone', interface='public').get('URL', '')
        openstack_ingress = self.os_manager.get_ingress_svc_external_ip()
        hostname = urlparse(endpoint_url).hostname
        locust_tag = ['keystone']
        additional_options = f"--add-host {hostname}:{openstack_ingress} "
        container_name_prefix = f"-{cluster_name}" + f"-{cluster_namespace}"
        return self.run_loadtools_base(locust_tags=locust_tag, docker_run_additional_option=additional_options,
                                       container_name_prefix=container_name_prefix)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def start_swarm(self, user_count=None, spawn_rate=1):
        """
        Start swarm
        Args:
            user_count: Peak number of concurrent users, should be at least
                        the same number as number of applications that we test.
                        if not set autodetect.
            spawn_rate: Rate to spawn users at (users per second)

        Returns: None

        """
        LOG.info("Start swarm")
        user_count = user_count or self.default_user_count
        LOG.info(f"Target host: {self.refapp_url} user count: {user_count} spawn rate: {spawn_rate}")
        cmd = CURL_CMD + "--request POST '{}/swarm' --header 'accept: */*' " \
                         "--form 'user_count=\"{}\"' " \
                         "--form 'spawn_rate=\"{}\"' " \
                         "--form 'host=\"{}\"'".format(self.loadtools_url, user_count, spawn_rate, self.refapp_url)

        if settings.SI_LOADTOOLS_CHECK_KEYSTONE_WORKLOAD or settings.SI_LOADTOOLS_CHECK_TF_API_WORKLOAD:
            identity_username = settings.SI_LOADTOOLS_OS_ADMIN_USERNAME
            identity_password = settings.SI_LOADTOOLS_OS_ADMIN_PASSWORD
            self.os_manager.create_openstack_admin_user(username=identity_username, password=identity_password)
            endpoint_url = self.os_manager.get_service_endpoint(
                service_name='keystone', interface='public').get('URL', '') + 'v3'
            cmd += (f" --form 'os_username=\"{identity_username}\"' "
                    f"--form 'os_password=\"{identity_password}\"' "
                    f"--form 'os_auth_url=\"{endpoint_url}\"' ")
            if settings.SI_LOADTOOLS_CHECK_TF_API_WORKLOAD:
                tf_endpoint_url = self.os_manager.get_service_endpoint(
                    service_name='tungstenfabric', interface='public').get('URL', '')
                cmd += (f" --form 'os_tf_url=\"{tf_endpoint_url}\"' ")

        result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=180)
        if result.exit_code != 0:
            raise LoadToolsException("Error while starting swarm", result)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def start_swarm_for_lma(
            self, user_count=1, spawn_rate=1,
            keycloak_url=None, alerta_url=None,
            grafana_url=None, prometheus_url=None, iam_user=None,
            kibana_url=None, alertmanager_url=None,
            iam_password=None):
        """
        Start swarm
        Args:
            user_count: Peak number of concurrent users, should be at least
                        the same number as number of applications that we test.
                        if not set autodetect.
            keycloack_url: url of keycloack server.
            alerta_url: url of alerta server.
            grafana_url: url of grafana server.
            prometheus_url: url of prometheus server.
            kibana_url: url of kibana server.
            alertmanager_url: url of alertmanager server.
            iam_user: iam user.
            iam_password: password to iam user.
            spawn_rate: Rate to spawn users at (users per second)

        Returns: None

        """
        LOG.info("Start swarm")
        cmd = ''
        user_count = user_count
        cmd_base = CURL_CMD + "--request POST '{}/swarm' --header 'accept: */*' " \
                              "--form 'user_count=\"{}\"' " \
                              "--form 'spawn_rate=\"{}\"' " \
                              "--form 'iam_user=\"{}\"' " \
                              "--form 'iam_password=\"{}\"' " \
                              "--form 'keycloack_url=\"{}\"' ".format(self.loadtools_url,
                                                                      user_count,
                                                                      spawn_rate,
                                                                      iam_user,
                                                                      iam_password,
                                                                      keycloak_url)
        if alerta_url:
            cmd = cmd_base + "--form 'alerta_url=\"{}\"' ".format(alerta_url)
        if grafana_url:
            cmd = cmd_base + "--form 'grafana_url=\"{}\"' ".format(grafana_url)
        if prometheus_url:
            cmd = cmd_base + "--form 'prometheus_url=\"{}\"' ".format(prometheus_url)
        if kibana_url:
            cmd = cmd_base + "--form 'kibana_url=\"{}\"' ".format(kibana_url)
        if alertmanager_url:
            cmd = cmd_base + "--form 'alertmanager_url=\"{}\"' ".format(alertmanager_url)
        if not cmd:
            cmd = cmd_base
        LOG.info(f"Target host with cmd {cmd}")
        result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=180)
        if result.exit_code != 0:
            raise LoadToolsException("Error while starting swarm", result)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def start_swarm_for_keystone(self, user_count=None, spawn_rate=1, endpoint_ip=''):
        """
        Start swarm
        Args:
            user_count: Peak number of concurrent users, should be at least
                        the same number as number of applications that we test.
                        if not set autodetect.
            spawn_rate: Rate to spawn users at (users per second)
            endpoint_ip: Keystone endpoint ip
            hostname: endpoint host (e.g. keystone.it.just.works)
        Returns: None
        """
        LOG.info("Start swarm")
        user_count = user_count or self.default_user_count
        LOG.info(f"Target host: {endpoint_ip} user count: {user_count} spawn rate: {spawn_rate}")
        cmd = CURL_CMD + "--request POST '{}/swarm' --header 'accept: */*' " \
                         "--form 'user_count=\"{}\"' " \
                         "--form 'spawn_rate=\"{}\"' " \
                         "--form 'host=\"{}\"'".format(self.loadtools_url, user_count, spawn_rate, endpoint_ip)
        identity_username = settings.SI_LOADTOOLS_OS_ADMIN_USERNAME
        identity_password = settings.SI_LOADTOOLS_OS_ADMIN_PASSWORD
        self.os_manager.create_openstack_admin_user(username=identity_username, password=identity_password)
        endpoint_url = self.os_manager.get_service_endpoint(
            service_name='keystone', interface='public').get('URL', '') + 'v3'
        cmd += (f" --form 'os_username=\"{identity_username}\"' "
                f"--form 'os_password=\"{identity_password}\"' "
                f"--form 'os_auth_url=\"{endpoint_url}\"' ")
        result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=180)
        if result.exit_code != 0:
            raise LoadToolsException("Error while starting swarm", result)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def start_swarm_for_keycloak(
            self, user_count=1, spawn_rate=1,
            keycloak_url=None,
            iam_user=None,
            iam_password=None):
        """
        Start swarm
        Args:
            user_count: Peak number of concurrent users, should be at least
                        the same number as number of applications that we test.
                        if not set autodetect.
            keycloak_url: url of keycloak server.
            iam_user: iam user.
            iam_password: password to iam user.
            spawn_rate: Rate to spawn users at (users per second)

        Returns: None

        """
        LOG.info("Start swarm")
        user_count = user_count
        cmd = CURL_CMD + "--request POST '{}/swarm' --header 'accept: */*' " \
                         "--form 'user_count=\"{}\"' " \
                         "--form 'spawn_rate=\"{}\"' " \
                         "--form 'iam_user=\"{}\"' " \
                         "--form 'iam_password=\"{}\"' " \
                         "--form 'keycloack_url=\"{}\"' ".format(self.loadtools_url,
                                                                 user_count,
                                                                 spawn_rate,
                                                                 iam_user,
                                                                 iam_password,
                                                                 keycloak_url)

        LOG.info(f"Target host with cmd {cmd}")
        result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=180)
        if result.exit_code != 0:
            raise LoadToolsException("Error while starting swarm", result)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def stop_swarm(self, loadtools_url=None):
        """
        Stop swarm
        Returns: None
        """
        if not loadtools_url:
            loadtools_url = self.loadtools_url
        LOG.info("Going to stop swarm")
        cmd = CURL_CMD + "--request GET '{}/stop'  --max-time 120 ".format(loadtools_url)
        result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=180)
        if result.exit_code != 0:
            raise LoadToolsException("Error while stopping swarm", result)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def get_full_history(self, loadtools_url=None):
        """
        Get swarm full statistics
        Returns: yaml object
        """
        if not loadtools_url:
            loadtools_url = self.loadtools_url
        LOG.info("Get statistics")
        cmd = CURL_CMD + "--request GET '{}/stats/requests_full_history/csv'".format(loadtools_url)
        result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=600)
        if result.exit_code != 0:
            raise LoadToolsException("Error while receiving custom statistics", result)
        csv_reader = csv.DictReader(io.StringIO(result.stdout_str))
        data = []
        # NOTE(vsaienko): The data here may be huge, 4hours sets is about 800Mb of data.
        # Drop fields that we know we will not process later.
        fields_to_store = ["Name", "Timestamp", "Total Failure Count", "Total Request Count", "Type"]
        for row in csv_reader:
            data.append({k: v for k, v in row.items() if k in fields_to_store})
        return data

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def save_docker_logs(self, file_name, container_name=None):
        """
        Get logs from the loadtools docker container and save to artifacts
        Returns: None
        """
        if not container_name:
            container_name = self.__container_name
        LOG.info("Save logs from loadtools docker container on seed node")
        logs_cmd = f"docker logs {container_name} 2>&1"
        logs_str = self.remote.execute(verbose=self.__verbose, command=logs_cmd, timeout=600).stdout_str
        with open(f"{settings.ARTIFACTS_DIR}/{file_name}", 'w') as f:
            f.write(logs_str)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def exit(self, loadtools_url=None, container_name=None):
        """
        Shutdown container and waiting for it to disappear on seed node
        Returns: None
        """
        if not loadtools_url:
            loadtools_url = self.loadtools_url
        if not container_name:
            container_name = self.__container_name
        LOG.info("Stop loadtools docker container on seed node")
        cmd = CURL_CMD + "--request GET '{}/exit'  --max-time 120 ".format(loadtools_url)
        result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=300)
        if result.exit_code != 0:
            raise LoadToolsException("Error while shutting down", result)

        def _is_container_delete():
            ps_cmd = "docker ps --format '{{json .Names}}'"
            existing_containers = self.remote.execute(verbose=self.__verbose, command=ps_cmd, timeout=180).stdout_str
            LOG.info(f"Existing containers: {existing_containers}")
            return container_name not in existing_containers

        def _is_container_cleaned_up():
            ps_cmd = "docker ps -a --format '{{json .Names}}'"
            existing_containers = self.remote.execute(verbose=self.__verbose, command=ps_cmd, timeout=180).stdout_str
            LOG.info(f"Existing containers: {existing_containers}")
            return container_name not in existing_containers

        LOG.info(f"Wait until container '{container_name}' to be removed")
        waiters.wait(_is_container_delete,
                     interval=30,
                     timeout_msg=f"Docker container {container_name} delete timeout")
        if settings.FORCE_LOAD_TOOLS_CONTAINER_CLEANUP:
            rm_cmd = f"docker rm -f {container_name}"
            result = self.remote.execute(verbose=self.__verbose, command=rm_cmd, timeout=180)
            if result.exit_code != 0:
                LOG.info(f'Cmd for removal container returns {result.exit_code} exit code')
            waiters.wait(_is_container_cleaned_up,
                         interval=30,
                         timeout_msg=f"Docker container {container_name} removal timeout")

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def receive_report(self, file_name='locust_report.html'):
        """
        Receive swarm report in HTML format
        Args:
            file_name: the report file name to store in artifacts

        Returns: None

        """
        LOG.info("Get load html report")
        cmd = CURL_CMD + "--request GET '{}/stats/report?download=1'".format(self.loadtools_url)
        result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=300)
        if result.exit_code != 0:
            raise LoadToolsException("Error while receiving report", result)
        with open(f"{settings.ARTIFACTS_DIR}/{file_name}", 'w') as f:
            f.write(result.stdout_str)

    @retry((LoadToolsException, ExecHelperTimeoutError), delay=15, tries=3, logger=LOG)
    def run_smoke(self):
        """
        Simple curl call to RefApp to verify is it accessible from host and alive

        Returns: None

        """
        LOG.info("Smoke call to RefApp to verify is it accessible and alive")
        cmd = CURL_CMD + '--request GET -s -o /dev/null -I -w "%{http_code}" ' + self.refapp_url
        result = self.remote.execute(verbose=self.__verbose, command=cmd, timeout=60)
        if result.exit_code != 0:
            raise LoadToolsException("Error while receiving response", result)
        elif result.stdout_str != "200":
            raise LoadToolsException(f"Status code '{result.stdout_str}' not as expected '200'. Smoke test failed",
                                     result)
