from si_tests import logger
from si_tests.clients.iam import keycloak_client
from si_tests import settings
from si_tests.utils import utils, waiters

LOG = logger.logger


class KeycloakManager(object):
    """Keycloak manager"""

    def __init__(self, manager):
        self._manager = manager
        self.__keycloak = None
        self.__cluster = None

    @property
    def manager(self):
        return self._manager

    @property
    def cluster(self):
        if not self.__cluster:
            self.__cluster = self.manager.get_mgmt_cluster()
        return self.__cluster

    @property
    def keycloak_adm(self):
        if not self.__keycloak:
            kip = self.manager.get_keycloak_ip()
            pwd = self.manager.get_secret_data('iam-api-secrets',
                                               'kaas', 'keycloak_password')
            return keycloak_client.KeycloakAdminClient(ip=kip,
                                                       user=settings.KEYCLOAK_USER,
                                                       password=pwd)

    def iamuser_exists(self, name):
        users = self.manager.get_iamusers()
        user = [x for x in users if x.name == name]
        return True if user else False

    def wait_iamuser_created(self, iamusername, timeout=300, interval=20):
        """Wait for iamuser to be created.

        Creating user is possible via keycloak_admin_client. But iamuser as resource will be created by iam-api when
        use in keycloak will be created.

        :param iamusername: {username}-{uid[:8]} where username - it's username in Keycloak, uid -
        it's uid of username in keycloak
        :param timeout
        :param interval
        """

        LOG.info(f"Waiting for iamuser {iamusername} to be created")
        waiters.wait(
            lambda: self.iamuser_exists(iamusername), timeout=timeout, interval=interval,
            timeout_msg="Timeout waiting for iamuser to be created")
        LOG.info(f"Iamuser {iamusername} created")

    def wait_iamuser_deleted(self, iamusername, timeout, interval):
        """Wait for iamuser to be deleted.

        :return:
        """
        LOG.info(f"Waiting for iamuser {iamusername} to be deleted")
        waiters.wait(
            lambda: not self.iamuser_exists(iamusername), timeout=timeout, interval=interval,
            timeout_msg="Timeout waiting for iamuser to be created")
        LOG.info(f"Iamuser {iamusername} deleted")

    def create_user_and_wait(self, username, password, email, timeout=300, interval=20):
        """Create user via Keycloak client and wait it creation as iamuser resource in k8s

        :param username:
        :param password:
        :param email:
        :param timeout:
        :param interval:
        :return: IAMUser
        """
        uid = self.keycloak_adm.user_create(username=username,
                                            password=password,
                                            email=email)
        iamusername = f"{username}-{uid[:8]}"
        self.wait_iamuser_created(iamusername, timeout=timeout, interval=interval)
        iamuser = self.manager.get_iamuser(iamusername)
        return iamuser, iamusername

    def delete_user_and_wait(self, username, timeout=300, interval=20):
        """Create user via Keycloak client and wait it creation as iamuser resource in k8s

        :param username:
        :param timeout:
        :param interval:
        """
        uid = self.keycloak_adm.get_id_by_username(username)
        self.keycloak_adm.user_delete(username=username)
        iamusername = f"{username}-{uid[:8]}"
        self.wait_iamuser_deleted(iamusername, timeout=timeout, interval=interval)

    def configure_keyclock_mariadb_backup(self, enabled=True,
                                          backup_type='full',
                                          backup_timeout=21600,
                                          allow_unsafe_backup=False,
                                          backups_to_keep=3,
                                          full_backup_cycle=604800,
                                          schedule_time='0 0 * * *'):
        """Configure MariaDB backups of keycloak
    https://docs.mirantis.com/container-cloud/latest/operations-guide/manage-mgmt/backup-maria/conf-maria-backup.html

        :param enabled: Enabled keycloak backups or not
        :param backup_type: full or incremental
        :param backup_timeout: backup timeout
        :param allow_unsafe_backup: Allow backup if cluster is degraded
        :param backups_to_keep: How much backups saved
        :param full_backup_cycle: Time in seconds between full backups. Will be used only if incremental backup set
        :param schedule_time: Cron-type cycle schedule
        :return:
        """
        LOG.info('Will enable MCC MariaDB backups of Keycloak')
        mariadb_chunk = {
            'mariadb': {
                'conf': {
                    'phy_backup': {
                        'enabled': enabled,
                        'backup_type': backup_type,
                        'backup_timeout': backup_timeout,
                        'allow_unsafe_backup': allow_unsafe_backup,
                        'backups_to_keep': backups_to_keep,
                        'full_backup_cycle': full_backup_cycle,
                        'schedule_time': schedule_time
                    }
                }
            }
        }
        iam_with_backup = {
            'name': 'iam',
            'values': {
                'keycloak': mariadb_chunk
            }
        }

        mgmt = (self.cluster.data['spec'].get('providerSpec', {}).get('value', {}).get('kaas', {}).
                get('management', {}))
        LOG.info(f">>> Management spec before changes : {mgmt}")
        if not mgmt.get('helmReleases', []):
            mgmt['helmReleases'] = []
            mgmt['helmReleases'].append(iam_with_backup)
        else:
            iam = next((x for x in mgmt['helmReleases'] if x['name'] == 'iam'), None)
            if not iam:
                mgmt['helmReleases'].append(iam_with_backup)
            else:
                # Assuming if iam is present - values also there. owherwise - why is iam present, right?
                keycloak = iam['values'].setdefault('keycloak', {})
                mariadb = keycloak.setdefault('mariadb', {})
                conf = mariadb.setdefault('conf', {})

                if conf.get('phy_backup'):
                    LOG.warning("MariaDB Keycloak backup settings will be overwritten!")
                conf['phy_backup'] = mariadb_chunk['mariadb']['conf']['phy_backup']
        LOG.info('>>> Will apply this this management spec for cluster')
        LOG.info(mgmt)
        spec = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'kaas': {
                            'management': mgmt,
                        }
                    }
                }
            }
        }
        self.cluster.patch(body=spec)

    def restore_mariadb_keycloak_backup(self, backup_name, replica_restore_timeout=7200):
        """Restore MariaDB Keycloak backup by given backup name
    https://docs.mirantis.com/container-cloud/latest/operations-guide/manage-mgmt/backup-maria/restore-maria-backup.html

        :param backup_name: Backup name
        :param replica_restore_timeout: Optional. Replica restore timeout
        :return:
        """
        LOG.info('Will configure restore MCC MariaDB backup of Keycloak')
        mariadb_restore_chunk = {
            'mariadb': {
                'manifests': {
                    'job_mariadb_phy_restore': True
                },
                'conf': {
                    'phy_restore': {
                        'backup_name': backup_name,
                        'replica_restore_timeout': replica_restore_timeout
                    }
                }
            }
        }
        # If we gonna restore smth - we should have already configured backup. Otherwise, it's possible,
        # but kinda strange and not the case.
        mgmt = (self.cluster.data['spec'].get('providerSpec', {}).get('value', {}).get('kaas', {}).
                get('management', {}))
        LOG.info(f">>> Management spec before changes : {mgmt}")
        iam = next((x for x in mgmt['helmReleases'] if x['name'] == 'iam'), None)
        iam['values']['keycloak']['mariadb']['manifests'] = mariadb_restore_chunk['mariadb']['manifests']
        iam['values']['keycloak']['mariadb']['conf']['phy_restore'] = (
            mariadb_restore_chunk)['mariadb']['conf']['phy_restore']
        LOG.info('>>> Will apply this this management spec for cluster')
        LOG.info(mgmt)
        spec = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'kaas': {
                            'management': mgmt,
                        }
                    }
                }
            }
        }
        self.cluster.patch(body=spec)

    def reset_mcc_mariadb_backup_settings(self):
        """Reset MariaDB backup settings.
        Default MCC backup settings will be restored. Any values of MariaDB Keycloak backup will be cleared.

        :return:
        """
        changes_made = False
        LOG.info('Will reset MCC MariaDB backup settings to default')
        mgmt = (self.cluster.data['spec'].get('providerSpec', {}).get('value', {}).get('kaas', {}).
                get('management', {}))
        LOG.info(f">>> Management spec before changes : {mgmt}")
        iam = next((x for x in mgmt['helmReleases'] if x['name'] == 'iam'), None)
        if not iam:
            LOG.info('Nothing to reset')
            return changes_made

        keycloak = iam['values'].get('keycloak', {})
        mariadb = keycloak.get('mariadb', {})

        if not mariadb:
            LOG.info('Nothing to reset')
            return changes_made

        manifests = mariadb.get('manifests', {})
        conf = mariadb.get('conf', {})

        if not conf:
            LOG.info('Nothing to reset')
            return changes_made

        if manifests.get('job_mariadb_phy_restore', {}):
            manifests['job_mariadb_phy_restore'] = {}
            changes_made = True
        if conf.get('phy_backup', {}):
            conf['phy_backup'] = {}
            changes_made = True
        if conf.get('phy_restore', {}):
            conf['phy_restore'] = {}
            changes_made = True

        LOG.info('>>> Will apply this this management spec for cluster')
        LOG.info(mgmt)
        spec = {
            'spec': {
                'providerSpec': {
                    'value': {
                        'kaas': {
                            'management': mgmt,
                        }
                    }
                }
            }
        }
        self.cluster.patch(body=spec)
        return changes_made

    def mariadb_volume_mgmt(self,
                            namespace,
                            image,
                            cmd='ls /var/backup/base',
                            pvc_name='mariadb-phy-backup-data',
                            timeout=300):
        """Exec commands related to mariadb volume folder. (Check backups, cleanup backups, etc)

        :param namespace: namespace where MariaDB lives
        :param image: MariaDB image
        :param cmd: cmd (usual ls to base but can be changed to 'tree' to parse incremental ones)
        :param pvc_name: can be overriden if set in cluster object
        :param timeout: timeout of pod execution
        :return:
        """
        pod_name = f"check-backup-helper-{utils.gen_random_string(6)}"
        body = {
            "apiVersion": "v1",
            "kind": "Pod",
            "metadata": {
                "name": pod_name,
                "namespace": namespace,
                "labels": {
                    "application": "si-tests"
                }
            },
            "spec": {
                "containers": [
                    {
                        "name": "helper",
                        "securityContext": {
                            "allowPrivilegeEscalation": False,
                            "runAsUser": 0,
                            "readOnlyRootFilesystem": True
                        },
                        "command": [
                            "/bin/bash",
                            "-c",
                            f"{cmd}"
                        ],
                        "image": image,
                        "imagePullPolicy": "IfNotPresent",
                        "volumeMounts": [
                            {
                                "name": "pod-tmp",
                                "mountPath": "/tmp"
                            },
                            {
                                "mountPath": "/var/backup",
                                "name": "mysql-backup"
                            }
                        ]
                    }
                ],
                "restartPolicy": "Never",
                "volumes": [
                    {
                        "name": "pod-tmp",
                        "emptyDir": {}
                    },
                    {
                        "name": "mariadb-secrets",
                        "secret": {
                            "secretName": "mariadb-secrets",
                            "defaultMode": 292
                        }
                    },
                    {
                        "name": "mariadb-bin",
                        "configMap": {
                            "name": "mariadb-bin",
                            "defaultMode": 365
                        }
                    },
                    {
                        "name": "mysql-backup",
                        "persistentVolumeClaim": {
                            "claimName": pvc_name
                        }
                    }
                ]
            }
        }
        pod = self.cluster.k8sclient.pods.create(name=pod_name, namespace=namespace, body=body)
        try:
            pod.wait_phase(['Succeeded', 'Failed'],
                           timeout=timeout, interval=1)
            logs = pod.get_logs()
            return logs
        except Exception as e:
            LOG.error(f"Error while running pod {pod_name}:\n{e}")
            raise e
        finally:
            pod.delete()
