#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import cachetools.func as cachetools_func
from si_tests import logger
from si_tests import settings

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from si_tests.managers.kaas_manager import Cluster


LOG = logger.logger


class HAManager(object):
    """HA manager"""

    def __init__(self, cluster: "Cluster"):
        self._cluster: "Cluster" = cluster

    @property
    def cluster(self) -> "Cluster":
        return self._cluster

    @property
    @cachetools_func.ttl_cache(ttl=300)
    def k8sclient(self):
        return self._cluster.k8sclient

    def pids_of_process_in_container_for_containerd(self, machine=None, cont_id=None,
                                                    process_name=None, cont_namespace="k8s.io"):
        """ Pause process inside container for containerd runtime by process name
        """
        assert machine, "No machine for pids_of_process_in_container_for_containerd function"
        assert cont_id, "No container id for pids_of_process_in_container_for_containerd function"
        assert process_name, "No process name for pids_of_process_in_container_for_containerd function"
        assert cont_namespace, "No container namespace for pids_of_process_in_container_for_containerd function"

        LOG.info(f"Accessing {machine.name}")
        ctr_container_id = machine._run_cmd(f"sudo ctr --namespace {cont_namespace} container ls | "
                                            f"grep {cont_id} | cut -d ' ' -f 1",
                                            ssh_key=settings.HA_TEST_PRIVATE_KEY_FILE).stdout_str
        cont_pids = machine._run_cmd(f"sudo ctr --namespace {cont_namespace} tasks ps {ctr_container_id} | "
                                     f"cut -d ' ' -f 1 | grep -v PID",
                                     ssh_key=settings.HA_TEST_PRIVATE_KEY_FILE).stdout_str
        pids = ""
        for pid in cont_pids.split("\n"):
            result = machine._run_cmd(
                f"ps -p {pid} | grep -w {process_name}",
                check_exit_code=False,
                ssh_key=settings.HA_TEST_PRIVATE_KEY_FILE).stdout_str
            if result:
                pids = pids + " " + pid

        assert pids, f"No PIDs for {process_name} process in container with id {ctr_container_id}"
        LOG.info(f"Detected PIDs: {pids}")

        return pids
