import exec_helpers
import yaml

from concurrent import futures
from paramiko.ssh_exception import NoValidConnectionsError
from si_tests import settings
from si_tests import logger
from si_tests.deployments.utils import (file_utils, commons)
from si_tests.managers.kaas_manager import Manager
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.managers.ovn_migration_manager import OvsOvnMigrationManager
from si_tests.utils import utils, templates
from si_tests.utils.utils import Provider
from si_tests.managers.openstack_client_manager import OpenStackClientManager
from si_tests.utils.utils import verify, check_test_result
from si_tests.deployments.deploy_workloads import PerNodeWorkload
from retry import retry
from si_tests.utils.workload_report import BMWorkloadDowntimeReport, VMWorkloadDowntimeReport
from si_tests.utils.templates import render_template

import multiprocessing
import pytest
import re
import os.path

LOG = logger.logger

TEMPLATE_PATH = file_utils.join(os.path.dirname(os.path.abspath(__file__)),
                                "../tests/lcm/templates/heat_stack_host.yaml")


@pytest.fixture(scope='session')
def ssh_keys():
    return utils.generate_keys()


@pytest.fixture(scope='session')
def autoset_migration_mode(kaas_manager):
    yield

    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    if not re.search(r"mos.?-.*", cluster.clusterrelease_version):
        msg = "Skipping annotation configuration for non-Openstack clusters"
        LOG.info(msg)
        return
    LOG.info("Setting migration mode for different compute types")
    dpdk_machines = [m for m in cluster.get_machines(machine_type="worker") if
                     m.get_k8s_node().data.get('metadata', {}).get('labels', {}).get(
                         'openstack-compute-node-dpdk', '') == 'enabled']
    nova_machines = [m for m in cluster.get_machines(machine_type="worker") if
                     m.get_k8s_node().data.get('metadata', {}).get('labels', {}).get(
                         'openstack-compute-node', '') == 'enabled']
    nova_machines = [m for m in nova_machines if m not in dpdk_machines]
    if dpdk_machines:
        if len(dpdk_machines) == 1:
            LOG.info("Only 1 dpdk compute found. Setting migration mode to SKIP")
            dpdk_machines[0].add_k8s_node_annotations(
                annotations={'openstack.lcm.mirantis.com/instance_migration_mode': 'skip'})
        else:
            LOG.info("More then 1 dpdk compute found. Setting migration mode to LIVE")
            for machine in dpdk_machines:
                machine.add_k8s_node_annotations(
                    annotations={'openstack.lcm.mirantis.com/instance_migration_mode': 'live'})
    else:
        LOG.info("Machines with label openstack-compute-node-dpdk are not found. Skipping")

    if nova_machines:
        if len(nova_machines) == 1:
            LOG.info("Only 1 compute found in nova zone. Setting migration mode to SKIP")
            nova_machines[0].add_k8s_node_annotations(
                annotations={'openstack.lcm.mirantis.com/instance_migration_mode': 'skip'})
        else:
            LOG.info("More then 1 compute found in nova zone. Setting migration mode to LIVE")
            for machine in nova_machines:
                machine.add_k8s_node_annotations(
                    annotations={'openstack.lcm.mirantis.com/instance_migration_mode': 'live'})
    else:
        LOG.info("Machines with label openstack-compute-node are not found")


def verify_stack_workload(openstack_client_manager, stack_name):
    ssh_access_key = openstack_client_manager.stack.output_show(
        [stack_name, "heat_key", "--column", "output_value"])["output_value"]
    pkey = utils.get_rsa_key(ssh_access_key)

    @retry((TimeoutError, NoValidConnectionsError, ConnectionResetError), tries=5, delay=30)
    def _ssh_instance(ip):
        LOG.info(f"Attemp to connect to VM with ip: {ip} in stack {stack_name}")
        auth = exec_helpers.SSHAuth(username="cirros", password='', key=pkey)
        ssh = exec_helpers.SSHClient(
            host=ip, port=22, auth=auth)
        ssh.logger.addHandler(logger.console)
        ssh.sudo_mode = True
        return ssh

    servers = [x["resource_name"] for x in
               openstack_client_manager.stack.resource_list([stack_name, "--filter", "type=OS::Nova::Server"],
                                                            yaml_output=True)]
    assert servers, f"No servers existed for stack {stack_name}"

    def _get_fip_only(server):
        server_info = openstack_client_manager.stack.resource_show([stack_name, server])
        network = list(server_info.get('attributes').get('addresses').keys())[0]
        for address in server_info.get('attributes').get('addresses').get(network):
            addr_type = address['OS-EXT-IPS:type']
            if addr_type == 'floating':
                LOG.info("Verify instance {0} with floating ip {1}".format(
                    server_info.get('attributes').get('OS-EXT-SRV-ATTR:hostname'), address['addr']))
                return address['addr']

    failed_results = {}
    ips = []
    with futures.ThreadPoolExecutor() as executor:
        jobs = {executor.submit(_get_fip_only, server): server for server in servers}
        for future in futures.as_completed(jobs):
            server = jobs[future]
            try:
                data = future.result()
                ips.append(data)
            except Exception as e:
                raise ValueError(f"Getting server {server} IP finished with exception: {e}")
    for ip in ips:
        ssh = _ssh_instance(ip)
        expected_string = utils.gen_random_string(10)
        ssh.check_call('echo {expected_string} > test.txt'.format(expected_string=expected_string))
        res = ssh.check_call('cat test.txt')
        if res.stdout_brief != expected_string:
            failed_results[ip] = {'actual': res.stdout_brief, 'expected': expected_string}
            LOG.error(f"Instance with IP {ip} failed to check")

    assert not failed_results, f"Some servers return wrong response in stack {stack_name}:\n{yaml.dump(failed_results)}"
    LOG.info("All instances checks passed")


def _check_stacks(stacks_names, child_cluster, openstack_client_manager: OpenStackClientManager):
    verify_after_process = {}

    # Create and run processes for stack creation
    for stack in stacks_names:
        verify_after_process[stack] = {'process': multiprocessing.Process(target=openstack_client_manager.check_stack,
                                                                          args=(stack,), name='check_stack', ),
                                       'exit_code': None}

        verify_after_process[stack]['process'].start()

    # Wait for all processes finished
    for k, v in verify_after_process.items():
        v['process'].join()
        v['exit_code'] = v['process'].exitcode

    # Check exit codes
    assert all([v['exit_code'] == 0 for k, v in verify_after_process.items()]), (
        f"Next stacks were not checked succesfully:\n"
        f"{[k for k, v in verify_after_process.items() if v['exit_code'] != 0]}")
    if not child_cluster.tf_enabled() or (child_cluster.tf_enabled() and settings.RUN_ON_REMOTE):
        LOG.info("Create and run processes for verifying stacks workloads")
        for stack in stacks_names:
            verify_after_process[stack] = {'process': multiprocessing.Process(target=verify_stack_workload,
                                                                              args=(openstack_client_manager, stack,),
                                                                              name='verify_stack',), 'exit_code': None}
            verify_after_process[stack]['process'].start()

        # Wait for all processes finished
        for k, v in verify_after_process.items():
            v['process'].join()
            v['exit_code'] = v['process'].exitcode

        # Check exit codes
        assert all([v['exit_code'] == 0 for k, v in verify_after_process.items()]), (
            f"Next stacks were not checked succesfully:\n"
            f"{[k for k, v in verify_after_process.items() if v['exit_code'] != 0]}")
    else:
        LOG.info("Skip checking stacks workloads when TF and not RUN_ON_REMOTE")


@pytest.fixture(scope='session')
def prepare_heat_stacks_before_test(kaas_manager, request, ssh_keys):
    """Create heat stacks on each OpenStack compute host before tests, and remove after tests"""

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)
    cr_before = child_cluster.clusterrelease_version

    if "mos" not in cr_before or child_cluster.provider is not Provider.baremetal:
        LOG.info("Skipping 'prepare_heat_stacks_before_test' fixture because of not suitable cluster")
        yield None
        return

    child_kubeconfig_name, child_kubeconfig = child_cluster.get_kubeconfig_from_secret()
    with open('child_conf', 'w') as f:
        f.write(child_kubeconfig)

    openstack_client_manager = OpenStackClientManager(kubeconfig='child_conf')
    os_manager = OpenStackManager(kubeconfig='child_conf')
    # Check encrypted_volume type exists in osdpl and create stack with machine from encrypted volume if it exists
    if settings.OPENSTACK_ENCRYPTED_VOLUME:
        assert os_manager.is_encrypted_volume_type_described(), "Encrypted volume type WAS NOT found in osdpl, " \
                                                                "but settings.OPENSTACK_ENCRYPTED_VOLUME is enabled"
    template = templates.render_template(TEMPLATE_PATH, options={"ssh_public_key": ssh_keys["public"],
                                                                 "ssh_private_key": ssh_keys["private"]})
    heat_template = yaml.safe_load(template)
    with open(TEMPLATE_PATH, 'w') as f:
        f.write(yaml.dump(heat_template))

    # Create test stacks with instances on different hosts
    # Get compute hosts names:

    LOG.info("Get compute hosts names")
    hosts = openstack_client_manager.host.list([])
    LOG.info("Unset quotas for workloads")
    openstack_client_manager.quota.set(["--volumes", "-1", "--instances", "-1", "--cores", "-1",
                                        "--routers", "-1", settings.MOSK_WORKLOAD_PROJECT_NAME])
    stacks_data = {}
    create_stack_process = {}
    verify_stack_process = {}
    for hs in hosts:
        if hs.get('Service') == 'compute' and 'ironic' not in hs.get('Host Name'):
            hostname = hs['Host Name']
            node_labels = child_cluster.k8sclient.nodes.get(name=hostname).read().to_dict().\
                get('metadata', {}).get('labels', {})
            if node_labels.get('openstack-compute-node-dpdk', '') == 'enabled':
                # Create a flavor with huge pages
                LOG.info("Create a flavor with huge pages")
                flavor_name = "m1.tiny.dpdk-" + hostname
                flavor_params = ["--ram", "1024", "--disk", "5", "--vcpus", "2", "--property", "hw:mem_page_size=2048",
                                 flavor_name]
                keystone_client_pod = [
                    pod for pod in os_manager.get_os_pods() if
                    'keystone-client' in pod.name]
                assert len(keystone_client_pod) > 0, (
                    "No pods found with keystone-client name prefix")
                keystone_client_pod = keystone_client_pod[0]
                flavors_cmd = [
                    '/bin/sh', '-c',
                    'PYTHONWARNINGS=ignore::UserWarning openstack flavor list --all --long -f json']
                flavors = yaml.safe_load(
                    keystone_client_pod.exec(flavors_cmd))
                current_flavor = [flavor for flavor in flavors if flavor.get("Name") == flavor_name]
                if current_flavor:
                    LOG.info(f'Flavor with name: {flavor_name} already exists')
                else:
                    LOG.info(f'Creating new flavor with name: {flavor_name}')
                    openstack_client_manager.flavor.create(flavor_params)

                stack_params = {"host_name": hs['Zone'] + ":" + hostname, "flavor1": "m1.tiny.dpdk-" + hostname,
                                "key_name": "keypair-" + hostname, "image_name_1":
                                openstack_client_manager.cirros_image_name}
                stacks_data["stack-" + hostname] = stack_params

            elif node_labels.get('openstack-compute-node', '') == 'enabled':
                stack_params = {"host_name": hs['Zone'] + ":" + hostname, "key_name": "keypair-" + hostname,
                                "image_name_1": openstack_client_manager.cirros_image_name}
                stacks_data["stack-" + hostname] = stack_params

    # Create and start processes for stacks creation
    for stack_name, stack_data in stacks_data.items():
        create_stack_process[stack_name] = {'process': multiprocessing.Process(
            target=openstack_client_manager.ensure_stack,
            args=(request, stack_name, TEMPLATE_PATH, 'child_conf', stack_data, False),
            name='create_stack', ), 'exit_code': None}
        create_stack_process[stack_name]['process'].start()

    # Waiting for all processes finished
    for k, v in create_stack_process.items():
        v['process'].join()
        v['exit_code'] = v['process'].exitcode

    # Check processes exit codes
    assert all([v['exit_code'] == 0 for k, v in create_stack_process.items()]), (
        f"Next stacks were not created succesfully:\n"
        f"{[k for k, v in create_stack_process.items() if v['exit_code'] != 0]}")

    _check_stacks(stacks_data.keys(), child_cluster, openstack_client_manager)

    yield stacks_data

    openstack_client_manager.reload_client()

    _check_stacks(stacks_data.keys(), child_cluster, openstack_client_manager)

    for stack_name, _ in stacks_data.items():
        LOG.info(f"Removing stack {stack_name}")
        verify_stack_process[stack_name] = multiprocessing.Process(target=openstack_client_manager.stack.delete,
                                                                   args=([stack_name, "-y", "--wait"],),
                                                                   name='delete_stack',)
        verify_stack_process[stack_name].start()
    for k, v in verify_stack_process.items():
        v.join()


@pytest.fixture(scope='function')
def check_heat_stacks_after_test(prepare_heat_stacks_before_test, kaas_manager, request):
    """Check stacks, created in the fixture 'prepare_heat_stacks_before_test', after each test

    Expects dictionary in 'prepare_heat_stacks_before_test', where .keys() are stacks names.
    If 'prepare_heat_stacks_before_test' is empty or None - assume that there is no checks to verify.
    """

    yield

    if not prepare_heat_stacks_before_test:
        LOG.info("Skipping 'check_heat_stacks_after_test' fixture because no stacks were created")
        return

    if check_test_result(request, ['skipped', 'failed']):
        LOG.warning("Test wasn't successful, so skip cluster condition check after test")
        return

    stacks_names = prepare_heat_stacks_before_test.keys()

    # check stacks after the test
    openstack_client_manager = OpenStackClientManager(kubeconfig='child_conf')

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)

    openstack_client_manager.reload_client()
    commons.LOG.info("Check heat stacks after test")
    _check_stacks(stacks_names, child_cluster, openstack_client_manager)


@pytest.fixture(scope='function')
def mos_workload_downtime_report(func_name, request):
    """Allows to create workload downtime report on MOSK baremetal and virtual ci labs

        The differences of MOSK virtual ci lab are:
        - no mgmt cluster is present
        - prometheus is set up without integration with IAM and without authorization
        These differences require to use separate class for downtime report
        When running on baremetal labs both KUBECONFIG_MGMT_PATH and KUBECONFIG_PATH
        should be passed to si tests.
        In order to downtime report can be skipped, @pytest.mark.skipif should be used,
        or similar solutions based on @pytest.mark functionality.
    """

    # Test can be and (should be) skipped by some fixture like @pytest.mark.skipif,
    # in that case downtime report will be skipped at the beginning
    if check_test_result(request, ['skipped']):
        LOG.warning("Test was skipped, so skip workload downtime report")
        yield
        return
    if not settings.MOSK_WORKLOAD_DOWNTIME_REPORT:
        LOG.info("MOSK_WORKLOAD_DOWNTIME_REPORT is disabled, skipping workload downtime report")
        yield
        return

    release_key = 'mos'

    if settings.KUBECONFIG_MGMT_PATH:
        kubeconfig_path = settings.KUBECONFIG_MGMT_PATH
    else:
        kubeconfig_path = settings.KUBECONFIG_PATH
    kaas_manager = Manager(kubeconfig=kubeconfig_path)
    if kaas_manager.api.kaas_kaasreleases.available:
        ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
        child_cluster = ns.get_cluster(settings.TARGET_CLUSTER)
        report = BMWorkloadDowntimeReport(func_name, release_key, child_cluster)
    else:
        LOG.info(f"Kubeconfig {kubeconfig_path} is not a Management cluster config")
        report = VMWorkloadDowntimeReport(func_name, release_key)
    report.set_up()
    yield
    # Test can be failed or skipped by pytest.skip() inside test,
    # in that case downtime report saving will be skipped after test is finished.
    if check_test_result(request, ['skipped', 'failed']):
        LOG.warning("Test wasn't successful, so skip downtime report saving after test")
        return
    report.save()


@pytest.fixture()
def check_pods_during_update(os_manager):
    """Checks whether pods were restarted during the update."""
    pods_should_not_be_restarted_name_patterns = settings.PODS_SHOULD_NOT_BE_RESTARTED_NAME_PATTERNS

    def get_list_of_pods(env_var_):
        pods = []
        for namespace, patterns in env_var_.items():
            for pattern in patterns:
                pods += os_manager.api.pods.list(namespace, pattern)
        pods = \
            [x for x in pods if not (
                    x.data.get('metadata', {}).get('owner_references', [{}]) and
                    x.data.get('metadata', {}).get('owner_references', [{}])[0].get('kind') == 'Job')]
        return pods

    if pods_should_not_be_restarted_name_patterns:
        pods_before_update = get_list_of_pods(pods_should_not_be_restarted_name_patterns)
    else:
        LOG.info("PODS_SHOULD_NOT_BE_RESTARTED_NAME_PATTERNS is absent, "
                 "skipping check that pods are same before and after the update")

    yield

    if pods_should_not_be_restarted_name_patterns:
        pods_after_update = get_list_of_pods(pods_should_not_be_restarted_name_patterns)
        verify(pods_after_update == pods_before_update,
               f"Pods before and after the update are different. {pods_before_update} != {pods_after_update}",
               f"Pods before and after the update are same. {pods_before_update} == {pods_after_update}")


@pytest.fixture(scope='function')
def check_ceph_keyrings(kaas_manager):
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    child_cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    if child_cluster.is_os_deployed() and child_cluster.is_ceph_deployed:
        kubectl_client = child_cluster.k8sclient
        secret_names = ['rook-ceph-client-cinder', 'rook-ceph-client-glance', 'rook-ceph-client-nova']
        namespace = settings.ROOK_CEPH_NS

        def check_secrets_present(client, secrets, ns):
            dict = []
            for secret in secrets:
                if client.secrets.present(namespace=ns, name=secret):
                    key = kubectl_client.secrets.get(namespace=namespace, name=secret).\
                        data.get('data', {}).get('adminKey', {})
                    dict.append({secret: key})
                else:
                    raise Exception(f"Secret {secret} not found")
            return dict

        LOG.info('Check that secrets exist')
        secret_dict = check_secrets_present(kubectl_client, secret_names, namespace)
        LOG.info(secret_dict)
        yield

        LOG.info('Check that secrets still exist')
        secret_dict_after_test = check_secrets_present(kubectl_client, secret_names, namespace)
        LOG.info(secret_dict_after_test)

        diff = [(secret_dict[i], secret_dict_after_test[i]) for i in range(
            len(secret_dict)) if secret_dict[i] != secret_dict_after_test[i]]

        assert len(diff) == 0, f"Some adminKey was changed during test: {diff}"

        LOG.info("Ceph keyring not changed. OK")
    else:
        LOG.info("Managed cluster without openstack - skip check")
        yield
        return


@pytest.fixture(scope='session')
def mos_per_node_workload(openstack_client_manager):

    per_node_workload = PerNodeWorkload(openstack_client_manager)
    per_node_workload.deploy()
    per_node_workload.check()
    yield per_node_workload
    per_node_workload.ostcm.reload_client()
    per_node_workload.check()
    per_node_workload.reboot_instance_with_volume()
    per_node_workload.collect_logs()
    per_node_workload.delete()


@pytest.fixture(scope='function')
def mos_per_node_workload_check_after_test(mos_per_node_workload, request):
    yield

    if check_test_result(request, ['skipped', 'failed']):
        LOG.warning("Test wasn't successful, so skip pernode workloads checking")
        return

    mos_per_node_workload.ostcm.reload_client()
    mos_per_node_workload.check()


@pytest.fixture(scope='session')
def mcc_per_node_workload(kaas_manager, request):

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)
    cr_before = child_cluster.clusterrelease_version

    if "mos" not in cr_before or child_cluster.provider is not Provider.baremetal:
        LOG.info("Skipping 'mcc_create_per_node_workload' fixture because of not suitable cluster")
        yield None
        return

    child_kubeconfig_name, child_kubeconfig = child_cluster.get_kubeconfig_from_secret()
    with open('child_conf', 'w') as f:
        f.write(child_kubeconfig)

    openstack_client_manager = OpenStackClientManager(kubeconfig='child_conf')

    per_node_workload = PerNodeWorkload(openstack_client_manager)
    per_node_workload.deploy()
    per_node_workload.check()
    yield per_node_workload
    per_node_workload.ostcm.reload_client()
    per_node_workload.check()
    per_node_workload.reboot_instance_with_volume()
    per_node_workload.collect_logs()
    per_node_workload.delete()


@pytest.fixture(scope='function')
def mcc_per_node_workload_check_after_test(mcc_per_node_workload, request):
    yield

    if not mcc_per_node_workload:
        LOG.info("Skipping 'mcc_per_node_workload_check_after_test' fixture because no stacks were created")
        return

    if check_test_result(request, ['skipped', 'failed']):
        LOG.warning("Test wasn't successful, so skip pernode workloads checking")
        return

    mcc_per_node_workload.ostcm.reload_client()
    mcc_per_node_workload.check()


@pytest.fixture(scope='session')
def network_backend(os_manager):
    if os_manager.is_tf_enabled:
        return 'tf'
    return 'ovs'


@pytest.fixture(scope='session')
def mariadb_backup_remote_sync(os_manager):
    if os_manager.is_db_backup_sync_remote_enabled():
        return 'enabled'
    return 'disabled'


@pytest.fixture(scope='function')
def skip_by_network_backend(request, network_backend):
    if request.node.get_closest_marker('network_backend'):
        if request.node.get_closest_marker('network_backend').args[0] != network_backend:
            pytest.skip(f"Skipped with network backend: {network_backend}")


@pytest.fixture(scope='function')
def skip_by_mariadb_backup_remote_sync(request, mariadb_backup_remote_sync):
    if request.node.get_closest_marker('mariadb_backup_remote_sync'):
        if request.node.get_closest_marker('mariadb_backup_remote_sync').args[0] != mariadb_backup_remote_sync:
            pytest.skip(f"Skipped as mariadb backup remote sync {mariadb_backup_remote_sync}")


@pytest.fixture(scope='function')
def tf_db_check_pod(tf_manager):
    basepath = os.path.dirname(os.path.abspath(__file__))
    pod_yaml = file_utils.join(basepath, "../templates/pods/tfdb_check_pod.yaml")

    pvc_name = tf_manager.get_db_pvc_name()
    tfdbbackup = tf_manager.tfdbbackup()
    image = tfdbbackup.data["spec"]["image"]

    template = render_template(pod_yaml, {'PVC_NAME': pvc_name, 'IMAGE': image})
    pod_data = yaml.load(template, Loader=yaml.SafeLoader)
    LOG.debug(pod_data)

    if tf_manager.is_tf_tfdbbackup_remote_enabled():
        remoteSync = tf_manager.tfoperator().data["spec"]["features"]["dbBackup"]["remoteSync"]

        pod_data["spec"]["containers"][0]['image'] = image
        pod_data["spec"]["containers"][0]['env'].extend([
            {
                "name": f"RCLONE_CONFIG_{tf_manager.rclone_remote.upper()}_TYPE",
                "value": remoteSync["type"],
            },
            {
                "name": f"RCLONE_CONFIG_{tf_manager.rclone_remote.upper()}_PROVIDER",
                "value": remoteSync["provider"],
            },
            {
                "name": f"RCLONE_CONFIG_{tf_manager.rclone_remote.upper()}_PATH",
                "value": remoteSync["path"],
            },
            {
                "name": f"RCLONE_CONFIG_{tf_manager.rclone_remote.upper()}_ENDPOINT",
                "value": remoteSync["endpoint"],
            },
            {
                "name": f"RCLONE_CONFIG_{tf_manager.rclone_remote.upper()}_ACCESS_KEY_ID",
                "valueFrom": {
                    "secretKeyRef": {
                        "name": remoteSync["secretData"]["secretName"],
                        "key": remoteSync["secretData"]["accessKeyName"],
                    }
                }
            },
            {
                "name": f"RCLONE_CONFIG_{tf_manager.rclone_remote.upper()}_SECRET_ACCESS_KEY",
                "valueFrom": {
                    "secretKeyRef": {
                        "name": remoteSync["secretData"]["secretName"],
                        "key": remoteSync["secretData"]["secretAccessKeyName"],
                    }
                }
            }
        ])

    pod = tf_manager.api.pods.create(namespace='tf', body=pod_data, log_body=False)
    pod.wait_phase(['Running'])

    yield pod

    pod.delete()


@pytest.fixture(scope='session')
def ovn_migration_manager(os_manager):
    return OvsOvnMigrationManager(os_manager)


@pytest.fixture(scope='session')
def ovn_migration_checker(ovn_migration_manager):
    yield
    try:
        ovn_migration_manager.check_state()
    finally:
        ovn_migration_manager.collect_logs()
