import time
from urllib.parse import urljoin

import requests
from requests.auth import AuthBase

from si_tests import logger
from si_tests.clients.iam.keycloak_client import KeycloakUserClient
from si_tests.utils.helpers import retry_429

LOG = logger.logger


class TokenAuth(AuthBase):
    def __init__(self, token, auth_scheme='Bearer'):
        self.token = token
        self.auth_scheme = auth_scheme

    def __call__(self, request):
        request.headers['Authorization'] = f'{self.auth_scheme} {self.token}'
        return request


class HttpClient(object):
    def __init__(self, base_url=None, user=None, password=None, verify=None):
        self.base_url = base_url
        self.kwargs = {}
        self.username = user
        self.password = password
        if verify is not None:
            self.kwargs.update({"verify": verify})
        if self.username and self.password:
            self.login()

    def login(self):
        self.kwargs.update({"auth": (self.username, self.password)})

    def set_base_url(self, base_url):
        self.base_url = base_url

    @retry_429(delay=40, tries=3)
    def request(self, url, method, headers=None, body=None,
                raise_on_error=True, **kwargs):
        LOG.debug(
            "Sending request to: {}, body: {}, headers: {}, kwargs: {}".format(
                url, body, headers, kwargs))

        kwargs.update(self.kwargs)
        r = requests.request(method, urljoin(self.base_url, url),
                             headers=headers, data=body, **kwargs)
        if raise_on_error and not r.ok and r.status_code != 401:
            LOG.error(
                "Response code: {}, content: {}, headers: {}".format(
                    r.status_code,
                    r.content,
                    r.headers
                )
            )
            raise requests.HTTPError(r.content, response=r)
        return r

    def post(self, url, headers=None, body=None, **kwargs):
        return self.request(url, "POST", headers=headers, body=body, **kwargs)

    def get(self, url, **kwargs):
        return self.request(url, "GET", **kwargs)

    def put(self, url, body=None, **kwargs):
        return self.request(url, "PUT", body=body, **kwargs)

    def delete(self, url, **kwargs):
        return self.request(url, "DELETE", **kwargs)


class HttpClientOpenId(HttpClient):

    def __init__(self, base_url, keycloak_ip, user, password,
                 client_id=None, verify=None):
        self.keycloak_ip = keycloak_ip
        self.client_id = client_id
        super().__init__(base_url, user, password, verify)

    def login(self):
        """Authorize with username/password and get new token"""
        self.keycloak_client = KeycloakUserClient(self.keycloak_ip,
                                                  self.username,
                                                  self.password,
                                                  client_id=self.client_id,
                                                  realm_name="iam")
        result = self.keycloak_client.get_openid_token()
        LOG.debug(f"OpenID data for UI:\n{result}")
        self.token_creation_time = int(time.time())
        self.refresh_expire = result['expires_in']
        self.login_expire = result['refresh_expires_in']
        LOG.info("Got new UI auth token")
        self.kwargs.update({"auth": TokenAuth(result['id_token'])})

    def refresh_token(func):
        def wrap(self, *args, **kwargs):
            if int(time.time()) > self.token_creation_time + self.login_expire:
                LOG.info("Re-login with OpenID")
                self.login()
            elif int(time.time()) > self.token_creation_time + self.refresh_expire:
                LOG.info("Refresh OpenID token")
                result = self.keycloak_client.refresh_token()
                LOG.info("UI auth token has been refreshed")
                self.token_creation_time = int(time.time())
                self.kwargs.update({"auth": TokenAuth(result['access_token'])})

            return func(self, *args, **kwargs)

        return wrap

    @refresh_token
    @retry_429(delay=40, tries=3)
    def request(self, url, method, headers=None, body=None,
                raise_on_error=True, **kwargs):

        return super().request(url, method, headers, body, raise_on_error, **kwargs)
