#!/bin/env python3

import os
import yaml
import logging
import argparse


def setup_logging():
    """Set up the logging configuration."""
    logging.basicConfig(level=logging.INFO,
                        format='%(asctime)s - %(levelname)s - %(message)s',
                        handlers=[
                            logging.StreamHandler()
                        ])


def find_yaml_files(directory):
    """Find all .yaml files in the specified directory."""
    yaml_files = []
    for root, _, files in os.walk(directory):
        for file in files:
            if file.endswith(".yaml"):
                yaml_files.append(os.path.join(root, file))
    return yaml_files


def load_and_check(file_path):
    """Load and log the content of a yaml file."""
    ids = {'ubuntu/bionic': 'ubuntu_18.04',
           'ubuntu/focal': 'ubuntu_20.04',
           'ubuntu/jammy': 'ubuntu_22.04',
           'ubuntu/noble': 'ubuntu_24.04', }
    error_detected = False
    with open(file_path, 'r') as file:
        try:
            logging.info(f"Loading  {file_path}")
            content = yaml.safe_load(file)
        except yaml.YAMLError as e:
            logging.error(f"Error loading {file_path}: {e}")
            return
        if content['management']['baremetal'].get('allowedDistributions', ''):
            for i in content['management']['baremetal']['allowedDistributions']:
                if i['id'] == 'ubuntu/focal' and not i['os_version'] == ids['ubuntu/focal']:
                    logging.error(f"Error in {file_path}")
                    error_detected = True
                if i['id'] == 'ubuntu/jammy' and not i['os_version'] == ids['ubuntu/jammy']:
                    logging.error(f"Error in {file_path}")
                    error_detected = True
                if i['id'] not in ids.keys():
                    raise Exception(f"Test dont know id:{i['id']}. Please update test!")
        if error_detected:
            raise Exception('Mistakes has been found')


def main(the_dir):
    """Main function to find, load, and log yaml files."""
    setup_logging()
    yaml_files = find_yaml_files(the_dir)
    if not yaml_files:
        logging.info("No .yaml files found in the directory.")
    else:
        yaml_files.sort()
        for file_path in yaml_files:
            load_and_check(file_path)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Search and YAML files in a directory.')
    parser.add_argument('--directory', type=str, help='The directory path to search for .yaml files')
    args = parser.parse_args()
    main(args.directory)
