# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import re
import time
from datetime import datetime

import pytest
import yaml

from si_tests import logger
from si_tests.deployments.utils import commons
from si_tests.managers.clustercheck_mos_manager import ClusterCheckMosManager
from si_tests.utils import waiters

LOG = logger.logger


class TestTFDBBackup(object):
    # Workload details
    lb_url = "http://10.11.12.104:80"
    network_name = None
    test_tag = "tf-database-test-tag"
    custom_tag = "tf-database-backup-tag"
    backup_name = None  # backup without custom_tag

    # DBBackup data
    rclone_remote = None
    backup_path_local = "/data/backups"

    # Openstack client manager
    os_client_manager = None

    @pytest.fixture(scope='function')
    def skip_by_remote_sync(self, tf_manager):
        if not tf_manager.is_tf_tfdbbackup_remote_enabled():
            pytest.skip("RemoteSync feature disabled")

    @staticmethod
    def _set_cls_attr(request, stack_outputs, attr):
        for output in stack_outputs:
            if output['output_key'] == attr:
                setattr(request.cls, attr, output['output_value'])

    @pytest.fixture(scope='class', autouse=True)
    def setup_cls(self, request, openstack_client_manager, tf_manager):
        request.cls.os_client_manager = openstack_client_manager
        request.cls.rclone_remote = tf_manager.rclone_remote
        LOG.info("Create heat stack with loadbalancer before DB Backup/Restore procedures")
        stack = ClusterCheckMosManager.created_stack_tf_lb(request, openstack_client_manager)
        self._set_cls_attr(request, stack['outputs'], 'lb_url')
        self._set_cls_attr(request, stack['outputs'], 'network_name')

    def is_test_network_present(self):
        cmd_net = ['/bin/sh', '-c', f'openstack network show {self.network_name} -f yaml 2>/dev/null']
        network = yaml.safe_load(self.os_client_manager.client.exec(cmd_net))
        return True if network['name'] == self.network_name else False

    def is_test_tag_set(self, custom_tag=None):
        tag = custom_tag if custom_tag else self.test_tag
        cmd_tag_in_net = ['/bin/sh', '-c', f'openstack network show {self.network_name} -c tags -f yaml 2>/dev/null']
        tags_list = yaml.safe_load(self.os_client_manager.exec(cmd_tag_in_net))
        return True if tag in tags_list['tags'] else False

    def set_network_test_tag(self, custom_tag=None):
        tag = custom_tag if custom_tag else self.test_tag
        cmd_add_tag_for_net = ['/bin/sh', '-c', f'openstack network set {self.network_name} --tag {tag}']
        self.os_client_manager.client.exec(cmd_add_tag_for_net)

    @staticmethod
    def _make_list(raw_list, prefix=""):
        result = []
        for line in raw_list:
            pair = line.split()
            result.append({"hash": pair[0], "file": pair[1].replace(prefix, "")})
        return result

    @staticmethod
    def is_list_eq(list1, list2):
        if len(list1) != len(list2):
            return False
        diff = [i for i in list1 if i not in list2]
        return diff == []

    def get_backups_local(self, pod):
        output = pod.exec(["find", self.backup_path_local, "-type", "f", "-exec", "md5sum", "{}", ";"])
        raw_list = (output.strip().split("\n"))
        items = '\n'.join(raw_list)
        commons.LOG.info(f"Local backups:\n{items}")
        if not items:
            return []
        return self._make_list(raw_list, f"{self.backup_path_local}/")

    def get_backups_remote(self, pod, remote_sync_data):
        output = pod.exec(["rclone", "--quiet", "--no-check-certificate", "md5sum",
                           f"{self.rclone_remote}:{remote_sync_data['path']}/{remote_sync_data['backupDir']}"])
        raw_list = (output.strip().split("\n"))
        items = '\n'.join(raw_list)
        commons.LOG.info(f"Backups on remote storage:\n{items}")
        return self._make_list(raw_list)

    @staticmethod
    def get_latest_backup_file(filenames):
        pattern = re.compile(r'db-dump-(\d{8})-(\d{6})\.json')

        def extract_timestamp(fname):
            match = pattern.search(fname)
            if not match:
                return None
            date_part, time_part = match.groups()
            return datetime.strptime(date_part + time_part, "%Y%m%d%H%M%S")

        # Create a list of (filename, timestamp) tuples
        dated_files = [
            (fname, extract_timestamp(fname))
            for fname in filenames
            if extract_timestamp(fname) is not None
        ]

        if not dated_files:
            return None

        # Sort by datetime and return the file with the latest timestamp
        return max(dated_files, key=lambda x: x[1])[0]

    @pytest.mark.dependency(name="test_tf_dbbackup")
    def test_tf_dbbackup(self, show_step, request, tf_manager, os_manager, openstack_client_manager):
        """ Check automatically backup TF data

        Scenario:
            1. Check LB functionality of created heat stack (Setup class)
            2. Enable TFDBBackup via TF operator CR
            3. Trigger TFDBBackup cronjob and waited for it to complete successfully.
            4. Get name of backup file (will be used in restore test)
            5. Set a custom tag on test network
            6. Trigger TFDBBackup cronjob to create a backup with custom network tag.
            7. Disable TFDBBackup via TF operator CR and wait cronjob will be deleted
        """

        show_step(1)
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

        show_step(2)
        tf_manager.db_backup(state=True)

        show_step(3)
        tf_manager.trigger_tfdbbackup_cronjob()

        show_step(4)
        tf_db_check_pod = request.getfixturevalue("tf_db_check_pod")
        local_backups = self.get_backups_local(tf_db_check_pod)
        files = [backup['file'] for backup in local_backups]
        request.cls.backup_name = self.get_latest_backup_file(files)
        LOG.info(f"Initial backup file: {self.backup_name}")

        show_step(5)
        self.set_network_test_tag(custom_tag=self.custom_tag)
        assert self.is_test_tag_set(custom_tag=self.custom_tag), f"Test tag {self.custom_tag} wasn't found."

        show_step(6)
        tf_manager.trigger_tfdbbackup_cronjob()

        local_backups = self.get_backups_local(tf_db_check_pod)
        LOG.info(f"Backup files:\n{local_backups}")

        files = [backup['file'] for backup in local_backups]
        latest_backup = self.get_latest_backup_file(files)
        assert latest_backup != self.backup_name, (f"Latest backup ({latest_backup}) and backup without custom "
                                                   f"network tag ({self.backup_name}) are the same file.")

        show_step(7)
        tf_manager.db_backup(state=False)

    @pytest.mark.usefixtures('skip_by_remote_sync')
    @pytest.mark.dependency(name="test_tf_dbbackup_remote", depends=["test_tf_dbbackup"])
    def test_tf_backup_remote_storage(self, show_step, tf_manager, tf_db_check_pod):
        """ Check remote sync to S3 storage:

        Scenario:
            1. Get list of backup files with their md5sum from remote storage
            2. Get list of backup files with their md5sum from local filesystem
            3. Check if both lists are equal each other
        """

        if not tf_manager.is_tf_tfdbbackup_remote_enabled():
            pytest.skip("RemoteSync feature disabled")

        tfoperator = tf_manager.tfoperator()
        remote_sync_data = tfoperator.data["spec"]["features"]["dbBackup"]["remoteSync"]

        show_step(1)
        remote_backups = self.get_backups_remote(tf_db_check_pod, remote_sync_data)

        show_step(2)
        local_backups = self.get_backups_local(tf_db_check_pod)

        show_step(3)
        commons.LOG.info("Assert if the remote backup doesn't match the local one.")
        assert self.is_list_eq(local_backups, remote_backups)

    @pytest.mark.dependency(name="test_tf_dbrestore", depends=["test_tf_dbbackup"])
    def test_tf_dbrestore(self, tf_manager, os_manager, openstack_client_manager, show_step):
        """ Check automatically restore TF data

        Scenario:
            1. Remove tfdbrestore before test if needed
            2. Check network objects, including LB functionality.
            3. Attach network tag for comparing in database between backup and restore
            4. Enable restore via TF operator CR and wait for restore process to be completed
            5. Wait for TF Operator health status and check contrail api readiness
            6. Verify that the network exists and the load balancer is functioning correctly.
            7. Check that network contains only custom tag (latest backup was selected automatically).
        """

        show_step(1)
        # Check that we don't have tfdbrestore objects before test,
        # https://docs.mirantis.com/mosk/latest/single/#restore-tf-data
        if tf_manager.is_tfdbrestore_present():
            LOG.info('Removing tfdbrestore object before test')
            tf_manager.tfdbrestore().delete()

        show_step(2)
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

        show_step(3)
        self.set_network_test_tag()
        assert self.is_test_tag_set(), f"Test tag {self.test_tag} wasn't found."

        show_step(4)
        tf_manager.db_restore()

        show_step(5)
        # Wait for a while before check health status.
        time.sleep(60)
        tf_manager.wait_tfoperator_healthy()
        waiters.wait(ClusterCheckMosManager.check_contrail_api_readiness,
                     predicate_args=[openstack_client_manager],
                     timeout=300, interval=15,
                     timeout_msg="Timeout: Contrail api is not Ready")

        show_step(6)
        assert self.is_test_network_present(), f"Can't find test network {self.network_name} after DB Restore"
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

        show_step(7)
        assert not self.is_test_tag_set(), f"Test tag {self.test_tag} is still present after DB Restore."
        assert self.is_test_tag_set(custom_tag=self.custom_tag), f"Test tag {self.custom_tag} wasn't found."

    @pytest.mark.dependency(depends=["test_tf_dbbackup_remote"])
    def test_tf_dbrestore_remote(self, tf_manager, openstack_client_manager, tf_db_check_pod, show_step):
        """ Check synchronization of backups from remote storage for restore TF data

        Scenario:
            1. Remove tfdbrestore before test if needed
            2. Attach network tag for comparing in database between backup and restore
            3. Delete all DB backups from local storage
            4. Enable DB restore specifying DB backup file (without custom network tag)
            5. Wait for TF Operator health status and check contrail api readiness
            6. Verify that the network exists and the load balancer is functioning correctly.
            7. Check that network doesn't contain any tags
        """

        if not tf_manager.is_tf_tfdbbackup_remote_enabled():
            pytest.skip("RemoteSync feature disabled")

        show_step(1)
        # Check that we don't have tfdbrestore objects before test,
        # https://docs.mirantis.com/mosk/latest/single/#restore-tf-data
        if tf_manager.is_tfdbrestore_present():
            LOG.info('Removing tfdbrestore object before test')
            tf_manager.tfdbrestore().delete()

        show_step(2)
        self.set_network_test_tag()
        assert self.is_test_tag_set(), f"Test tag {self.test_tag} wasn't found."

        show_step(3)
        tf_db_check_pod.exec(['/bin/sh', '-c', f"rm -f {self.backup_path_local}/*"])

        # Check local backups were deleted
        local_backups = self.get_backups_local(tf_db_check_pod)
        assert len(local_backups) == 0, f"Not all backup files were deleted:\n{local_backups}"

        show_step(4)
        tf_manager.db_restore(backup_file=self.backup_name)

        show_step(5)
        # Wait for a while before check health status.
        time.sleep(60)
        tf_manager.wait_tfoperator_healthy()
        waiters.wait(ClusterCheckMosManager.check_contrail_api_readiness,
                     predicate_args=[openstack_client_manager],
                     timeout=300, interval=15,
                     timeout_msg="Timeout: Contrail api is not Ready")

        show_step(6)
        assert self.is_test_network_present(), f"Can't find test network {self.network_name} after DB Restore"
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

        show_step(7)
        assert not self.is_test_tag_set(), f"Test tag {self.test_tag} is still present after DB Restore."
        assert not self.is_test_tag_set(custom_tag=self.custom_tag), (f"Test tag {self.custom_tag} is still "
                                                                      f"present after DB Restore.")
