import pytest
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters

LOG = logger.logger


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures("introspect_child_deploy_objects")
def test_add_nodes(kaas_manager, check_sg, proxy_manager, openstack_client, _):
    """Add nodes to the child cluster."""
    # Collecting env data
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    machine_image_name = settings.KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME.split(',')[0]
    machine_flavor_name = settings.KAAS_CHILD_CLUSTER_MACHINE_FLAVOR_NAME
    machine_az_name = settings.KAAS_CHILD_CLUSTER_AZ_NAME
    boot_from_volume = settings.OPENSTACK_BOOT_FROM_VOLUME
    boot_volume_size = settings.OPENSTACK_BOOT_VOLUME_SIZE

    # step 001 - Get namespace
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    region = kaas_manager.get_mgmt_cluster().region_name
    stacklight_label = {"key": "stacklight", "value": "enabled"}
    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    if settings.KAAS_OFFLINE_DEPLOYMENT:
        offline_security_group = [check_sg['name']]
    else:
        offline_security_group = None

    LOG.info("Negative checks for nodeLabels")
    worker_machine = cluster.get_machines(machine_type="worker")[0]
    try:
        non_existing = {"key": "nonexiting", "value": "enabled"}
        worker_machine.add_labels([non_existing])
    except Exception as e:
        LOG.info(e)
        if e.status != 400:
            raise e
    else:
        raise Exception("Unexpected success: 'nonexiting' label was assigned")

    assert non_existing not in worker_machine.nodeLabels, \
        "Not allowed label was successfully set"
    LOG.info("==='nonexiting' label was not assigned")

    # check duplicates
    # PRODX-8361
    """
    try:
        worker_machine.add_labels([stacklight_label])
    except Exception as e:
        LOG.info(e)
        message = json.loads(e.body)['message']
        if e.status != 400 or 'duplicate' not in message:
            raise e
    else:
        raise Exception("Unexpected success: duplicated label was assigned")

    assert worker_machine.nodeLabels.count(stacklight_label) == 1, \
        "Duplicated label was assigned: {}".format(worker_machine.nodeLabels)
    """

    try:
        non_existing = {"key": "nonexiting", "value": "enabled"}
        cluster.create_os_machine(
            node_type="node",
            node_flavor=machine_flavor_name,
            node_image=machine_image_name,
            node_az=machine_az_name,
            boot_from_volume=boot_from_volume,
            boot_volume_size=boot_volume_size,
            region=region,
            node_labels=[non_existing])
    except Exception as e:
        LOG.info(e)
        if e.status != 400:
            raise e
    else:
        raise Exception("Unexpected success: machine was created "
                        "with 'nonexiting' label")
    LOG.info("===Machine with 'nonexiting' was not created")
    LOG.info("========================================================")

    LOG.info("Adding master machine")
    # step 003 - Add 1 master node
    master_node1 = cluster.create_os_machine(
        node_type="master",
        node_flavor=machine_flavor_name,
        node_image=machine_image_name,
        node_az=machine_az_name,
        boot_from_volume=boot_from_volume,
        boot_volume_size=boot_volume_size,
        region=region,
        node_sg=offline_security_group)
    LOG.info('Master machine {}'.format(master_node1))

    master_node2 = cluster.create_os_machine(
        node_type="master",
        node_flavor=machine_flavor_name,
        node_image=machine_image_name,
        node_az=machine_az_name,
        boot_from_volume=boot_from_volume,
        boot_volume_size=boot_volume_size,
        region=region,
        node_sg=offline_security_group)
    LOG.info('Master machine {}'.format(master_node2))

    LOG.info("Adding worker machine")
    # step 004 - Add 1 slave node
    slave_node = cluster.create_os_machine(
        node_type="node",
        node_flavor=machine_flavor_name,
        node_image=machine_image_name,
        node_az=machine_az_name,
        boot_from_volume=boot_from_volume,
        boot_volume_size=boot_volume_size,
        region=region,
        node_sg=offline_security_group)
    LOG.info('Added worker machine {}'.format(slave_node))

    failed_bfv_machines = {}
    new_machines = []
    for m_node in [master_node1, master_node2, slave_node]:

        # Waiting for master node become Ready
        cluster.check.wait_machine_status_by_name(machine_name=m_node.name,
                                                  timeout=3600,
                                                  expected_status='Ready')
        new_machines.append(cluster.get_machine(name=m_node.name))

        # Check that bootFromVolume option worked correctly
        if boot_from_volume:
            try:
                cluster.check.check_boot_from_volume_by_machine_name(machine_name=m_node.name,
                                                                     openstack_client=openstack_client,
                                                                     boot_volume_size=boot_volume_size)
            except (AssertionError, Exception) as error:
                failed_bfv_machines.update({m_node.name: error})

    if boot_from_volume:
        if failed_bfv_machines:
            raise Exception(f'bootFromVolume checked failed, list of failed machines with errors: '
                            f'\n {yaml.dump(failed_bfv_machines)}')
        else:
            LOG.info("All machines from cluster deployed with option bootFromVolume successfully")

    # Waiting for machines are Ready
    cluster.check.check_machines_status(timeout=1800)
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()

    # TODO: gvlasov_cleanup_runtime
    # Add temporary check for actual runtime on added node for 2.27 -> 2.28 -> 2.29
    if settings.DESIRED_RUNTIME:
        cluster.check.compare_machines_runtime_with_desired(new_machines, machine_is_new=True)

    # Check that Machine hostname is created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines(machines=new_machines)

    new_machine = [x for x in cluster.get_machines(machine_type="worker")
                   if not x.nodeLabels][0]

    machines = [
        x for x in cluster.get_machines(machine_type="worker")
        if 'nodeLabels' in x.spec['providerSpec']['value'].keys() and
           stacklight_label in x.nodeLabels
    ]
    if machines:
        LOG.info("Assigning 'stacklight' label to new worker machine")
        try:
            new_machine.add_labels([stacklight_label])
        except Exception as e:
            LOG.info(e)

        assert stacklight_label in new_machine.nodeLabels, \
            "Label was not set"
        waiters.wait(lambda: worker_machine.check_k8s_nodes_has_labels(),
                     timeout=300, interval=10)
    else:
        LOG.info("Found no machines with stacklight label."
                 "Skipping adding/removing label check")

    # Collecting artifacts
    cluster.store_k8s_artifacts()

    # Check/wait for correct docker service replicas in cluster
    ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
    cluster.check.check_actual_expected_docker_services(
        changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name})
    cluster.check.check_k8s_pods()
    cluster.check.check_actual_expected_pods()
    cluster.check.check_cluster_readiness()
    cluster.provider_resources.save_artifact()

    if settings.CHILD_MKE_CUSTOM_CERTIFICATES:
        host = cluster.get_load_balancer_address()
        cluster.check.check_cert_conversation(host)
        cluster.check.check_cert_equal()
