import pytest


from si_tests import settings
from si_tests import logger
from si_tests.utils import utils
from si_tests.deployments.utils import equinix_mos_child_helpers
from si_tests.utils.utils import Provider

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures("log_method_time")
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}".format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_equinix_child_mos(kaas_manager, show_step, _):
    """Deploy child cluster on top of management cluster.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating equinixmetal credential
        5. Create MOS child cluster
        6. Check cluster readiness

    """

    # Collecting env data
    show_step(1)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    LOG.info("Available releases: {}".format(kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()
    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    slave_nodes_count = int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
    master_nodes_count = int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT)
    provider = Provider.equinixmetal
    rnd_string = utils.gen_random_string(4)
    region = kaas_manager.get_mgmt_cluster().region_name

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    namespaces = [n.name for n in kaas_manager.get_namespaces()]
    if namespace_name not in namespaces:
        ns = kaas_manager.create_namespace(namespace_name)
    elif namespace_name in namespaces:
        ns = kaas_manager.get_namespace(namespace_name)

    show_step(3)
    # ability to use already created credentials (for deployments on hosted kaas)
    if settings.KAAS_EQUINIX_CHILD_PUBLIC_KEY_NAME:
        public_key_name = settings.KAAS_EQUINIX_CHILD_PUBLIC_KEY_NAME
    else:
        # Creating public key
        public_key_name = "{cluster_name}-{rnd_string}-key".format(cluster_name=cluster_name, rnd_string=rnd_string)
        LOG.info("Public key name - {public_key_name}".format(public_key_name=public_key_name))
        with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
            pub_key_content = pub_key.read()
        ns.create_publickey(public_key_name, pub_key_content)

    show_step(4)
    if settings.KAAS_EQUINIX_CHILD_CREDENTIAL_NAME:
        credential_name = settings.KAAS_EQUINIX_CHILD_CREDENTIAL_NAME
    else:
        # Creating Equinix Metal credentials
        credential_name = "{cluster_name}-{rnd_string}-cred".format(cluster_name=cluster_name, rnd_string=rnd_string)
        LOG.info("Credential name - {credential_name}".format(credential_name=credential_name))
        ns.create_equinix_credential(
            credential_name,
            settings.KAAS_EQUINIX_API_TOKEN,
            settings.KAAS_EQUINIX_PROJECT_ID,
            region=region,
        )

    show_step(5)
    # cluster = ns.get_cluster(cluster_name)
    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        credentials_name=credential_name,
        provider=provider.provider_name,
        region=region,
        dns=child_dns,
        dedicated_controlplane=settings.KAAS_EQUINIX_CHILD_DEDICATED_CONTROLPLANE,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        public_key_name=public_key_name,
        secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED,
    )

    facility = cluster.spec['providerSpec']['value'].get('facility')
    assert facility, (
        f"Cluster {cluster.namespace}/{cluster.name} don't have 'facility' key in the providerSpec.value")

    openstack_control_labels = {
        "openstack-control-plane": "enabled",
        "openstack-gateway": "enabled",
        "openvswitch": "enabled",
        "stacklight": "enabled",
    }
    openstack_compute_labels = {"openstack-compute-node": "enabled", "openvswitch": "enabled"}

    openstack_control_nodes_count = 3

    contr_labels = [{"key": k, "value": v} for k, v in openstack_control_labels.items()]
    comp_labels = [{"key": k, "value": v} for k, v in openstack_compute_labels.items()]

    contr_machine_type = "worker"
    if settings.KAAS_EQUINIX_CHILD_DEDICATED_CONTROLPLANE:
        for node in range(master_nodes_count):
            cluster.create_equinix_machine(
                node_type="master",
                os=settings.KAAS_EQUINIX_OS,
                billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
                machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_MASTER,
                region=region,
                facility=facility
            )
        for node in range(openstack_control_nodes_count):
            cluster.create_equinix_machine(
                node_type="node",
                os=settings.KAAS_EQUINIX_OS,
                billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
                machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_OS_CONTROL,
                region=region,
                facility=facility,
                node_labels=contr_labels,
            )
    else:
        contr_machine_type = "control"
        for node in range(openstack_control_nodes_count):
            cluster.create_equinix_machine(
                node_type="master",
                os=settings.KAAS_EQUINIX_OS,
                billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
                machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_OS_CONTROL,
                region=region,
                facility=facility,
                node_labels=contr_labels,
            )

    slave_nodes_count -= openstack_control_nodes_count

    for node in range(slave_nodes_count):
        cluster.create_equinix_machine(
            node_type="node",
            os=settings.KAAS_EQUINIX_OS,
            billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
            machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_WORKER,
            region=region,
            facility=facility,
            node_labels=comp_labels,
        )

    # Waiting for machines are Ready and helmbundles are deployed
    try:
        cluster.check.check_machines_status()
        cluster.check.check_cluster_nodes()
    finally:
        cluster.store_machines_artifacts()

    openstack_control_machines = [
        x
        for x in cluster.get_machines(machine_type=contr_machine_type)
        if "nodeLabels" in x.spec["providerSpec"]["value"].keys()
        and contr_labels == x.spec["providerSpec"]["value"]["nodeLabels"]
    ]
    assert len(openstack_control_machines) == openstack_control_nodes_count, "Not all machines have expected nodeLabels"
    openstack_compute_machines = [
        x
        for x in cluster.get_machines(machine_type="worker")
        if "nodeLabels" in x.spec["providerSpec"]["value"].keys()
        and comp_labels == x.spec["providerSpec"]["value"]["nodeLabels"]
    ]
    assert len(openstack_compute_machines) == slave_nodes_count, "Not all machines have expected nodeLabels"

    equinix_mos_child_helpers.configure_equinix_mos_networking(cluster)
    equinix_mos_child_helpers.configure_equinix_mos_ceph(ns, openstack_control_machines, openstack_compute_machines)

    for machine in cluster.get_machines():
        machine.check_k8s_nodes_has_labels()

    cluster.check.check_k8s_nodes()
    # https://mirantis.jira.com/browse/PRODX-18603
    if not settings.SI_OPENID_AUTH:
        cluster.check.check_helmbundles()

    show_step(6)
    cluster.check.check_cluster_readiness(timeout=3000, interval=60)

    # with openid user get_helmbundle for child cluster is not allwed. so skip
    # the checks which rely on get_helmbundle
    #  https://mirantis.jira.com/browse/PRODX-18603
    if not settings.SI_OPENID_AUTH:
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods(timeout=3200)

    try:
        health_info = cluster.check.get_ceph_health_detail()
        assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                     f'Current ceph health status: {health_info}'
    except AssertionError:
        cluster.check.wait_ceph_health_status(timeout=3600, interval=30)

    # TODO: list_storage_class method doesn't support relogin on 401 error,
    # https://mirantis.jira.com/browse/PRODX-18770
    if not settings.SI_OPENID_AUTH:
        cluster.check.check_ceph_pvc()
    cluster.check.check_overlay_encryption_functionality()

    # Collecting artifacts
    cluster.store_k8s_artifacts()
    cluster.provider_resources.save_artifact()
