#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os
import pytest

from si_tests import logger
from si_tests import settings
from si_tests.managers import bootstrap_manager, si_config_manager
from si_tests.utils import utils

LOG = logger.logger


class TestSetupBaremetalEnvBV2Airgap():
    @pytest.mark.usefixtures('log_step_time', 'log_method_time')
    def test_setup_bm_env_bv2_airgap(self, show_step, docker_image):
        """Test the accessibility of all files on the CDN.

        Scenario:
            1. airgapped.sh inspect
            2. airgapped.sh demo --init
            3. Run docker-compose
            4. Copy/install demo-ca.crt
            5. airgapped.sh sync
            6. airgapped.sh export
            7. airgapped.sh push
            8. airgapped.sh validate
        """
        # Basic checks
        assert settings.MCC_CDN_DOCKER, "Value MCC_CDN_DOCKER is not defined"
        assert settings.MCC_CDN_BINARY, "Value MCC_CDN_BINARY is not defined"
        assert settings.MCC_CDN_DEBIAN, "Value MCC_CDN_DEBIAN is not defined"
        assert settings.MCC_CDN_MCR_REPO, "Value MCC_CDN_MCR_REPO is not defined"
        assert settings.KAAS_AIRGAP_TOOLS_IMAGE, "Value KAAS_AIRGAP_TOOLS_IMAGE is not defined"
        assert settings.KAAS_RELEASE_ARTIFACTS_TAR_URL, "Value KAAS_RELEASE_ARTIFACTS_TAR_URL is not defined"

        si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
        run_on_remote = si_config.data.get('run_on_remote', {}).get('RUN_ON_REMOTE', False)
        assert (settings.RUN_ON_REMOTE or run_on_remote), "Test work only with RUN_ON_REMOTE option"

        LOG.info("Starting MCC air-gapped flow test.")

        bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()
        bootstrap.wait_for_seed_node_ssh()
        # Prepare environment variables for commands
        home_dir = f'/home/{settings.SEED_SSH_LOGIN}'
        airgap_cdn_host = settings.SEED_STANDALONE_EXTERNAL_IP
        env_vars = bootstrap.prepare_airgap_env(home_dir, docker_image)
        envs_string = utils.make_export_env_strting(env_vars)

        # Build cluster release arguments
        cluster_release_args = bootstrap.build_cluster_release_args_airgap()

        # install docker-compose
        airgapped_workspace = '/opt/airgapped'
        docker_dir = os.path.join(airgapped_workspace, "docker")
        install_dc_cmd = (
            "if ! command -v docker-compose >/dev/null 2>&1; then "
            "sudo apt-get update && sudo apt-get install -y docker-compose; "
            "fi"
        )
        remote = bootstrap.remote_seed()
        remote.check_call(f"{envs_string}; {install_dc_cmd}", raise_on_err=True, verbose=True)

        show_step(1)
        mcc_full_version = bootstrap.get_kaas_release_name()
        bootstrap.run_airgapped_command("inspect --unique", mcc_full_version, cluster_release_args, env_vars=env_vars)
        remote.check_call(f"{envs_string}; df -h")

        show_step(2)
        bootstrap.run_airgapped_command(
            action="demo --init --san-ip",
            mcc_version=None,
            extra_args=f"{airgap_cdn_host}",
            env_vars=env_vars,
        )

        show_step(3)
        assert os.path.isdir(docker_dir), f"[ERROR] Directory {docker_dir} not found!"
        remote.check_call(f"{envs_string}; cd {docker_dir} && docker-compose up -d", raise_on_err=True, verbose=True)

        show_step(4)
        remote.check_call(
            f'{envs_string};'
            'wget --tries=5 --no-verbose --show-progress --waitretry=15 '
            f'--retry-connrefused -O {home_dir}/demo-ca.crt http://localhost:8080/demo-ca.crt'
        )
        remote.check_call(
            f"{envs_string}; sudo cp {home_dir}/demo-ca.crt /usr/local/share/ca-certificates/demo-ca.crt",
            raise_on_err=True,
            verbose=True
        )
        remote.check_call(f"{envs_string}; sudo update-ca-certificates", raise_on_err=True, verbose=True)
        remote.check_call(f"{envs_string}; sudo systemctl restart docker", raise_on_err=True, verbose=True)

        show_step(5)
        bootstrap.run_airgapped_command(
            "sync",
            mcc_full_version,
            cluster_release_args,
            env_vars=env_vars,
            timeout=4800,
        )

        show_step(6)
        bootstrap.run_airgapped_command(
            "export",
            mcc_full_version,
            cluster_release_args,
            env_vars=env_vars,
        )

        show_step(7)
        bootstrap.run_airgapped_command(
            "push",
            mcc_full_version,
            cluster_release_args,
            extra_args="--url-map 'mirantis.azurecr.io=127.0.0.1' --insecure-registry",
            env_vars=env_vars,
        )

        show_step(8)
        validate_args = (
            f"--url-map 'https://binary.mirantis.com=https://{airgap_cdn_host}:8081' "
            f"--url-map 'https://mirror.mirantis.com=https://{airgap_cdn_host}:8082' "
            f"--url-map 'https://repos.mirantis.com=https://{airgap_cdn_host}:8083' "
            f"--url-map 'mirantis.azurecr.io={airgap_cdn_host}' "
            "--files --images"
        )
        bootstrap.run_airgapped_command(
            "validate",
            mcc_full_version,
            cluster_release_args,
            extra_args=validate_args,
            env_vars=env_vars,
        )

        LOG.info("[INFO] Airgapped flow completed successfully.")
