import yaml
import json
import requests

from si_tests import settings
from si_tests import logger
from si_tests.utils import templates as utils

LOG = logger.logger


class ProxyManager(object):
    """Deploy and manage lightweight proxy inside management cluster"""

    def __init__(self, kubectl_client):
        """kubectl_client: k8s client to KaaS management cluster"""
        LOG.info('Initialized proxy manager')
        self._kubectl_client = kubectl_client
        self.prx_name = settings.KAAS_ADDITIONAL_PROXY_NAME
        self.prx_ns = settings.KAAS_ADDITIONAL_PROXY_NS
        self.prx_sts = [x for x in self._kubectl_client.statefulsets.list(
            namespace=self.prx_ns) if x.name == self.prx_name]
        self.prx_svc = [x for x in self._kubectl_client.services.list(
            namespace=self.prx_ns) if x.name == self.prx_name]
        self.prx_str = None

    def deploy_proxy_sts(self):
        """Deploy proxy inside management cluster
        """
        LOG.info('Get or deploy proxy statefulset')
        if self.prx_sts:
            self.prx_sts = self.prx_sts[0]
            LOG.info("Proxy statefulset is present on environment.")
            LOG.info("Read proxy statefulset: {0}".format(self.prx_sts.read()))
        else:
            options = {
                'KAAS_ADDITIONAL_PROXY_USER':
                    settings.KAAS_ADDITIONAL_PROXY_USER,
                'KAAS_ADDITIONAL_PROXY_PASS':
                    settings.KAAS_ADDITIONAL_PROXY_PASS,
                'KAAS_ADDITIONAL_PROXY_NAME':
                    settings.KAAS_ADDITIONAL_PROXY_NAME,
                'KAAS_ADDITIONAL_PROXY_IMAGE':
                    settings.KAAS_ADDITIONAL_PROXY_IMAGE
            }
            templates_sts = utils.render_template(
                settings.KAAS_ADDITIONAL_PROXY_YAML, options)
            LOG.debug(templates_sts)
            json_body = json.dumps(yaml.load(templates_sts,
                                             Loader=yaml.SafeLoader))
            self.prx_sts = self._kubectl_client.statefulsets.create(
                name=settings.KAAS_ADDITIONAL_PROXY_NAME,
                namespace=settings.KAAS_ADDITIONAL_PROXY_NS,
                body=json.loads(json_body))
            self.prx_sts.wait_readiness()

    def deploy_proxy_svc(self):
        """Deploy proxy svc to obtain external address

        :return:
        """
        LOG.info('Get or deploy proxy service')
        if self.prx_svc:
            self.prx_svc = self.prx_svc[0]
            LOG.info("Proxy service is present on environment.")
            LOG.info("Read proxy svc: {}".format(self.prx_svc.read()))
        else:
            options = {
                'KAAS_ADDITIONAL_PROXY_NAME':
                    settings.KAAS_ADDITIONAL_PROXY_NAME
            }
            templates_svc = utils.render_template(
                settings.KAAS_ADDITIONAL_PROXY_SVC_YAML, options)
            LOG.debug(templates_svc)
            json_body = json.dumps(yaml.load(templates_svc,
                                             Loader=yaml.SafeLoader))
            self.prx_svc = self._kubectl_client.services.create(
                name=settings.KAAS_ADDITIONAL_PROXY_NAME,
                namespace=settings.KAAS_ADDITIONAL_PROXY_NS,
                body=json.loads(json_body))
            self.prx_svc.wait_for_external_ip()

    def generate_proxy_string(self):
        """Generate access proxy string to use in proxyobject

        :return:
        """
        ext_ip = self.prx_svc.get_external_ip()
        ports = self.prx_svc.get_ports()
        proxy = "http://{user}:{passw}@{ip}:{port}".format(
            user=settings.KAAS_ADDITIONAL_PROXY_USER,
            passw=settings.KAAS_ADDITIONAL_PROXY_PASS,
            ip=ext_ip,
            port=ports[0].port
        )
        self.prx_str = proxy
        return self.prx_str

    def check_proxy_conn(self):
        proxy_d = {
            "http": self.prx_str,
            "https": self.prx_str
        }
        check_url = "http://mirantis.azurecr.io"
        try:
            requests.get(url=check_url, proxies=proxy_d, timeout=20)
        except Exception as e:
            LOG.error(e)
            LOG.info("Unexpected proxy behaviour. Could not connect to {0} via"
                     "proxy {1}".format(check_url, self.prx_str))
            raise e
