import boto3 as aws_client
from botocore import exceptions
from botocore.config import Config
from si_tests import logger

LOG = logger.logger


class AwsManager(object):
    """Aws manager"""

    def __init__(self,
                 aws_access_key_id='',
                 aws_secret_access_key='',
                 region_name='',
                 config_kwargs=None):
        self.aws_access_key_id = aws_access_key_id
        self.aws_secret_access_key = aws_secret_access_key
        self.region_name = region_name
        config_kwargs = config_kwargs or {
            'retries': {'max_attempts': 10, 'mode': 'standard'}}
        self.config = Config(**config_kwargs)

    @property
    def aws_ec2_client(self):
        return aws_client.resource(
            'ec2', aws_access_key_id=self.aws_access_key_id,
            aws_secret_access_key=self.aws_secret_access_key,
            region_name=self.region_name, config=self.config)

    # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#client # noqa
    @property
    def aws_service_client(self):
        return aws_client.client(
            'ec2', aws_access_key_id=self.aws_access_key_id,
            aws_secret_access_key=self.aws_secret_access_key,
            region_name=self.region_name, config=self.config)

    @property
    def aws_lb_client(self):
        return aws_client.client(
            'elb', aws_access_key_id=self.aws_access_key_id,
            aws_secret_access_key=self.aws_secret_access_key,
            region_name=self.region_name, config=self.config)

    @property
    def aws_iam_client(self):
        return aws_client.client(
            'iam', aws_access_key_id=self.aws_access_key_id,
            aws_secret_access_key=self.aws_secret_access_key,
            region_name=self.region_name, config=self.config)

    @property
    def aws_cloudformation_client(self):
        return aws_client.client(
            'cloudformation', aws_access_key_id=self.aws_access_key_id,
            aws_secret_access_key=self.aws_secret_access_key,
            region_name=self.region_name, config=self.config)

    def get_cloudformation_stacks(self, show_deleted=False):
        stacks = [s for s in self.aws_cloudformation_client.list_stacks().get(
            'StackSummaries', [])]
        return stacks if show_deleted else [s for s in stacks if s.get(
            'StackStatus') != 'DELETE_COMPLETE']

    def get_cloudformation_stack(self, name):
        stack = [s for s in self.get_cloudformation_stacks() if s.get(
            'StackName') == name]
        return stack[0] if stack else {}

    def get_users(self):
        return self.aws_iam_client.list_users().get('Users', {})

    def get_vpcs(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_vpcs
        return self.aws_service_client.describe_vpcs(Filters=filters)

    def get_vpc(self, vpc_id=''):
        return self.aws_ec2_client.Vpc(vpc_id)

    def get_subnets(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_subnets
        return self.aws_service_client.describe_subnets(Filters=filters)

    def get_subnet(self, subnet_id=''):
        return self.aws_ec2_client.Subnet(subnet_id)

    def get_volumes(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_volumes
        return self.aws_service_client.describe_volumes(Filters=filters)

    def get_volume(self, volume_id=''):
        return self.aws_ec2_client.Volume(volume_id)

    def get_lbs(self):
        return \
            self.aws_lb_client.describe_load_balancers()[
                'LoadBalancerDescriptions']

    def get_route_tables(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_route_tables
        return self.aws_service_client.describe_route_tables(Filters=filters)

    def get_route_table(self, route_table_id=''):
        return self.aws_ec2_client.RouteTable(route_table_id)

    def get_instances(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_instances
        reservations = \
            self.aws_service_client.describe_instances(
                Filters=filters)['Reservations']
        instances = []
        for r in reservations:
            if len(r['Instances']) == 1:
                instances.append(r['Instances'][0])
            else:
                for i in r['Instances']:
                    instances.append(i)
        return instances

    def get_instance(self, instance_id=''):
        return self.aws_ec2_client.Instance(instance_id)

    def get_network_acls(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_network_acls
        return self.aws_service_client.describe_network_acls(Filters=filters)

    def get_network_acl(self, network_acl_id=''):
        return self.aws_ec2_client.NetworkAcl(network_acl_id)

    def get_network_interfaces(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_network_interfaces
        return self.aws_service_client.describe_network_interfaces(
            Filters=filters
        )

    def get_network_interface(self, network_interface_id=''):
        return self.aws_ec2_client.NetworkInterface(network_interface_id)

    def get_internet_gateways(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_internet_gateways
        return self.aws_service_client.describe_internet_gateways(
            Filters=filters
        )

    def get_internet_gateway(self, internet_gateway_id=''):
        return self.aws_ec2_client.InternetGateway(internet_gateway_id)

    def get_security_groups(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_security_groups
        return self.aws_service_client.describe_security_groups(
            Filters=filters
        )

    def get_security_group(self, security_group_id=''):
        return self.aws_ec2_client.SecurityGroup(security_group_id)

    def get_nat_gateways(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_nat_gateways
        return self.aws_service_client.describe_nat_gateways(Filters=filters)

    def get_addresses(self, filters=[]):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.describe_addresses
        return self.aws_service_client.describe_addresses(Filters=filters)

    def get_waiter(self, waiter_type):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.get_waiter
        return self.aws_service_client.get_waiter(waiter_type)

    def get_instance_by_name(self, instance_name=''):
        aws_instance_id = ''
        instance_ = self.aws_ec2_client.instances.filter(
            Filters=[{'Name': 'tag:Name', 'Values': [instance_name]}])
        for ins in instance_:
            aws_instance_id = ins.id
        if aws_instance_id != '':
            return self.get_instance(instance_id=aws_instance_id)
        else:
            raise Exception(f"Instance ({instance_name}) not found!")

    def get_instance_state_by_name(self, instance_name=''):
        instance = self.get_instance_by_name(instance_name)
        return instance.state['Name']

    def instance_power_action(self, instance_name='', action=''):
        instance = self.get_instance_by_name(instance_name)
        if action in ['reboot', 'stop', 'start']:
            try:
                response = None
                if action == 'reboot':
                    response = instance.reboot()
                elif action == 'stop':
                    response = instance.stop()
                elif action == 'start':
                    response = instance.start()
                LOG.info(f"{action} success!!! , {response}")
            except exceptions.ClientError as e:
                LOG.info(f"Error , {e}")
        else:
            raise Exception(f"Action ({action}) absent!")

    def get_all_resources(self):
        resources = {}
        resources['vms'] = self.get_instances()
        resources['vpcs'] = self.get_vpcs().get('Vpcs', [])
        resources['volumes'] = self.get_volumes().get('Volumes', [])
        resources['lbs'] = self.aws_lb_client.describe_load_balancers().get(
            'LoadBalancerDescriptions', [])
        resources['subnets'] = self.get_subnets().get('Subnets', [])
        resources['route_tables'] =\
            self.get_route_tables().get('RouteTables', [])
        resources['sgs'] =\
            self.get_security_groups().get('SecurityGroups', [])
        resources['network_interfaces'] =\
            self.get_network_interfaces().get('NetworkInterfaces', [])
        resources['internet_gateways'] =\
            self.get_internet_gateways().get('InternetGateways', [])
        resources['ips'] = self.get_addresses().get('Addresses', [])
        resources['nat_gateways'] =\
            self.get_nat_gateways().get('NatGateways', [])

        return resources

    def create_volume(self, **kwargs):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.create_volume
        return self.aws_service_client.create_volume(**kwargs)

    def create_subnet(self, **kwargs):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.create_subnet
        return self.aws_service_client.create_subnet(**kwargs)

    def create_network_interface(self, **kwargs):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.create_network_interface
        return self.aws_service_client.create_network_interface(**kwargs)

    def attach_network_interface(self, **kwargs):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.attach_network_interface
        return self.aws_service_client.attach_network_interface(**kwargs)

    def attach_volume(self, **kwargs):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.attach_volume
        return self.aws_service_client.attach_volume(**kwargs)

    def modify_network_interface_attribute(self, **kwargs):
        # https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html#EC2.Client.modify_network_interface_attribute
        return self.aws_service_client.modify_network_interface_attribute(**kwargs)
