#!/usr/bin/python
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from ansible.errors import AnsibleError, AnsibleFilterError
import random
import re

re_mac = re.compile(r'^([a-f0-9]{2}:){5}[a-f0-9]{2}$')
re_mac_delim = re.compile(r'[:-]')


class MacGenerator:
    def __init__(self, prefix=None):
        if prefix:
            self.template = [x.strip() for x in prefix.lower().split(':')[:6]]
        else:
            self.template = []
        self.template.extend('*' * (6 - len(self.template)))

    def random(self):
        mac = []
        for x, item in enumerate(self.template):
            if item in ['*', '**']:
                mac.append(random.randint(0, 255))
            else:
                try:
                    num = int(item, 16)
                    if num > 255:
                        raise AnsibleFilterError(
                            f"Bad number in MAC address template "
                            f"'{self.template}', pos {x}.")
                    mac.append(num)
                except ValueError:
                    raise AnsibleFilterError(
                        f"Bad value in MAC address prefix "
                        f"'{self.template}', pos {x}.")

        return tuple(mac)

    def mac2str(self, mac):
        return ':'.join([f'{x:02x}' for x in mac])

    def str2mac(self, mac_string):
        return tuple([int(x, 16) for x in mac_string.split(':')])

    def allocate(self, macs=None, count=1, default=None):
        """
        Generate a list to be filled by .generate(),
        with some MACs preconfigured from `macs` input.
        Input validated to be a valid MAC string and checked
        against duplicates.
        Empty cells filled with `default` value.
        """
        allocated = list()

        if macs:
            for item in macs[:count]:
                mac = default
                if item:
                    mac_str = re_mac_delim.sub(':', str(item).strip().lower())
                    if not re_mac.match(mac_str):
                        raise Exception(f"Invalid MAC string: '{item}'")
                    mac = self.str2mac(mac_str)
                    if mac in allocated:
                        raise Exception(f"Duplicated MAC: {mac_str}")
                allocated.append(mac)

        if len(allocated) < count:
            allocated.extend([default] * (count - len(allocated)))

        return allocated

    def generate(self, count=1, static_macs=None):
        macs = self.allocate(static_macs, count=count)
        mac_idx = 0
        attempts = 0
        max_attempts = count + 255

        while mac_idx < count:
            mac = macs[mac_idx]

            if mac is None:
                while mac is None:
                    attempts += 1
                    if attempts > max_attempts:
                        raise AnsibleError(
                            f"Failed to generate {count} MACs in "
                            f"{attempts} attempts, {mac_idx} succeeded.")

                    mac = self.random()
                    if mac in macs:
                        mac = None

                macs[mac_idx] = mac

            mac_idx += 1

        return [self.mac2str(x) for x in macs]


def random_macs(macs, count=1, prefix=None, seed=None):
    if seed:
        random.seed(seed)
    macgen = MacGenerator(prefix=prefix)
    return macgen.generate(count, static_macs=macs)


class FilterModule:
    def filters(self):
        return {
            'random_macs': random_macs
        }
