import time
import yaml

from si_tests import logger
from yaml.scanner import ScannerError

LOG = logger.logger


class OpenStackResource:
    def __init__(self, ocm, os_cloud="admin", timeout=120):
        self.ocm = ocm
        self.os_cloud = os_cloud
        self.timeout = timeout

    def _do_exec(self, cmd):
        result = {}
        connection = self.ocm.exec(cmd, _preload_content=False)
        connection.run_forever(timeout=self.timeout)
        result["stdout"] = connection.read_stdout()
        result["stderr"] = connection.read_stderr()
        connection.close()

        if result["stderr"]:
            LOG.error(f"Command '{' '.join(cmd)}' returned stderr: {result['stderr']}")
        return result

    def _exec(self, method, *args, yaml_output=True, combined_output=True):
        """
        :param method: operation on resource i.e "list", "create"
        :param args: list of string positional args should go first i.e.:
            ["--flavor=m1.tiny"]
            ["vm-test-0", "--flavor=m1.tiny"]
        """
        command = ['env', 'OS_CLOUD=' + self.os_cloud, 'PYTHONWARNINGS=ignore::UserWarning',
                   'openstack', self.resource, method, *args]

        if yaml_output:
            command += ['-f', 'yaml']

        result = self._do_exec(command)
        if yaml_output:
            try:
                yaml_res = yaml.safe_load(result["stdout"])
                result["stdout"] = yaml_res
            except ScannerError:
                raise Exception(f"Response YAML deserialization error. Plain response:\n{result}")

        if combined_output:
            return result
        else:
            return result["stdout"]

    def list(self, args, yaml_output=True, combined_output=False):
        return self._exec('list',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def create(self, args, yaml_output=True, combined_output=False):
        return self._exec('create',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def delete(self, args, yaml_output=False, combined_output=False):
        return self._exec('delete',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def show(self, args, combined_output=False):
        return self._exec('show',
                          *args,
                          combined_output=combined_output)

    def set(self, args, combined_output=False):
        return self._exec('set',
                          *args,
                          yaml_output=False,
                          combined_output=combined_output)

    def add(self, args, yaml_output=False, combined_output=False):
        return self._exec('add',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def check(self, args, yaml_output=False, combined_output=False):
        return self._exec('check',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    @property
    def resource(self):
        return self._resource

    def wait_absent(self, name_or_id, timeout=60, interval=10):
        started = time.time()
        while [res for res in self.list([]) if (res["ID"] == name_or_id or res["Name"] == name_or_id)]:
            if int(time.time()) - int(started) > timeout:
                raise TimeoutError(f"Timed out waiting {self._resource} {name_or_id} is deleted")
            time.sleep(interval)


class Aggregate(OpenStackResource):
    _resource = 'aggregate'

    def add_host(self, *args, combined_output=False):
        return self._exec('add host',
                          *args,
                          yaml_output=False,
                          combined_output=combined_output)

    def remove_host(self, *args, combined_output=False):
        return self._exec('remove host',
                          *args,
                          yaml_output=False,
                          combined_output=combined_output)


class ZoneBlacklist(OpenStackResource):
    _resource = 'zone blacklist'


class Role(OpenStackResource):
    _resource = 'role'

    def remove(self, args, yaml_output=False, combined_output=False):
        return self._exec('remove',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)


class User(OpenStackResource):
    _resource = 'user'


class Image(OpenStackResource):
    _resource = 'image'


class Server(OpenStackResource):
    _resource = 'server'

    def migrate(self, args, yaml_output=False, combined_output=False):
        return self._exec('migrate',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def reboot(self, args, yaml_output=False, combined_output=False):
        return self._exec('reboot',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def event_list(self, args, yaml_output=True, combined_output=False):
        return self._exec('event list',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def event_show(self, args, yaml_output=True, combined_output=False):
        return self._exec('event show',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)


class Flavor(OpenStackResource):
    _resource = 'flavor'


class Stack(OpenStackResource):
    _resource = 'stack'

    def resource_list(self, args, yaml_output=False, combined_output=False):
        return self._exec('resource list',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def event_list(self, args, yaml_output=False, combined_output=False):
        return self._exec('event list',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def resource_show(self, args, yaml_output=True, combined_output=False):
        return self._exec('resource show',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def output_show(self, args, yaml_output=True, combined_output=False):
        return self._exec('output show',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def exists(self, name):
        stacks = self.list([])
        match_stacks = [stack['Stack Name'] for stack in stacks if stack['Stack Name'] == name]
        if len(match_stacks) == 1:
            return True
        if len(match_stacks) > 1:
            raise Exception(f"More than one stack {name} found.")
        return False


class ResourceProvider(OpenStackResource):
    _resource = "resource provider"


class ResourceProviderUsage(OpenStackResource):
    _resource = "resource provider usage"


class Network(OpenStackResource):
    _resource = 'network'


class NetworkAgent(OpenStackResource):
    _resource = 'network agent'


class Host(OpenStackResource):
    _resource = 'host'


class Notification(OpenStackResource):
    _resource = 'notification'


class Hypervisor(OpenStackResource):
    _resource = 'hypervisor'


class ComputeService(OpenStackResource):
    _resource = 'compute service'


class Subnet(OpenStackResource):
    _resource = 'subnet'


class Segment(OpenStackResource):
    _resource = 'segment'

    def host_create(self, args, yaml_output=False, combined_output=False):
        return self._exec('host create',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def host_update(self, args, yaml_output=False, combined_output=False):
        return self._exec('host update',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def host_list(self, args, yaml_output=True, combined_output=False):
        return self._exec('host list',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)


class Volume(OpenStackResource):
    _resource = 'volume'


class VolumeService(OpenStackResource):
    _resource = 'volume service'


class Secret(OpenStackResource):
    _resource = 'secret'


class Quota(OpenStackResource):
    _resource = 'quota'


class Console(OpenStackResource):
    _resource = 'console'

    def log_show(self, args, combined_output=False):
        return self._exec('log show',
                          *args,
                          yaml_output=False,
                          combined_output=combined_output)


class Keypair(OpenStackResource):
    _resource = 'keypair'

    def wait_absent(self, name_or_id, timeout=60, interval=10):
        started = time.time()
        while [res for res in self.list([]) if res["Name"] == name_or_id]:
            if int(time.time()) - int(started) > timeout:
                raise TimeoutError(f"Timed out waiting {self._resource} {name_or_id} is deleted")
            time.sleep(interval)


class Port(OpenStackResource):
    _resource = 'port'


class Floatingip(OpenStackResource):
    _resource = 'floating ip'


class Loadbalancer(OpenStackResource):
    _resource = 'loadbalancer'

    def amphora_list(self, args, yaml_output=False, combined_output=False):
        return self._exec('amphora list',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def amphora_show(self, args, yaml_output=True, combined_output=False):
        return self._exec('amphora show',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def pool_list(self, args, yaml_output=True, combined_output=False):
        return self._exec('pool list',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def status_show(self, args, combined_output=False):
        return self._exec('status show',
                          *args,
                          yaml_output=False,
                          combined_output=combined_output)


class Baremetal(OpenStackResource):
    _resource = 'baremetal'

    def node_create(self, args, yaml_output=False, combined_output=False):
        return self._exec('node create',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def node_delete(self, args, yaml_output=False, combined_output=False):
        return self._exec('node delete',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def node_list(self, args, yaml_output=True, combined_output=False):
        return self._exec('node list',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def node_manage(self, args, yaml_output=False, combined_output=False):
        return self._exec('node manage',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)

    def node_provide(self, args, yaml_output=False, combined_output=False):
        return self._exec('node provide',
                          *args,
                          yaml_output=yaml_output,
                          combined_output=combined_output)
