import pytest

from si_tests import logger
from si_tests import settings

LOG = logger.logger


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
def test_upgrade_tf(kaas_manager):
    """Upgrade tf control and data planes
    Scenario:
        1. Get tfoperator
        2. Update control plane over patching tfoperator with new tf version
        3. Wait pods are running in tf namespace
        4. Start dataplane updates: get vrouter-agent pods
        5. Delete vrouter-agent pods
        6. Wait pods in tf namespace are running
        7. Check cluster readiness

    Required environment variables:
        KUBECONFIG : config to access KaaS cluster
        TARGET_NAMESPACE : child cluster namespace in KaaS
        TARGET_CLUSTER : child cluster name in KaaS
        OPENSTACK_UPDATE_TF_VERSION : version to update
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    tf_update_version = settings.OPENSTACK_UPDATE_TF_VERSION
    operator_name = 'openstack-tf'
    operator_ns = 'tf'
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)
    tfoperator = child_cluster.k8sclient.tfoperator_v2.get(
        name=operator_name, namespace=operator_ns)
    LOG.info(f"tf operator {tfoperator}")
    # Patch operator spec
    tfoperator.patch({"spec":
                      {"settings":
                       {"tfVersion": tf_update_version}}})
    LOG.info(f'Tf operator spec after patch {tfoperator.data["spec"]}')
    # check pods are running after control plane update
    child_cluster.check.check_k8s_pods(
        target_namespaces=operator_ns, pods_prefix='tf')
    # Update data plane (need to delete vrouter-agent pods wait when ready)
    pods = child_cluster.k8sclient.pods.list(
        namespace=operator_ns, name_prefix='tf-vrouter-agent')
    # Delete vrouter agent pods and wait rescheduling
    LOG.info("Delete vrouter-agent pods and check they are rescheduled")
    for pod in pods:
        child_cluster.k8sclient.pods.api.delete_namespaced_pod(
            name=pod.name, namespace=operator_ns)
    LOG.info('wait pods rescheduled and running')
    upd_pods = child_cluster.k8sclient.pods.list(
        namespace=operator_ns, name_prefix='tf-vrouter-agent')
    for pod in upd_pods:
        pod.wait_phase('Running')
    child_cluster.check.check_k8s_pods(
        target_namespaces=operator_ns, pods_prefix='tf')
    child_cluster.check.check_cluster_readiness()
