import pytest

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Cluster, Machine, Manager  # noqa: F401
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.utils import waiters
from si_tests.utils import packaging_version as version

LOG = logger.logger


def wait_ceph_status(cluster: Cluster, expected_status="HEALTH_OK"):
    ceph_health_details = cluster.check.get_ceph_status_details()
    current_status = ceph_health_details.get('health', {}).get('status')
    if current_status == expected_status:
        LOG.info(f"Expected Ceph status {current_status} equal to current")
        return True
    else:
        LOG.info(f"Expected Ceph status = {expected_status}, but current = {current_status}. Waiting...")
        return False


def is_precaching_required(master):
    LOG.info("Checking whether to wait image-precaching or not")
    precaching_required_labels = ['openstack-', 'openvswitch', 'tf']
    master_labels = master.get_k8s_node().data.get('metadata', {}).get('labels', {}).keys()
    for label in master_labels:
        for role in precaching_required_labels:
            if label.startswith(role):
                LOG.info("image-precaching check will be required")
                return True
    LOG.info("image-precaching check is not required")
    return False


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('mcc_loadtest_prometheus')
@pytest.mark.usefixtures('mcc_loadtest_grafana')
@pytest.mark.usefixtures('mcc_loadtest_alerta')
@pytest.mark.usefixtures('mcc_loadtest_keycloak')
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
def test_replace_master_node(kaas_manager: Manager, show_step):
    """Replace BM control plane node.

    Scenario:
        1. Find master node where VIP hosted
        2. Collect data about TF if it is enabled (optional)
        3. Replace master Machine
        4. Add new node to the Ceph cluster
        5. Delete all Pending openstack pvc for pods (optional)
        6. Cleanup old node TF data (pvc, pv, configdb, analyticsdb) (optional)
        7. Check OpenStack readiness (optional)
        8. Check cluster readiness
    """

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = kaas_manager.get_namespace(namespace_name)
    LOG.info("Namespace name - %s", namespace_name)

    cluster = ns.get_cluster(cluster_name)
    LOG.info("Cluster name - %s", cluster_name)

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    show_step(1)
    master = cluster.get_keepalive_master_machine()
    check_precaching = is_precaching_required(master)

    show_step(2)
    tf_data = cluster.day2operations.collect_tf_data_before_replace(master)

    show_step(3)
    old_k8s_node_name = master.get_k8s_node_name()
    new_machine = cluster.day2operations.replace_baremetal_machine_and_bmh(master)
    cluster.check.check_deleted_node(old_k8s_node_name)
    # Waiting for machines are Ready
    cluster.check.check_machines_status(timeout=1800)
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()

    new_master_node_name = new_machine.get_k8s_node_name()
    # Check that Machine hostname is created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines(machines=[new_machine])
    # Check machine runtime
    if settings.DESIRED_RUNTIME:
        cluster.check.compare_machines_runtime_with_desired([new_machine])

    show_step(4)
    if cluster.is_child:
        LOG.info("Ceph cluster re-balancing may take some time. Wait timeout is 1h.")
        # Waiting for actual Ceph status from Ceph tools pod
        if settings.CEPH_EXEC_CRASH_ARCHIVE_ALL:
            # exec 'ceph crash archive-all' to wipe info about last crash warnings.
            cluster.check.exec_ceph_tools_command('ceph crash archive-all', return_json=False,
                                                  raise_on_fail=True)
        LOG.info("Wait Ceph HEALTH_OK status in Ceph tools")
        waiters.wait(lambda: wait_ceph_status(cluster), timeout=3600, interval=30)
        # Wait until KaaS update Cluster kind with Ceph status
        LOG.info("Wait Ceph HEALTH_OK status in cluster object")
        try:
            health_info = cluster.check.get_ceph_health_detail()
            assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                         f'Current ceph health status: {health_info}'
        except AssertionError:
            cluster.check.wait_ceph_health_status(timeout=600, interval=30)

    if cluster.clusterrelease_version.startswith(settings.MOSK_RELEASE_PREFIX) \
            and cluster.is_os_deployed():
        child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
        with open('child_conf', 'w') as f:
            f.write(child_kubeconfig)
        os_manager = OpenStackManager(kubeconfig='child_conf')
        if check_precaching:
            LOG.info("Wait image-precaching Ready")
            selector = f"spec.nodeName={new_master_node_name}"

            image_precaching_pod_timeout = 1800
            image_precaching_pod_timeout_msg = f"Image-precaching pod not found on node {new_master_node_name}"
            cluster.k8sclient.pods.wait_pod_present(timeout=image_precaching_pod_timeout,
                                                    interval=15,
                                                    timeout_msg=image_precaching_pod_timeout_msg,
                                                    namespace="openstack",
                                                    name_prefix='image-precaching',
                                                    field_selector=selector)
            image_precaching_pod = cluster.k8sclient.pods.list(namespace="openstack",
                                                               name_prefix='image-precaching',
                                                               field_selector=selector)
            assert image_precaching_pod, f"Image-precaching pod not found on node {new_master_node_name}"
            image_precaching_pod = image_precaching_pod[0]
            image_precaching_pod.wait_ready(timeout=5400)

        if version.parse(cluster.clusterrelease_version) < version.parse('mosk-17-1-0-rc-24-1'):
            show_step(5)  # Delete all Pending pvc for pods
            cluster.delete_pending_openstack_pods()

        cluster.day2operations.cleanup_ovn_db_inactive_members(new_master_node_name)

        if new_machine.has_nodelabels([{'key': 'openstack-control-plane', 'value': 'enabled'}]):
            LOG.info("Perform actions for openstack control plane role")
            cluster.day2operations.recreate_octavia_resources_and_check(
                os_manager, old_k8s_node_name, new_master_node_name)

        show_step(6)  # Cleanup old node TF data (pvc, pv, configdb, analyticsdb)
        cluster.day2operations.tf_data_cleanup(tf_data, old_k8s_node_name)

        show_step(7)
        LOG.info("Wait osdpl health status=Ready")
        os_manager.wait_openstackdeployment_health_status(timeout=1800)
        LOG.info("Wait os jobs to success and pods to become Ready")
        os_manager.wait_os_resources(timeout=1800)

    show_step(8)
    # Check/wait for correct docker service replicas in cluster
    ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
    cluster.check.check_actual_expected_docker_services(
        changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name})
    cluster.check.check_k8s_pods()
    cluster.check.check_actual_expected_pods(timeout=3200)
    cluster.check.check_cluster_readiness()
    cluster.check.check_diagnostic_cluster_status()
    cluster.check.check_deploy_stage_success()

    cluster.check.check_bmh_inventory_presense()
