import pytest

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Machine
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.utils import waiters

LOG = logger.logger


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")  # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('post_action_stop_mm_mode')
def test_reboot_machine(kaas_manager, show_step):
    """Reboot master/worker machine test

    Scenario:
            1. Initiate cluster maintenance
            2. Verify machine(s) reboot
            3. Stop cluster maintenance
            4. Check cluster readiness
            5. Exit from mm mode (post action)

    """

    managed_ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = managed_ns.get_cluster(settings.TARGET_CLUSTER)

    master = cluster.get_keepalive_master_machine()
    worker_machines: list[Machine] = cluster.get_machines(machine_type='worker')

    target_machines = [master] + worker_machines

    show_step(1)
    cluster.cluster_maintenance_start()

    show_step(2)
    for machine in target_machines:
        LOG.info(f"Verify machine '{machine.name}'")
        LOG.info(f"k8s node {machine.get_k8s_node_name()}")
        _verify_reboot(managed_ns, machine)
        ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
        cluster.check.check_actual_expected_docker_services(
            changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name})
        LOG.info(f"Reboot test machine '{machine.name}' done\n\n")

    show_step(3)
    cluster.cluster_maintenance_stop()
    #  Check runtime
    if settings.DESIRED_RUNTIME:
        cluster.check.compare_machines_runtime_with_desired(target_machines)

    show_step(4)
    if cluster.is_mosk:
        LOG.info("Check OpenStack resources")
        child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
        with open('child_conf', 'w') as f:
            f.write(child_kubeconfig)
        os_manager = OpenStackManager(kubeconfig='child_conf')
        os_manager.wait_os_resources(timeout=600, interval=20)
        if cluster.tf_enabled():
            cluster.mos_check.check_cassandra_nodes_config(
                os_manager=os_manager,
                actualize_nodes_config=settings.TF_CASSANDRA_NODES_CLENAUP)
            cluster.mos_check.check_vrouter_pods('tf-vrouter-agent', os_manager=os_manager)

    cluster.check.check_k8s_pods()
    cluster.check.check_cluster_readiness()


def _verify_reboot(managed_ns, machine):
    cluster = managed_ns.get_cluster(settings.TARGET_CLUSTER)
    init_uptime = machine.get_uptime()
    target_machine_node_name = machine.get_k8s_node_name()
    machine.machine_maintenance_start()
    machine.set_baremetalhost_power(online=False)

    machine_ip = machine.public_ip or machine.internal_ip
    LOG.info("Wait until machine is no longer available via ICMP")
    waiters.wait(lambda: not waiters.icmp_ping(machine_ip), interval=5, timeout=600)
    machine.set_baremetalhost_power(online=True)
    LOG.info("Wait until machine is available via SSH")
    waiters.wait_tcp(machine_ip, port=22, timeout=600)

    machine.machine_maintenance_stop()

    def _node_status(expected_status='ready'):
        docker_client = cluster.get_dockerclient()
        node = docker_client.node_inspect(machine_name=target_machine_node_name)[0]
        node_status = node['Status']['State']
        if node_status == expected_status:
            LOG.info(f"Current docker status is {node_status}")
            return True
        else:
            LOG.info(f"Expected docker status is '{expected_status}' but current is '{node_status}'. Waiting...")
            return False

    LOG.info("Check docker node status")
    waiters.wait(_node_status, interval=60, timeout=600)

    LOG.info("Check node reboot count")
    current_uptime = machine.get_uptime()
    assert current_uptime > init_uptime, \
        "Machine '{}' has not been rebooted. Uptime before reboot: {} and after reboot: {}".format(machine.name,
                                                                                                   init_uptime,
                                                                                                   current_uptime)
