import abc
import configparser
import pytest
import yaml

from datetime import datetime
from urllib.parse import urlparse

from si_tests import logger, settings
from si_tests.managers.openstack_client_manager import OpenStackClientManager
from si_tests.managers.osctl_client_manager import OsCtlManager
from si_tests.utils import packaging_version as version

LOG = logger.logger

ROTATION_TS_FORMAT = "%Y-%m-%d %H:%M:%S.%f"


class BaseRotationClass(object):
    def __init__(self, service_name, type="service"):
        self.ocm = OpenStackClientManager()
        self.osm = self.ocm.os_manager
        self.osctlm = OsCtlManager()
        self.service_name = service_name
        self.type = type
        self.rotation_start_ts = None

    def rotate(self, creds_types):
        types = ""
        for t in creds_types:
            types += f" --type {t}"

        # Triggering the procedure of admin's or(and) services' passwords rotation
        cmd = f"osctl credentials rotate --osdpl {settings.OSH_DEPLOYMENT_NAME} {types} --wait"

        self.rotation_start_ts = datetime.strftime(datetime.utcnow(), ROTATION_TS_FORMAT)
        LOG.info(f'Starting "{creds_types}" credentials rotation at {self.rotation_start_ts}')
        LOG.info(f"Executing cmd '{cmd}'")
        self.osctlm.exec(cmd, timeout=settings.OPENSTACK_PASSWORD_ROTATION_TIMEOUT)
        # Check statuses
        self.osm.wait_os_deployment_status(timeout=settings.OPENSTACK_PASSWORD_ROTATION_TIMEOUT)
        # Also we need to check osdplst of all services, because they can still be in IN_PROGRESS state
        LOG.info("Wait osdpl health status=Ready")
        self.osm.wait_openstackdeployment_health_status(timeout=1800)
        LOG.info("Wait osdplst all services health status=Ready")
        LOG.info(f"{creds_types} credentials rotation has been completed")

    @abc.abstractmethod
    def get_current_credentials(self):
        pass

    def _timestamps_supported(self):
        return version.parse(self.osm.os_controller_version()) >= version.parse("0.15.0")

    def _compare_timestamps(self, old, new):
        old_unix_ts = datetime.strptime(old, ROTATION_TS_FORMAT)
        new_unix_ts = datetime.strptime(new, ROTATION_TS_FORMAT)
        return new_unix_ts > old_unix_ts

    def get_rotation_timestamps(self, creds_types):
        if not self._timestamps_supported():
            LOG.info(f"Skip getting rotation timestamps on version {self.osm.os_controller_version()}")
            return {}
        timestamps = {}
        for cr_type in creds_types:
            ts = self.osm.get_credentials_rotation_timestamp(cr_type)
            assert ts, f"{cr_type} credentials rotation timestamp not found"
            timestamps[cr_type] = ts
        return timestamps

    def verify_login_with_credentials(self, credentials):
        # Verifies logging in with the taken admin credentials
        if "identity" in credentials.keys():
            self.verify_keystone_login(credentials["identity"])

        # Checking DB login with the current credentials
        if "database" in credentials.keys():
            self.verify_database_login(credentials["database"])

        # Checking Rabbitmq login with the current credentials
        if "messaging" in credentials.keys():
            self.verify_messaging_login(credentials["messaging"])

    @abc.abstractmethod
    def verify_credentials(self, old_credentials, new_credentials):
        pass

    def verify_keystone_login(self, credentials):

        def _get_os_client_cmd(username, password,
                               unset=False, user_domain_name=None, project_name=None, project_domain_name=None):
            cmd = [
                   "PYTHONWARNINGS=ignore::UserWarning openstack",
                   "token",
                   "issue",
                   "--os-username",
                   username,
                   "--os-password",
                   password,
                   "--os-auth-url",
                   "http://keystone-api.openstack.svc.cluster.local:5000/v3",
                   "-f",
                   "yaml"]
            if user_domain_name:
                cmd.extend(["--os-user-domain-name", user_domain_name])
            if project_name:
                cmd.extend(["--os-project-name", project_name])
            if project_domain_name:
                cmd.extend(["--os-project-domain-name", project_domain_name])
            if unset:
                cmd = ["unset", "OS_CLOUD;"] + cmd
            return ' '.join(cmd)

        if self.type == "admin":
            cmd = _get_os_client_cmd(credentials['username'],
                                     credentials['password'])
        else:
            cmd = _get_os_client_cmd(credentials['username'],
                                     credentials['password'],
                                     unset=True,
                                     user_domain_name="service",
                                     project_name="service",
                                     project_domain_name="service")

        command = [
            "/bin/sh",
            "-c",
            cmd
        ]
        response_yaml = self.ocm.exec(command)
        response = yaml.safe_load(response_yaml)
        expected = {"expires", "id", "user_id", "project_id"}
        if not expected.issubset(set(response)):
            raise ValueError(
                "Cannot login on Keystone client pod with current admin credentials"
            )
        LOG.info(
            "Successfully logged in to Keystone client pod with the admin's credentials"
        )

    def verify_database_login(self, credentials):
        cmd = f"mysql --user={credentials['username']} -p{credentials['password']} --execute='select 1;'"
        response = self.osm.run_mysql_command(cmd)
        if "1" not in response:
            raise ValueError("Cannot login to DB with current password")
        LOG.info("Successfully logged in to DB with the admin password")

    def verify_messaging_login(self, credentials):
        # TODO(vsaienko): OpenStack service might be deployed with specific rmq.
        # Add a test to check specific rmq instance for service if used.
        cmd = f'rabbitmqctl authenticate_user {credentials["username"]} {credentials["password"]}'
        response = self.osm.run_bash_command(
            "openstack-rabbitmq-rabbitmq-0", cmd, container="rabbitmq"
        )
        if "Success" not in response:
            raise ValueError("Cannot login to rabbitmq with current password")
        LOG.info("Successfully logged in to Rabbitmq with the admin password")

    def verify_rotation_timestamps(self, creds_types, old_timestamps, new_timestamps):
        if not self._timestamps_supported():
            LOG.info(f"Skip verifying rotation timestamps on version {self.osm.os_controller_version()}")
            return
        for creds_type in creds_types:
            assert self._compare_timestamps(self.rotation_start_ts, new_timestamps[creds_type]), \
                f"New {creds_type} credentials rotation timestamp is not greater than rotation start timestamp"
            assert self._compare_timestamps(old_timestamps[creds_type], new_timestamps[creds_type]), \
                f"New {creds_type} credentials rotation timestamp is not greater than old one"

    def verify_rotation_timestamps_old_last(self, creds_types, old_timestamps, new_timestamps):
        if not self._timestamps_supported():
            LOG.info(f"Skip verifying rotation timestamps on version {self.osm.os_controller_version()}")
            return
        for creds_type in creds_types:
            assert self._compare_timestamps(old_timestamps[creds_type], new_timestamps[creds_type]), \
                f"Last {creds_type} credentials rotation timestamp is not greater than one before rotations"


class AdminRotationClass(BaseRotationClass):
    def get_current_credentials(self):
        # Getting admin's credentials
        # Getting admin's DB password
        admin_db_password = self.osm.get_db_password()

        # Getting admin's password for Rabbitmq
        admin_messaging_password = self.osm.get_messaging_password()

        # Getting admin's username & password on Keystone client's pod in etc/openstack/clouds.yaml file
        config_path = "/etc/openstack/clouds.yaml"
        cmd = f"cat {config_path}"
        service_config = self.ocm.exec(["bash", "-c", cmd])

        clouds_yaml = yaml.safe_load(service_config)
        LOG.info(f"Clouds.yaml data:\n{yaml.dump(clouds_yaml)}")
        LOG.info("Getting admin's username & password")
        cloud_auth = clouds_yaml["clouds"]["admin"]["auth"]
        admin_username = cloud_auth["username"]
        admin_password = cloud_auth["password"]
        LOG.info("Admin's username & password have been taken")
        current_creds = {
            "identity": {"username": admin_username, "password": admin_password},
            "database": {"username": "root", "password": admin_db_password},
            "messaging": {"username": "rabbitmq", "password": admin_messaging_password},
        }
        LOG.info(f"Current credentials:\n{yaml.dump(current_creds)}")

        return current_creds

    def verify_credentials(self, old_credentials, new_credentials):
        assert (
            old_credentials["identity"]["username"]
            != new_credentials["identity"]["username"]
        ), "Identity Admin username is not changed"
        assert (
            old_credentials["identity"]["password"]
            != new_credentials["identity"]["password"]
        ), "Identity Admin password is not changed"

        assert (
            old_credentials["database"]["username"]
            == new_credentials["database"]["username"]
        ), "Database Admin username is changed"
        assert (
            old_credentials["database"]["password"]
            != new_credentials["database"]["password"]
        ), "Database Admin password is not changed"

        assert (
            old_credentials["messaging"]["username"]
            == new_credentials["messaging"]["username"]
        ), "Messaging Admin username is changed"
        assert (
            old_credentials["messaging"]["password"]
            != new_credentials["messaging"]["password"]
        ), "Messaging Admin password is not changed"

        LOG.info("Verifying logging in with the new admin credentials")
        self.verify_login_with_credentials(new_credentials)

        LOG.info("Verifying that logging in with the old admin credentials still works")
        self.verify_keystone_login(old_credentials["identity"])

    def verify_credentials_old_last(self, old_credentials, last_credentials):
        assert (
            old_credentials["identity"]["username"]
            == last_credentials["identity"]["username"]
        ), "Identity Admin username after the 2nd rotation doesn't equal to the one before rotations"
        assert (
            old_credentials["identity"]["password"]
            != last_credentials["identity"]["password"]
        ), "Identity Admin password after the 2nd rotation equals to the one before rotations"

        assert (
            old_credentials["database"]["password"]
            != last_credentials["database"]["password"]
        ), "Database Admin password after the 2nd rotation equals to the one before rotations"

        assert (
            old_credentials["messaging"]["password"]
            != last_credentials["messaging"]["password"]
        ), "Messaging Admin password after the 2nd rotation equals to the one before rotations"


class ServiceCredentialRotationClass(BaseRotationClass):
    def __init__(self, service_name, pod_name_prefix, uses_messaging=True, uses_keystone=True,
                 config_path=None, db_config_section="database", db_connection_option="connection"):
        super().__init__(service_name)
        self.pod_name_prefix = pod_name_prefix
        self.config_path = (
            config_path or f"/etc/{self.service_name}/{service_name}.conf"
        )
        self.uses_messaging = uses_messaging
        self.uses_keystone = uses_keystone
        self.db_config_section = db_config_section
        self.db_connection_option = db_connection_option

    def get_service_config(self):
        pod_name = (
            self.osm.api.pods.list_starts_with(
                self.pod_name_prefix, self.osm.openstack_namespace
            )[0]
            .read()
            .metadata.name
        )
        cfg = self.osm.run_bash_command(pod_name, f"cat {self.config_path}")
        config = configparser.RawConfigParser(strict=False)
        config.read_string(cfg)
        return config

    def get_keystone_credentials(self, config):
        cfg = dict(config["keystone_authtoken"])
        result = {"username": cfg["username"], "password": cfg["password"]}
        return result

    def get_oslo_cache_credentials(self, config):
        cfg = dict(config["keystone_authtoken"])
        result = {"memcache_secret_key": cfg["memcache_secret_key"]}
        return result

    def get_database_credentials(self, config):
        connection_url = config[self.db_config_section][self.db_connection_option]
        url_parse = urlparse(connection_url)
        return {"username": url_parse.username, "password": url_parse.password}

    def get_messaging_credentials(self, config):
        transport_url = config["DEFAULT"]["transport_url"]
        url_parse = urlparse(transport_url)
        return {"username": url_parse.username, "password": url_parse.password}

    def get_current_credentials(self):
        config = self.get_service_config()
        LOG.info(f"Getting credentials for {self.service_name} service")
        credentials = {"database": self.get_database_credentials(config)}
        if self.service_name != "keystone":
            credentials["oslo_cache"] = self.get_oslo_cache_credentials(config)
        if self.uses_keystone:
            credentials["identity"] = self.get_keystone_credentials(config)
        if self.uses_messaging:
            credentials["messaging"] = self.get_messaging_credentials(config)
        return credentials

    def verify_credentials(self, old_credentials, new_credentials):
        # TODO(vsaienko): Add verification for other credentials like
        # * ensure ipsec key is not changed for neutron
        for creds_name in old_credentials.keys():
            if creds_name != "oslo_cache":
                assert (
                    old_credentials[creds_name]["username"]
                    != new_credentials[creds_name]["username"]
                ), f"Username is not changed for {creds_name}"
                assert (
                    old_credentials[creds_name]["password"]
                    != new_credentials[creds_name]["password"]
                ), f"Password is not changed for {creds_name}"
            else:
                assert (old_credentials["oslo_cache"]["memcache_secret_key"]
                        != new_credentials["oslo_cache"]["memcache_secret_key"]), "Memcache secret key after the 1nd"\
                                                                                  " rotation is the same as before"\
                                                                                  " rotations"

        LOG.info("Verifying logging in with the new credentials")
        self.verify_login_with_credentials(new_credentials)

        LOG.info("Verifying that logging in with the old credentials still works")
        self.verify_login_with_credentials(old_credentials)

    def verify_credentials_old_last(self, old_credentials, last_credentials):
        for creds_name in old_credentials.keys():
            if creds_name != "oslo_cache":
                assert (
                    old_credentials[creds_name]["username"]
                    == last_credentials[creds_name]["username"]
                ), f"{creds_name} service username after the 2nd rotation equals to the one before rotations"
                assert (
                    old_credentials[creds_name]["password"]
                    != last_credentials[creds_name]["password"]
                ), f"{creds_name} service password after the 2nd rotation equals to the one before rotations"
            else:
                assert (
                        old_credentials["oslo_cache"]["memcache_secret_key"]
                        == last_credentials["oslo_cache"]["memcache_secret_key"]), "Memcache secret key after the 2nd"\
                                                                                   " rotation isn't the same as before"\
                                                                                   " rotations"


class AodhServiceCredentialRotationClass(ServiceCredentialRotationClass):
    def verify_credentials(self, old_credentials, new_credentials):
        identity_old = old_credentials.pop("identity")
        identity_new = new_credentials.pop("identity")
        assert identity_old["username"] == identity_new["username"], "Username is changed for Aodh identity"
        assert identity_old["password"] != identity_new["password"], "Password isn't changed for Aodh identity"

        LOG.info("Verifying logging in with the new credentials")
        self.verify_keystone_login(identity_new)

        super().verify_credentials(old_credentials, new_credentials)

        old_credentials["identity"] = identity_old
        new_credentials["identity"] = identity_new


class HeatServiceCredentialRotationClass(ServiceCredentialRotationClass):
    def get_current_credentials(self):
        cfg = dict(self.get_service_config()["trustee"])
        trustee = {"username": cfg["username"],
                   "password": cfg["password"],
                   }
        credentials = super().get_current_credentials()
        credentials.update({"trustee": trustee})
        return credentials

    def verify_credentials(self, old_credentials, new_credentials):
        trustee_old = old_credentials.pop("trustee")
        trustee_new = new_credentials.pop("trustee")
        assert trustee_old["username"] == trustee_new["username"], "Username was changed for Heat trustee"
        assert trustee_old["password"] != trustee_new["password"], "Password wasn't changed for Heat trustee"

        LOG.info("Verifying logging in with the new trustee credentials")
        self.verify_keystone_login(trustee_new)

        super().verify_credentials(old_credentials, new_credentials)

        old_credentials["trustee"] = trustee_old
        new_credentials["trustee"] = trustee_new


@pytest.mark.usefixtures('mos_workload_downtime_report')
@pytest.mark.usefixtures('mos_loadtest_os_refapp')  # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('mos_per_node_workload_check_after_test')
def test_credentials_rotation(os_manager):
    """Verifies rotation procedure for admin, service credentials
    Parameters required for the test execution:
        - KUBECONFIG
    Scenario:
        1. Get old credentials
        1.1 Get old credentials rotation timestamp
        2. Check logins with old credentials
        3. Rotate service, admin credentials simulteniously
        4. Get new credentials
        5. Check login with new credentials
        5.1 Check login for admin keystone with old credentials
        5.2 Check new credentials rotation timestamp is greater than old one and greater than rotation start
            timestamp.
        6. Do another rotation service (for service creds check)
        6.1 Take last credentials. On this stage we have old credentials (creds before rotation),
            new credentials (creds after 1st rotation), last credentials (creds after 2nd rotation)
        6.2 Old credentials should be invalid.
        6.3 Old credentials.username == last credentials.username
            old credentials.password != last credentials.password
        6.4 Last credentials are valid
        6.6 Heat/Aodh (old creds.username == new creds.username == last creds.username)
        6.7 Check last credentials rotation timestamp > old credentials rotation timestamp
            Check last credentials rotation timestamp > new credentials rotation timestamp
            Check last credentials rotation timestamp > rotation start timestamp
    """
    # TODO(tleontovich) Delete after https://mirantis.jira.com/browse/FIELD-6500 fixed
    os_manager.wr_field_6500()

    services = [
        AdminRotationClass("admin", type="admin"),
        ServiceCredentialRotationClass("barbican", "barbican-api", db_config_section="DEFAULT",
                                       db_connection_option="sql_connection"),
        ServiceCredentialRotationClass("cinder", "cinder-volume-0"),
        ServiceCredentialRotationClass("designate", "designate-api-0"),
        ServiceCredentialRotationClass("glance", "glance-api", config_path="etc/glance/glance-api.conf"),
        HeatServiceCredentialRotationClass("heat", "heat-api"),
        ServiceCredentialRotationClass("keystone", "keystone-api", uses_keystone=False),
        ServiceCredentialRotationClass("neutron", "neutron-server"),
        ServiceCredentialRotationClass("nova", "nova-conductor-0"),
        ServiceCredentialRotationClass("octavia", "octavia-api"),
        ServiceCredentialRotationClass("placement", "placement-api", uses_messaging=False,
                                       db_config_section="placement_database"),
    ]
    osdpl_services = os_manager.get_osdpl_deployment().data["spec"]["features"]["services"]
    if "alarming" in osdpl_services:
        services.append(AodhServiceCredentialRotationClass("aodh", "aodh-api"))
    if "metering" in osdpl_services:
        services.append(ServiceCredentialRotationClass("ceilometer", "ceilometer-central"))
    if "metric" in osdpl_services:
        services.append(ServiceCredentialRotationClass("gnocchi", "gnocchi-api", uses_messaging=False))
    if "baremetal" in osdpl_services:
        services.append(ServiceCredentialRotationClass("ironic", "ironic-conductor-0"))
    if "shared-file-system" in osdpl_services:
        services.append(ServiceCredentialRotationClass("manila", "manila-scheduler-0"))
    if "instance-ha" in osdpl_services:
        services.append(ServiceCredentialRotationClass("masakari", "masakari-api"))
    old_credentials = {}
    new_credentials = {}
    last_credentials = {}
    old_rotation_timestamps = {}
    new_rotation_timestamps = {}
    last_rotation_timestamps = {}

    for service in services:
        old_credentials[service.service_name] = service.get_current_credentials()

        LOG.info("Verifying logging in with the old credentials")
        service.verify_login_with_credentials(old_credentials[service.service_name])

    # Starting 1st credentials rotation
    old_rotation_timestamps = services[0].get_rotation_timestamps(["admin", "service"])
    services[0].rotate(["admin", "service"])
    new_rotation_timestamps = services[0].get_rotation_timestamps(["admin", "service"])

    for service in services:
        new_credentials[service.service_name] = service.get_current_credentials()

        service.verify_credentials(
            old_credentials[service.service_name], new_credentials[service.service_name]
        )
    services[0].verify_rotation_timestamps(["admin", "service"], old_rotation_timestamps, new_rotation_timestamps)

    # Starting 2nd credentials rotation
    services[0].rotate(["admin", "service"])
    last_rotation_timestamps = services[0].get_rotation_timestamps(["admin", "service"])

    for service in services:
        last_credentials[service.service_name] = service.get_current_credentials()

        service.verify_credentials_old_last(
            old_credentials[service.service_name], last_credentials[service.service_name]
        )

        service.verify_credentials(
            new_credentials[service.service_name], last_credentials[service.service_name]
        )

        try:
            service.verify_login_with_credentials(old_credentials)
        except ValueError:
            LOG.info(f"Old {service.service_name} service credentials don't work after the second rotation as planned")
    services[0].verify_rotation_timestamps_old_last(["admin", "service"],
                                                    old_rotation_timestamps,
                                                    last_rotation_timestamps)
    services[0].verify_rotation_timestamps(["admin", "service"],
                                           new_rotation_timestamps,
                                           last_rotation_timestamps)
