import pytest
from kubernetes.client.rest import ApiException

from si_tests import logger, settings
from si_tests.utils import packaging_version as version
from si_tests.utils import waiters

LOG = logger.logger


@pytest.mark.usefixtures('restore_tfoperator_cr')
class TestTFAdmissionController(object):
    """Test TFOperator admission controller"""
    helmbundle_name = "tungstenfabric-operator"

    @pytest.fixture(scope='class', autouse=True)
    def restore_helmbundle(self, tf_manager):
        LOG.info("Backup helmbundle specs")
        spec_backup = tf_manager.get_tf_helmbundle().data['spec']
        yield
        LOG.info("Restore helmbundle specs")
        tf_manager.get_tf_helmbundle().patch({"spec": spec_backup})

    @pytest.fixture(autouse=True)
    def setup(self, tf_manager):
        self.validating_webhook = (f"{self.helmbundle_name}-validating-"
                                   f"webhook-configuration")
        self.validating_webhook_v2 = (f"{self.helmbundle_name}-v2-validating-"
                                      f"webhook-configuration")
        self.mutating_webhook = (f"{self.helmbundle_name}-mutating-webhook-"
                                 f"configuration")
        self.mutating_webhook_v2 = (f"{self.helmbundle_name}-v2-mutating-"
                                    f"webhook-configuration")

        self.api_admission = tf_manager.api.api_admission
        self.tf_helmbundle = tf_manager.get_tf_helmbundle()
        self.tf_manager = tf_manager
        self.tf_deployment = tf_manager.get_tfoperator_deployment()
        self.tfoperator_version = version.parse(tf_manager.tfoperator_version())
        self.tfoperator_cr = tf_manager.tfoperator(detect_api=True)

        self.set_admission_ctl(enabled=True)

    def set_admission_ctl(self, enabled=True):
        spec = self.tf_helmbundle.data['spec']
        chart_name = 'tungstenfabric-operator'
        for chart in spec['releases']:
            if chart['name'] == chart_name:
                values = chart['values']
                if values.get('admission', {}).get('enabled', True) != enabled:
                    values['admission'] = {'enabled': enabled}
                    gen = self.tf_deployment.read().status.observed_generation
                    self.tf_helmbundle.patch({"spec": spec})
                    waiters.wait(self.check_deployment, timeout=180,
                                 predicate_args=[gen])
                    waiters.wait(self.is_tf_deployment_ready, timeout=180)
                return
        raise Exception(f"Chart '{chart_name}' not found.")

    def check_deployment(self, gen):
        tf_deployment = self.tf_deployment.read()
        observed_generation = tf_deployment.status.observed_generation
        LOG.info(f"Wait {self.tf_deployment} deployment will be "
                 f"updated. Current generation: {observed_generation}")
        return observed_generation > gen

    def is_tf_deployment_ready(self):
        tf_deployment = self.tf_deployment.read()
        replicas = tf_deployment.spec.replicas
        ready_replicas = tf_deployment.status.ready_replicas
        LOG.info(f"Wait {self.tf_deployment} deployment replicas are ready: "
                 f"{ready_replicas}/{replicas}")
        return ready_replicas == replicas

    @staticmethod
    def check_webhook(func, *args, expected=True):
        try:
            func(*args)
            LOG.info("Webhook is present")
            return True and expected
        except ApiException as e:
            if e.status == 404:
                LOG.info("Webhook not found")
                return not expected

    def check_validating_webhook(self, expected=True):
        LOG.info(f'Check Validating Webhook {self.validating_webhook}')
        return self.check_webhook(
            self.api_admission.read_validating_webhook_configuration,
            self.validating_webhook, expected=expected,
        )

    def check_validating_webhook_apiv2(self, expected=True):
        LOG.info(f'Check Validating Webhook {self.validating_webhook_v2}')
        return self.check_webhook(
            self.api_admission.read_validating_webhook_configuration,
            self.validating_webhook_v2, expected=expected,
        )

    def check_mutating_webhook(self, expected=True):
        LOG.info(f'Check Mutating Webhook {self.mutating_webhook}')
        return self.check_webhook(
            self.api_admission.read_mutating_webhook_configuration,
            self.mutating_webhook, expected=expected,
        )

    def check_mutating_webhook_apiv2(self, expected=True):
        LOG.info(f'Check Mutating Webhook {self.mutating_webhook_v2}')
        return self.check_webhook(
            self.api_admission.read_mutating_webhook_configuration,
            self.mutating_webhook_v2, expected=expected,
        )

    @pytest.mark.skipif(settings.TF_OPERATOR_API_V2,
                        reason="TF Operator CR api v2 is used")
    def test_invalidation_request(self, show_step):
        """Check valid and non-valid requests.

            Scenario:
                1. Check validation webhook
                2. Send and check invalid request
        """

        show_step(1)
        assert self.check_validating_webhook(expected=True), \
            "Validating Admission Webhook isn't created"

        show_step(2)
        invalid_settings = {
            "settings": {
                "tfImage": {
                    "tfImageTag": "5.1.20220101011200"
                },
                "tfVersion": "21.4",
            }
        }
        with pytest.raises(ApiException) as exc_info:
            self.tfoperator_cr.patch({"spec": invalid_settings})
        assert exc_info.value.status == 403
        assert exc_info.value.reason == 'Forbidden'

    @pytest.mark.skipif(settings.TF_OPERATOR_API_V2,
                        reason="TF Operator CR api v2 is used")
    def test_mutation_webhook(self, show_step):
        """Check mutating webhook.

            Scenario:
                1. Check mutating webhook
                2. Set previous TF version
                3. Check TF version was changed to default
        """

        if self.tfoperator_version <= version.parse("0.11"):
            pytest.skip(f"TFOperator version {self.tfoperator_version} doesn't supported")
        default_version = "21.4"
        show_step(1)
        assert self.check_mutating_webhook(expected=True), \
            "Mutating Admission Webhook isn't created"

        show_step(2)
        settings = {
            "settings": {
                "tfVersion": "2011",
            }
        }
        self.tfoperator_cr.patch({"spec": settings})

        show_step(3)
        self.tfoperator_cr.read()
        assert self.tfoperator_cr.data['spec']['settings']['tfVersion'] == \
            default_version

    @pytest.mark.skipif(not settings.TF_OPERATOR_API_V2,
                        reason="TF Operator CR api v2 is not used")
    def test_invalidation_request_apiv2(self, show_step):
        """Check valid and non-valid requests.

            Scenario:
                1. Check validation webhook
                2. Send and check invalid request
        """

        show_step(1)
        assert self.check_validating_webhook_apiv2(expected=True), \
            "Validating Admission Webhook isn't created"

        show_step(2)
        invalid_settings = {
            "devOptions": {
                "tfImage": {
                    "tfImageTag": "2011.20240101011200"
                }
            },
            "tfVersion": "21.4",
        }
        with pytest.raises(ApiException) as exc_info:
            self.tfoperator_cr.patch({"spec": invalid_settings})
        assert exc_info.value.status == 403
        assert exc_info.value.reason == 'Forbidden'

    @pytest.mark.skipif(not settings.TF_OPERATOR_API_V2,
                        reason="TF Operator CR api v2 is not used")
    def test_mutation_webhook_apiv2(self, show_step):
        """Check mutating webhook.

            Scenario:
                1. Check mutating webhook
                2. Set previous TF version
                3. Check TF version was changed to default
        """
        if self.tfoperator_version <= version.parse("0.14"):
            pytest.skip("TFOperator version isn't supported")
        if not self.tf_manager.is_forceUpdate():
            pytest.skip("ForceUpdate is disabled")

        if self.tfoperator_version < version.parse("0.17.2"):
            default_version = "21.4"
            set_version = "2011"
        else:
            default_version = "24.1"
            set_version = "21.4"

        show_step(1)
        assert self.check_mutating_webhook_apiv2(expected=True), \
            "Mutating Admission Webhook isn't created"

        show_step(2)
        self.tfoperator_cr.patch({"spec": {"tfVersion": set_version}})

        show_step(3)
        self.tfoperator_cr.read()
        assert self.tfoperator_cr.data['spec']['tfVersion'] == \
               default_version

    def test_manage_admission_controller(self, tf_manager, show_step):
        """Enable/disable tf-operator admission controller via helmbundle.

            Scenario:
                1. Disable admission controller
                2. Check Validating/Mutating Webhooks Configuration were
                removed
                3. Enable admission controller
                4. Check Validating/Mutating Webhooks Configuration were created
        """
        show_step(1)
        self.set_admission_ctl(enabled=False)

        show_step(2)
        waiters.wait(self.check_validating_webhook, timeout=180,
                     predicate_kwargs={'expected': False})
        waiters.wait(self.check_mutating_webhook, timeout=10,
                     predicate_kwargs={'expected': False})
        if self.tfoperator_version >= version.parse("0.14"):
            waiters.wait(self.check_validating_webhook_apiv2, timeout=10,
                         predicate_kwargs={'expected': False})
            waiters.wait(self.check_mutating_webhook_apiv2, timeout=10,
                         predicate_kwargs={'expected': False})

        show_step(3)
        self.set_admission_ctl(enabled=True)

        show_step(4)
        # Start with TFOperator version 0.17.2 apiv1 was dropped
        if self.tfoperator_version < version.parse("0.17.2"):
            waiters.wait(self.check_validating_webhook, timeout=180,
                         predicate_kwargs={'expected': True})
            waiters.wait(self.check_mutating_webhook, timeout=10,
                         predicate_kwargs={'expected': True})
        if self.tfoperator_version >= version.parse("0.14"):
            waiters.wait(self.check_validating_webhook_apiv2, timeout=10,
                         predicate_kwargs={'expected': True})
            waiters.wait(self.check_mutating_webhook_apiv2, timeout=10,
                         predicate_kwargs={'expected': True})
