import pytest
import yaml
import base64

from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters, utils


LOG = logger.logger

cluster_name = settings.TARGET_CLUSTER
namespace_name = settings.TARGET_NAMESPACE
deploy_image_url = settings.IRONIC_DEPLOY_IMAGE_URL
deploy_image_name = settings.IRONIC_IMAGE_NAME
bm_flavor = settings.IRONIC_FLAVOR_NAME


@pytest.mark.usefixtures('log_method_time')
def test_add_ironic_node(kaas_manager, show_step):
    """Add bm compute node.

    Scenario:
        1. Gather nodes information from yaml file
        2. Get openstack-client pod
        3. Get images UUIDs
        4. Delete nodes and image before creating
        5. Get nodes from child data and start inspecting
        6. Wait for all nodes stay available
        7. Wait for compute services
        8. Run job cell-setup
        9. Wait for new hypervisors stay available
    """

    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    osdpl = cluster.k8sclient.openstackdeploymentstatus.get(name='osh-dev', namespace='openstack')
    openstack_version = osdpl.read().status['osdpl']['openstack_version']
    LOG.info(f"Namespace: {namespace_name}; Cluster: {cluster_name}; Openstack version: {openstack_version}")
    oc_name = "openstack-controller"
    if utils.is_rockoon_used(cluster.k8sclient):
        oc_name = "rockoon"

    show_step(1)
    child_data = utils.render_child_data(kaas_manager.si_config, dict())

    bm_hosts_data = child_data['ironic_nodes']
    show_step(2)
    client_pod = cluster.k8sclient.pods.list(
        namespace="openstack",
        name_prefix='keystone-client')
    assert len(client_pod) > 0, ("No pods found with prefix "
                                 "keystone-client in namespace "
                                 "openstack")
    show_step(2)
    client_pod = client_pod[0]

    cmd_hypervisor_list = ['/bin/sh', '-c',
                           'PYTHONWARNINGS=ignore::UserWarning'
                           ' openstack hypervisor list -f yaml']

    cmp_service_list = ['/bin/sh', '-c',
                        'PYTHONWARNINGS=ignore::UserWarning '
                        'openstack compute service list -f yaml']
    show_step(3)
    LOG.info("Get images UUIDs")
    iku_cmd = ['/bin/sh', '-c', 'PYTHONWARNINGS=ignore::UserWarning '
                                'openstack image list --tag  '
               f"{oc_name}:managed-image --tag "
               f"{oc_name}:baremetal-image-kernel -f yaml"]
    iku_list = yaml.safe_load(client_pod.exec(iku_cmd).strip())
    image_kernel_uuid = [h.get('ID') for h in iku_list if openstack_version in h.get('Name')][0]
    iru_cmd = ['/bin/sh', '-c', 'PYTHONWARNINGS=ignore::UserWarning '
                                'openstack image list --tag  '
               f"{oc_name}:managed-image --tag "
               f"{oc_name}:baremetal-image-ramdisk -f yaml"]
    iru_list = yaml.safe_load(client_pod.exec(iru_cmd).strip())
    image_ramdisk_uuid = [h.get('ID') for h in iru_list if openstack_version in h.get('Name')][0]

    show_step(4)
    resource_class_normalized = ''
    ironic_nodes = []
    # Delete image if exist
    LOG.info(f"Delete image {deploy_image_name}")
    client_pod.exec([
        '/bin/sh', '-c',
        f'PYTHONWARNINGS=ignore::UserWarning '
        f'openstack image delete {deploy_image_name}'],
        stderr=True, _request_timeout=600)
    # Force delete node before creating
    delete_nodes = [n['name'] for n in bm_hosts_data
                    if 'ironic' in n.get('si_roles', [])]
    for _node in delete_nodes:
        client_pod.exec(['/bin/sh', '-c',
                         'PYTHONWARNINGS=ignore::UserWarning '
                         'openstack baremetal node maintenance set'
                         f" {_node} &&"
                         f'PYTHONWARNINGS=ignore::UserWarning '
                         f'openstack baremetal node delete {_node}'])
    cmd_ironic_nodes_list = ['/bin/sh', '-c',
                             'PYTHONWARNINGS=ignore::UserWarning '
                             'openstack baremetal node list -f yaml '
                             '--field uuid provision_state']

    def check_deleted(count=None):
        nodes_count = client_pod.exec(cmd_ironic_nodes_list)
        bm_list = yaml.safe_load(nodes_count)
        bm_stat = len(bm_list)
        LOG.info(f"Count of existing BM: {bm_stat}")
        if bm_stat == count and len(bm_list) == count:
            return True
        else:
            return False
    waiters.wait(
        lambda: check_deleted(count=0),
        timeout=900, interval=30,
        timeout_msg="Some bm nodes is not deleted")

    hyper_count = len(yaml.safe_load(client_pod.exec(cmd_hypervisor_list)))
    show_step(5)
    for _node in bm_hosts_data:
        if 'ironic' in _node.get('si_roles', []):
            ironic_nodes.append(_node)
            ironic_bm_node_name = _node.get('name')
            ipmi_ip = _node['ipmi']['ipmi_ip']
            ipmi_port = _node['ipmi'].get('port', 623)
            ipmi_user = base64.b64decode(
                _node['ipmi']['username']).decode("utf-8")
            ipmi_password = base64.b64decode(
                _node['ipmi']['password']).decode("utf-8")
            pxe_mac = _node['networks'][0]['mac']
            node_mem_mb = _node['ironic_specific']['mem']
            node_cpus = _node['ironic_specific']['cpus']
            node_disk_gb = _node['ironic_specific']['disk']
            physnet = _node['ironic_specific']['physnet']
            multitenantcy = _node.get('ironic_specific')\
                .get('mutitenantcy').get('enabled')
            mt_params = ''
            mt_params_port = ''
            if multitenantcy:
                LOG.info(f"Multitenantcy enabled for <{ironic_bm_node_name}>")
                switch_name = _node['ironic_specific']['mutitenantcy'][
                    'switch_name']
                switch_port = _node['ironic_specific']['mutitenantcy'][
                    'switch_port']
                mt_params = ' --network-interface neutron'
                mt_params_port = \
                    f' --local-link-connection switch_info={switch_name}'\
                    ' --local-link-connection switch_id="FE:FE:FE:FE:FE:FE"'\
                    f' --local-link-connection port_id={switch_port}'
            resource_class = ('baremetal.ram' + str(node_mem_mb) + '.cpus'
                              + str(node_cpus) + '.disk' + str(node_disk_gb))
            resource_class_normalized = resource_class.upper().replace(".", "_")

            cmd_add_bm = ['/bin/sh', '-c',
                          'PYTHONWARNINGS=ignore::UserWarning openstack baremetal node create'
                          ' --driver ipmi'
                          ' --property cpu_arch=x86_64'
                          + mt_params +
                          f' --property memory_mb={node_mem_mb}'
                          f' --property cpus={node_cpus}'
                          f' --property local_gb={node_disk_gb}'
                          f' --driver-info deploy_kernel={image_kernel_uuid}'
                          f' --driver-info deploy_ramdisk={image_ramdisk_uuid}'
                          f' --driver-info ipmi_address={ipmi_ip}'
                          f' --driver-info ipmi_port={ipmi_port}'
                          f' --driver-info ipmi_username={ipmi_user}'
                          f' --driver-info ipmi_password={ipmi_password}'
                          f' --resource-class {resource_class}'
                          f' --name {ironic_bm_node_name} -f yaml']
            LOG.info(f"Create baremetal node <{ironic_bm_node_name}>. "
                     f"Command:\n {cmd_add_bm}\n")
            exec_result = client_pod.exec(cmd_add_bm)
            LOG.debug(exec_result)
            LOG.info("Get created UUID")
            node_uuid = yaml.safe_load(exec_result).get('uuid')

            cmd_add_port = ['/bin/sh', '-c',
                            f'PYTHONWARNINGS=ignore::UserWarning openstack baremetal port create'
                            f' {pxe_mac}'
                            f' --node {node_uuid}'
                            f' --physical-network {physnet}'
                            + mt_params_port]
            LOG.info(f"Create baremetal port for <{ironic_bm_node_name}>")
            client_pod.exec(cmd_add_port)
            # Starting from joga uefi is default so we need to set legacy in right way
            cmd_set_capabilities = ['/bin/sh', '-c',
                                    f'PYTHONWARNINGS=ignore::UserWarning openstack baremetal node set'
                                    f' {ironic_bm_node_name}'
                                    f' --property capabilities=boot_mode:bios']
            cmd_set_deploy_boot = ['/bin/sh', '-c',
                                   f'PYTHONWARNINGS=ignore::UserWarning openstack baremetal node set'
                                   f' {ironic_bm_node_name}'
                                   f' --driver-info deploy_boot_mode=bios']

            LOG.info(f"Set capabilities and deploy boot for <{ironic_bm_node_name}>")
            exec_result = client_pod.exec(cmd_set_capabilities)
            LOG.info(f"result of cmd {cmd_set_capabilities}: {exec_result}")
            exec_result = client_pod.exec(cmd_set_deploy_boot)
            LOG.info(f"result of cmd {cmd_set_deploy_boot}: {exec_result}")

            _node.update({'uuid': node_uuid})
            cmd_provide = ['/bin/sh', '-c',
                           f'PYTHONWARNINGS=ignore::UserWarning openstack baremetal node manage'
                           f' {node_uuid} && '
                           f'PYTHONWARNINGS=ignore::UserWarning '
                           f'openstack baremetal node provide {node_uuid}']
            client_pod.exec(cmd_provide)
            LOG.info("Provide baremetal node")

    assert len(ironic_nodes) >= 1, "BMH with si_roles 'ironic' not found" \
                                   " in child data file"

    LOG.info(f"Download and create image {deploy_image_name}")
    image_cmd_results = client_pod.exec([
        '/bin/sh', '-c',
        f'wget --connect-timeout 4 -t4 -O /tmp/i {deploy_image_url} &&'
        f'PYTHONWARNINGS=ignore::UserWarning '
        f'openstack image create {deploy_image_name} --disk-format qcow2 '
        '--public --container-format bare --file /tmp/i'],
        stderr=True, _request_timeout=900)

    LOG.info("Checking that image is exist")
    image_present = client_pod.exec([
        '/bin/sh', '-c',
        f'PYTHONWARNINGS=ignore::UserWarning '
        f'openstack image show {deploy_image_name} -c name -f value']).strip()
    assert image_present == deploy_image_name, \
        f"Err: Image was not created; Command result: {image_cmd_results}"

    LOG.info(f"Create baremetal flavor <{bm_flavor}>")
    flavor_cmd_results = client_pod.exec([
        '/bin/sh', '-c',
        f'PYTHONWARNINGS=ignore::UserWarning '
        f'openstack flavor create {bm_flavor} --property '
        'resources:VCPU=0 --property resources:MEMORY_MB=0 '
        '--property resources:DISK_GB=0 --property '
        'resources:CUSTOM_'
        f'{resource_class_normalized}=1 --disk 1'], stderr=True)
    LOG.debug(flavor_cmd_results)
    LOG.info("Checking that flavor is exist")
    flavor_present = client_pod.exec([
        '/bin/sh', '-c',
        f'PYTHONWARNINGS=ignore::UserWarning '
        f'openstack flavor show {bm_flavor} -c name -f value']).strip()
    assert flavor_present == bm_flavor, \
        f"Flavor was not created; Command result: {flavor_cmd_results}"

    cmd_ironic_nodes_list = ['/bin/sh', '-c',
                             'PYTHONWARNINGS=ignore::UserWarning '
                             'openstack baremetal node list -f yaml '
                             '--field uuid provision_state']
    exec_node_count = client_pod.exec(cmd_ironic_nodes_list)
    LOG.info(f"Get count of nodes. Command result: \n{exec_node_count}")
    ironic_nodes_count = len(yaml.safe_load(exec_node_count))

    def check_node_count(count=None):
        nodes_count = client_pod.exec(cmd_ironic_nodes_list)
        bm_list = yaml.safe_load(nodes_count)
        bm_stat = sum(h.get('Provisioning State') == 'available'
                      for h in bm_list)
        LOG.info(f"Count ready BM: {bm_stat} "
                 f"expected {ironic_nodes_count}")
        if bm_stat == count and len(bm_list) == count:
            return True
        else:
            return False

    def check_cmp_service_status():
        service_list = yaml.safe_load(client_pod.exec(cmp_service_list))
        service_states_up = [service for service in service_list if service.get('State') == 'up']
        LOG.info(f"Current service status {service_list}")
        if len(service_list) == len(service_states_up):
            return True
        else:
            return False

    def check_hypervisor_list(count=None):
        hyper_list = yaml.safe_load(client_pod.exec(cmd_hypervisor_list))
        hyper_states = sum(h.get('State') == 'up' for h in hyper_list)
        LOG.info(f"Count hv: {hyper_states}")
        if hyper_states == count and len(hyper_list) == count:
            return True
        else:
            return False

    show_step(6)

    waiters.wait(
        lambda: check_node_count(count=ironic_nodes_count),
        timeout=900, interval=30,
        timeout_msg="Some bm nodes is not provided by ironic")

    show_step(7)

    waiters.wait(
        lambda: check_cmp_service_status(),
        timeout=600, interval=10,
        timeout_msg="Some cmp service is not ready")

    show_step(8)
    cell_cron_job = cluster.k8sclient.cronjobs.list(
        name_prefix="nova-cell", namespace="openstack")[0]
    job = cell_cron_job.read()
    job_template = job.spec.job_template
    job_template.metadata.name = "nova-cell-setup-" + \
                                 utils.gen_random_string(3)
    cluster.k8sclient.jobs.create(body=job_template,
                                  namespace="openstack")

    show_step(9)
    new_hyper_count = hyper_count + ironic_nodes_count
    waiters.wait(
        lambda: check_hypervisor_list(count=new_hyper_count),
        timeout=300, interval=10,
        timeout_msg="new hypervisor not ready")
