from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters
import time
import random

LOG = logger.logger


def reboot(kaas_manager, aws_ec2_client, show_step, machine_type='control'):
    """
    :param machine_type: 'control' or 'worker'
    """
    namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
    nodes = cluster.get_machines(machine_type=machine_type)
    print_names = "\n"
    for n in nodes:
        print_names += n.name + "\n"
    LOG.info("%s nodes will be rebooted: \n %s", machine_type, print_names)
    kubectl_client = cluster.k8sclient
    # Waiting for machines are Ready and helmbundles are deployed
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()

    ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
    cluster.check.check_actual_expected_docker_services(
        changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name},
        timeout=200)

    cluster.check.check_k8s_pods()
    for node in nodes:
        aws_instance_name = node.data['metadata'][
            'annotations']['kaas.mirantis.com/uid']
        show_step(1)
        LOG.info("Accessing %s node and rebooting it"
                 ". AWS instance name %s",
                 node.name, aws_instance_name)
        show_step(2)
        init_uptime = node.get_uptime()
        show_step(3)
        aws_ec2_client.instance_power_action(instance_name=aws_instance_name,
                                             action='reboot')
        show_step(4)
        time.sleep(30)
        waiters.wait_pass(
            lambda: node.run_cmd("uptime"), timeout=120)
        waiters.wait_pass(
            lambda: kubectl_client.pods.list_all(), timeout=120)
        show_step(5)
        current_uptime = node.get_uptime()
        node.get_reboot_list()
        assert current_uptime > init_uptime, \
            "Node has not been rebooted. Uptime before reboot: {} and after reboot: {}".format(init_uptime,
                                                                                               current_uptime)
        LOG.info("{} has been rebooted".format(node.name))
        show_step(6)
        cluster.check.check_k8s_pods()
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()

    ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
    cluster.check.check_actual_expected_docker_services(
        changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name},
        timeout=200)

    cluster.check.check_k8s_pods()
    cluster.check.check_actual_expected_pods()


def shutdown(kaas_manager, aws_ec2_client, show_step, machine_type='control',
             state='off'):
    """
    :param machine_type: 'control' or 'worker'
    :param state: 'off' or 'on'
    :return:
    """
    namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
    nodes = cluster.get_machines(machine_type=machine_type)
    kubectl_client = cluster.k8sclient
    if state == 'on':
        # Get node name from annotation and power on
        name_by_annotation = cluster.get_cluster_annotation('ha.off.node.name')
        assert name_by_annotation is not None, \
            "Node with 'power off' status does not exist" \
            " inside the cluster annotation!"
        node_name = name_by_annotation['ha.off.node.name']
        LOG.info("Power %s for %s node!", state, node_name)
        nodes = [n for n in nodes if n.name == node_name]
        if nodes:
            node = nodes[0]
        else:
            raise Exception("Machine with name {} does not exist!"
                            .format(node_name))
    elif state == 'off':
        # Make sure no nodes are turned off.
        name_by_annotation = cluster.get_cluster_annotation('ha.off.node.name')
        assert name_by_annotation is None, \
            "One of the nodes is already turned off!"
        print_names = "\n"
        for n in nodes:
            print_names += n.name + "\n"
        LOG.info("Power %s one of %s nodes: \n %s", state,
                 machine_type, print_names)
        # Get random node for power off
        node = random.choice(nodes)
        # Compare pods if cluster have full set of alive nodes
        cluster.check.check_machines_status()
        cluster.check.check_cluster_nodes()
        cluster.check.check_k8s_nodes()

        ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
        cluster.check.check_actual_expected_docker_services(
            changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name},
            timeout=200)

        cluster.check.check_k8s_pods()
    aws_instance_name = node.data['metadata'][
        'annotations']['kaas.mirantis.com/uid']
    show_step(1)
    LOG.info("Accessing %s node and power %s (AWS instance name %s)",
             node.name, state, aws_instance_name)
    if state == 'off':
        show_step(3)
        aws_ec2_client.instance_power_action(instance_name=aws_instance_name,
                                             action='stop')
        time.sleep(30)
        cluster.add_cluster_annotation('ha.off.node.name', node.name)
    elif state == 'on':
        show_step(3)
        time.sleep(30)
        aws_ec2_client.instance_power_action(instance_name=aws_instance_name,
                                             action='start')
        waiters.wait_pass(
            lambda: node.run_cmd("uptime"), timeout=120)
        cluster.remove_cluster_annotation('ha.off.node.name')
    show_step(4)
    waiters.wait_pass(
        lambda: kubectl_client.pods.list_all(), timeout=120)
    show_step(5)
    cluster.check.check_k8s_pods()
    if state == 'on':
        # Compare pods if cluster have full set of alive nodes
        cluster.check.check_machines_status()
        cluster.check.check_cluster_nodes()
        cluster.check.check_k8s_nodes()

        ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
        cluster.check.check_actual_expected_docker_services(
            changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name},
            timeout=200)

        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods()


def test_ha_child_reboot_control_nodes(kaas_manager, aws_ec2_client,
                                       show_step, cluster_condition_check):
    """Reboot every CHILD cluster node (one at a time)
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Extract Aws child nodes
        2. Collect uptime for machines
        3. Reboot node
        4. Wait till node is accessible
        5. Check if machine has been rebooted
        6. Check that all pods are Running and Ready
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    reboot(kaas_manager, aws_ec2_client, show_step, 'control')


def test_ha_child_reboot_worker_nodes(kaas_manager, aws_ec2_client,
                                      show_step, cluster_condition_check):
    """Reboot every CHILD cluster node (one at a time)
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Extract Aws child nodes
        2. Collect uptime for machines
        3. Reboot node
        4. Wait till node is accessible
        5. Check if machine has been rebooted
        6. Check that all pods are Running and Ready
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    reboot(kaas_manager, aws_ec2_client, show_step, 'worker')


def test_ha_child_poweroff_control_node(kaas_manager, aws_ec2_client,
                                        show_step, cluster_condition_check):
    """Shutdown one of CHILD cluster node
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Extract Aws child nodes
        2. Get Aws server
        3. Node power off
        4. Wait for cluster will be reconciled
        5. Check that all pods are Running and Ready
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    shutdown(kaas_manager, aws_ec2_client, show_step, 'control', 'off')


def test_ha_child_poweron_control_node(kaas_manager, aws_ec2_client,
                                       show_step, cluster_condition_check):
    """Shutdown one of CHILD cluster node
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Extract Aws child nodes
        2. Get Aws server
        3. Node power on. And wait for ssh ...
        4. Wait for cluster will be reconciled
        5. Check that all pods are Running and Ready
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    shutdown(kaas_manager, aws_ec2_client, show_step, 'control', 'on')


def test_ha_child_poweroff_worker_node(kaas_manager, aws_ec2_client,
                                       show_step, cluster_condition_check):
    """Shutdown one of CHILD cluster node
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Extract Aws child nodes
        2. Get Aws server
        3. Node power off
        4. Wait for cluster will be reconciled
        5. Check that all pods are Running and Ready
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    shutdown(kaas_manager, aws_ec2_client, show_step, 'worker', 'off')


def test_ha_child_poweron_worker_node(kaas_manager, aws_ec2_client, show_step,
                                      cluster_condition_check):
    """Shutdown one of CHILD cluster node
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Extract Aws child nodes
        2. Get Aws server
        3. Node power on. And wait for ssh ...
        4. Wait for cluster will be reconciled
        5. Check that all pods are Running and Ready
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    shutdown(kaas_manager, aws_ec2_client, show_step, 'worker', 'on')
