#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import yaml
import pytest

from os import environ
from si_tests import settings
from si_tests import logger
import si_tests.utils.templates as templates_utils


LOG = logger.logger


def test_rulexam(os_manager, request, show_step):
    """Run rulexam tests
    Parameters required for the test execution:
        - KUBECONFIG
    Scenario:
        1. Run rulexam pod
        2. Wait till pod performs tests
        3. Collect logs from the rulexam pod
    """
    show_step(1)
    rulexam_tests_image = environ.get('RULEXAM_TESTS_IMAGE')
    if not rulexam_tests_image:
        msg = ("Rulexam tests image is not set. Skipping test.")
        LOG.warning(msg)
        pytest.skip(msg)
    LOG.info(f"Using rulexam tests image: {rulexam_tests_image}")
    options = {"rulexam_tests_image": rulexam_tests_image}

    templates = templates_utils.render_template(
        settings.RULEXAM_TESTS_POD_TEMPLATE, options
    )
    LOG.debug(templates)
    pod_body = yaml.load(templates, Loader=yaml.SafeLoader)
    pod = os_manager.api.pods.create(
        name=settings.RULEXAM_TESTS_POD_NAME,
        namespace=settings.RULEXAM_TESTS_POD_NAMESPACE,
        body=pod_body)
    request.addfinalizer(pod.delete)

    show_step(2)
    pod.wait_phase('Running')
    pod.wait_test(filepath='./test_*', timeout=settings.RULEXAM_TESTS_TIMEOUT)

    show_step(3)
    pod.cp_from_pod(source_dir="/var/lib/tests/",
                    compression="-z")

    # Download per-test reports as an archive
    pod.exec(["/bin/sh", "-c",
              "tar -czvf /tmp/reports.tar.gz /var/lib/rulexam/rulexam/reports/"
              ])
    pod.cp_from_pod(source_dir="/tmp",
                    source_file="reports.tar.gz",
                    destination=settings.ARTIFACTS_DIR)
