from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Manager
from si_tests.clients.k8s.diagnostic_diagnostic import DiagnosticDiagnostic
from si_tests.utils import waiters, utils

from typing import Dict
from datetime import datetime
from retry import retry
import pytz

LOG = logger.logger


class TestDiagnosticChecks():
    @retry(AttributeError, delay=5, tries=5, jitter=1, logger=LOG)
    def is_finished_at_before_now(self, diagnostic: DiagnosticDiagnostic) -> bool:
        f_at = diagnostic.finished_at
        LOG.info(f"Checking '.status.finished_at': {f_at}")
        return False if f_at == datetime.fromtimestamp(timestamp=0, tz=pytz.UTC) \
                        else datetime.now().astimezone(pytz.UTC) > f_at

    def test_diagnostic_checks(self, kaas_manager: Manager, show_step):
        """Start baremetal-provider related diagnostic checks

         Scenario:
             1. Check init state cluster
             2. Create Diagnostic object for the cluster
             3. Wait for diasnostic checks to be finished
             4. Verify all the checks have been successfully finished
         """

        cluster_name = settings.TARGET_CLUSTER
        ns = kaas_manager.get_namespace(namespace=settings.TARGET_NAMESPACE)
        cluster = ns.get_cluster(cluster_name)

        show_step(1)
        LOG.info(
            f"Check init state on the {cluster._cluster_type} cluster {cluster.namespace}/{cluster.name}")
        cluster.check.check_machines_status()
        cluster.check.check_cluster_readiness()
        cluster.check.check_k8s_nodes()

        show_step(2)
        diagnostic_obj = cluster.create_diagnostic_object(object_name=cluster_name+"-"+utils.gen_random_string(10))

        show_step(3)
        # check finishedAt is set, if yes -> go to the actual checks
        waiters.wait(lambda: self.is_finished_at_before_now(diagnostic_obj),
                     timeout=settings.DIAGNOSTIC_FINISHING_TIMEOUT,
                     timeout_msg="after creation diagnostic object "
                                 "it has not been finished")

        show_step(4)
        assert len(diagnostic_obj.error) == 0, (
            f"An unexpected error during the diagnostic has happened: {diagnostic_obj.error}"
        )

        failed_checks: Dict[str, str] = dict()  # name2message
        for name, check in diagnostic_obj.result.items():
            if not name.startswith('bm_'):
                continue

            LOG.info(f"Incoming check {name}: {check}")

            result, success = str(check['result']).lower(), bool(check['success'])
            if (not success) or (result not in ['pass', 'info']):
                failed_checks[name] = str(check['message'])

        assert len(failed_checks) == 0, (
            "The following checks have been failed:\n" +
            "\n".join(['{0} failed with the following message: {1}'.format(k, v) for k, v in failed_checks.items()])
        )
