import base64
import exec_helpers
import json
import pytest
import yaml

from si_tests import settings
from si_tests import logger
import si_tests.utils.templates as templates_utils
from si_tests.utils import utils, waiters, exceptions
from si_tests.utils.utils import Provider

LOG = logger.logger


# Used for vsphere params
def parse_vsphere_params(x):
    return ''.join((x.split("/", 3)[3:]))


def test_kaas_ui(kaas_manager, kaas_ui_prerequisites):
    # playwright only
    if settings.KAAS_UI_TEST_TYPE != 'new-e2e':
        pytest.fail(f"Wrong invocation - settings.KAAS_UI_TEST_TYPE/{settings.KAAS_UI_TEST_TYPE} != new-e2e")
    lb_host = None
    child_data = dict()
    writer_password = kaas_manager.si_config.get_keycloak_user_password('writer')
    machine_folder = facility = network_path = resourse_pool = datastore_path = rhel_path = ""
    eq2_cidr = eq2_dhcpRanges = eq2_gateway = eq2_node_ip = eq2_services_ip = eq2_vlanId = eq2_api = ""
    pod_namespace = settings.KAAS_UI_TEST_NAMESPACE

    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    if mgmt_cluster.provider == Provider.baremetal:
        cloud_provider = 'bm'
        cluster_name = settings.TARGET_CLUSTER
        namespace_name = settings.TARGET_NAMESPACE
        with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
            pub_key_content = pub_key.read()
        mgmt_version = mgmt_cluster.spec['providerSpec']['value']['kaas']['release']
        LOG.info(f"KaaS mgmt version is:{mgmt_version}")
        release_name = kaas_manager.si_config.version_generators.get_child_cluster_release_name()
        allowed_distros = kaas_manager.get_allowed_distributions(release_name)
        render_opts = {
            "mgmt_version": mgmt_version,
            "target_cluster": cluster_name,
            "target_namespace": namespace_name,
            "machine_distribution": allowed_distros.get(settings.DISTRIBUTION_RELEVANCE, {}).get('id', ''),
            "pub_key_content": pub_key_content,
        }
        # UI test require clusterrelease obj in extra_data
        LOG.info('Acquiring ClusterRelease obj')
        extra_data = kaas_manager.get_clusterrelease(name=settings.KAAS_CHILD_CLUSTER_RELEASE_NAME,
                                                     namespace='default').data
        child_data = utils.render_child_data(kaas_manager.si_config,
                                             render_opts,
                                             extra_data={'clusterrelease': extra_data})
        # TODO(alexz): need to delete it after..?
        _bm_secret_name = 'bm-child-data.yaml'
        LOG.info(f'Creating cm {settings.KAAS_UI_TEST_NAMESPACE}/{_bm_secret_name}')
        ns = kaas_manager.get_namespace(pod_namespace)
        ns.create_secret(_bm_secret_name,
                         {_bm_secret_name: str(base64.b64encode(yaml.dump(child_data).encode('ascii')).decode())})
    elif mgmt_cluster.provider == Provider.openstack:
        cloud_provider = 'openstack'
    elif mgmt_cluster.provider == Provider.aws:
        cloud_provider = 'aws'
    elif mgmt_cluster.provider == Provider.vsphere:
        cloud_provider = 'vsphere'
    elif mgmt_cluster.provider == Provider.equinixmetal:
        cloud_provider = 'equinix'
        facility = utils.get_available_equinix_metro(settings.KAAS_EQUINIX_USER_API_TOKEN,
                                                     settings.KAAS_EQUINIX_MACHINE_TYPE, 5)
    elif mgmt_cluster.provider == Provider.equinixmetalv2:
        cloud_provider = 'equinixmetalv2'
        cluster_network_config = None
        si_network_config = kaas_manager.si_config.data['network_config']
        for i in si_network_config.keys():
            if settings.KAAS_UI_CHILD_CLUSTER_NAME in i:
                cluster_network_config = si_network_config[i]

        if cluster_network_config:
            facility = utils.get_available_equinix_metro(
                settings.KAAS_EQUINIX_USER_API_TOKEN,
                settings.KAAS_EQUINIX_MACHINE_TYPE,
                5,
                cluster_network_config['metro'])
            child_network_config = cluster_network_config['networkSpec']
            eq2_cidr = child_network_config['cidr']
            eq2_dhcpRanges = child_network_config['dhcpRanges'][0]
            eq2_gateway = child_network_config['gateway']
            eq2_node_ip = child_network_config['includeRanges'][0]
            eq2_services_ip = child_network_config['metallbRanges'][0]
            eq2_vlanId = child_network_config['vlanId']
            eq2_api = child_network_config['loadBalancerHost']
        else:
            raise Exception(
                "The selected scenario does not support creating kaas-ui-child!"
                "Please check SI_SCENARIO in mgmt deploy job"
            )

    elif mgmt_cluster.provider == Provider.azure:
        cloud_provider = 'azure'
    else:
        raise Exception(f"Unknown provider: {mgmt_cluster.provider}")

    LOG.info(f"Selected provider:{cloud_provider}")

    if settings.KAAS_UI_TEST_IMAGE_TAG:
        kaas_version = settings.KAAS_UI_TEST_IMAGE_TAG
    else:
        pod_list = kaas_manager.api.api_core.list_namespaced_pod(
            namespace='kaas', label_selector='app.kubernetes.io/instance=kaas-ui').to_dict()['items'][0]
        kaas_version = pod_list['spec']['containers'][0]['image'].split(':')[1]

    pod_name = settings.KAAS_UI_TEST_POD_NAME + '-playwright'
    ns_name = settings.KAAS_UI_E2E_TEST_NS_PREFIX + '-playwright'

    # Delete the pod before test if it is running
    if kaas_manager.api.pods.present(name=pod_name, namespace=pod_namespace):
        LOG.info(f"Pod '{pod_name}' already exists, deleting")
        pod = kaas_manager.api.pods.get(name=pod_name, namespace=pod_namespace)
        pod.delete()

    kaas_service = kaas_manager.api.api_core.read_namespaced_service(name='kaas-kaas-ui',
                                                                     namespace='kaas').to_dict()
    LOG.debug(kaas_service)
    kaas_ui_ip = \
        kaas_service['status']['load_balancer']['ingress'][0]['ip'] or \
        kaas_service['status']['load_balancer']['ingress'][0]['hostname']
    cdn_region = settings.CDN_REGION
    artifacts_region_map = settings.ARTIFACTS_REGION_MAPPING
    ui_image_base = artifacts_region_map[cdn_region]["images_base_url"]
    if mgmt_cluster.provider == Provider.vsphere:
        machine_folder = parse_vsphere_params(settings.KAAS_VSPHERE_MACHINES_FOLDER)
        network_path = parse_vsphere_params(settings.KAAS_VSPHERE_NETWORK_PATH)
        resourse_pool = parse_vsphere_params(settings.KAAS_VSPHERE_RESOURCE_POOL_PATH)
        datastore_path = parse_vsphere_params(settings.KAAS_VSPHERE_DATASTORECLUSTER_PATH)
        rhel_path = parse_vsphere_params(settings.KAAS_VSPHERE_RHEL8_TEMPLATE_PATH)

    options = {
        'CHILD_CREATE_ONLY_SCENARIO': settings.KAAS_UI_CHILD_CREATE_ONLY_SCENARIO,
        'CLOUD_PROVIDER': cloud_provider,
        'DATASTORECLUSTER_PATH': datastore_path,
        'EQ2_API': eq2_api,
        'EQ2_CIDR': eq2_cidr,
        'EQ2_DHCP_RANGES': eq2_dhcpRanges,
        'EQ2_GATEWAY': eq2_gateway,
        'EQ2_NODE_IP': eq2_node_ip,
        'EQ2_SERVICES': eq2_services_ip,
        'EQ2_VLAN_ID': eq2_vlanId,
        'FACILITY': facility,
        'KAAS_UI_ADDRESS': kaas_ui_ip,
        'KAAS_UI_E2E_TEST_NS_NAME_PREFIX': ns_name,
        'KAAS_UI_PASSWORD': writer_password,
        'KAAS_UI_PORT': kaas_service['spec']['ports'][0]['port'],
        'KAAS_UI_TEST_IMAGE': f"{ui_image_base}/{settings.KAAS_UI_TEST_IMAGE}:{kaas_version}",
        'KAAS_UI_TEST_POD_NAME': pod_name,
        'KAAS_UI_TEST_TYPE': settings.KAAS_UI_TEST_TYPE,
        'KAAS_UI_USERNAME': 'writer',
        'LOADBALANCER_HOST': lb_host,
        'MACHINES_FOLDER': machine_folder,
        'NETWORK_PATH': network_path,
        'NEW_PATH': '/playwright-tests',
        'RESOURCE_POOL_PATH': resourse_pool,
        'RHEL_TEMPLATE_PATH': rhel_path,
    }

    templates = templates_utils.render_template(settings.KAAS_UI_TEST_POD_YAML, options)
    LOG.debug(templates)
    json_body = json.dumps(yaml.load(templates, Loader=yaml.SafeLoader))

    pod = kaas_manager.api.pods.create(name=pod_name, namespace=pod_namespace, body=json.loads(json_body))
    pod.wait_phase('Running', timeout=3000)
    LOG.info(f"UI tests uses image {pod.data['status']['container_statuses'][0]['image_id']}")

    try:
        pod.wait_test(filepath='./test_*', timeout=7200)
    finally:
        exec_helpers.Subprocess().check_call(f"mkdir -p {settings.ARTIFACTS_DIR}/reports")

        res = pod.exec(['/bin/sh', '-c', f'ls -la {settings.KAAS_UI_TEST_REPORTS_DIR}/reports'])
        LOG.info(f"Reports in pod {pod_namespace}/{pod_name}:{settings.KAAS_UI_TEST_REPORTS_DIR}/reports :\n{res}")

        pod.cp_from_pod(source_dir=settings.KAAS_UI_TEST_REPORTS_DIR + '/reports',
                        destination=settings.ARTIFACTS_DIR + '/reports', container=pod_name)
        pod.cp_from_pod(source_dir=settings.KAAS_UI_TEST_REPORTS_DIR + '/screenshots', container=pod_name)
        pod.cp_from_pod(source_dir=settings.KAAS_UI_TEST_REPORTS_DIR + '/videos', container=pod_name)

    if not settings.KEEP_KAAS_UI_TEST_POD_AFTER:
        pod.delete()
    if not pod.check_file_exists('./test_passed'):
        pytest.fail("Some kaas UI tests failed.See detailed report in artifacts")

    if cloud_provider == "bm":
        tgt_ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
        tgt_cluster = tgt_ns.get_cluster(name=settings.TARGET_CLUSTER)
        # START: copy-paste from test_create_physical_child_ceph_lma_istio_harbor.py
        if settings.MOSK_CHILD_DEPLOY_CEPH:
            LOG.info("Create MiraCeph using raw methods")
            if not child_data.get('miraceph_raw').get('spec').get('network', None):
                pytest.fail(f'No network settings has been provided in miraceph_raw!\n'
                            f'Please fix env_config_name={settings.ENV_CONFIG_NAME}')
            cluster_name = child_data['cluster_raw']['metadata']['name']
            tgt_ns.create_miraceph_raw(cluster_name=cluster_name, data=child_data['miraceph_raw'])
            try:
                waiters.wait(lambda: tgt_cluster.check.check_child_ceph_cluster(check_health=True),
                             timeout=90 * 60, interval=60)
            except exceptions.TimeoutError as e:
                LOG.error("Tired waiting for ceph cluster correct status,"
                          "events from management cluster:")
                tgt_ns.print_events()
                raise e

            ceph_cluster_pools = child_data['miraceph_raw']['spec']['pools']
            tgt_cluster.check.wait_default_storage(ceph_cluster_pools=ceph_cluster_pools, timeout=30 * 60)
            if tgt_cluster.is_sl_enabled():
                # wait while provider understands that ceph is alive
                # and re-generates helmbundle with stacklight
                tgt_cluster.check.wait_for_release_in_bundle(name="stacklight")
            tgt_cluster.check.check_actual_expected_pods(interval=30)
            # Check Ceph pvc
            tgt_cluster.check.check_ceph_pvc()
        # END: copy-paste from test_create_physical_child_ceph_lma_istio_harbor.py
        # perform checks
        tgt_cluster.check.check_cluster_readiness(timeout=3600, interval=90)
        tgt_cluster.check.check_helmbundles()
        tgt_cluster.check.check_deploy_stage_success()
        tgt_cluster.check.check_actual_expected_docker_services()
        tgt_cluster.check.check_k8s_pods()
        tgt_cluster.check.check_actual_expected_pods(interval=30, check_all_nss=True)
        tgt_cluster.check.check_actual_expected_kernel_versions()
        tgt_cluster.check.check_actual_expected_distribution()
        #  we need to store data only for -bm clusters, since in -bm ci we mimic -ui deployment as usual deployment
        # so need to have all expected artifacts.
        tgt_cluster.store_k8s_artifacts()
