import pytest
from retry import retry

from si_tests import settings
from si_tests import logger
from si_tests.clients.dashboard import k8s_dashboard_client

LOG = logger.logger


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
def test_k8s_dashboard(kaas_manager, keycloak_ip, _):
    """Get resource names from kubernetes-dashboard and match to names from api
    """
    # Get child cluster api client
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    child_client = cluster.k8sclient

    # Get kubernetes ip and port
    service = child_client.services.get(name='kubernetes-dashboard-proxy',
                                        namespace='kube-system')
    ip = service.get_external_ip()
    service_port = [s for s in service.get_ports() if s.name == 'http']
    assert len(service_port) > 0, (
        "No http port found for the service kubernetes-dashboard-proxy")
    port = service_port[0].port

    # Get objects from kubernetes dashboard
    dashboard_url = "http://{0}:{1}".format(ip, port)
    writer_password = kaas_manager.si_config.get_keycloak_user_password('writer')
    d_client = k8s_dashboard_client.K8sDashboardClient(
        keycloak_ip, dashboard_url, 'writer', writer_password)

    @retry(AssertionError, delay=20, tries=3, logger=LOG)
    def wait_resources(check_res, d_client, child_client):
        check_plural = check_res + "s"
        LOG.info("Check dashboard resource <{0}>".format(check_plural))
        d_resources = d_client.get_resource(check_res)
        d_res_names = [obj["objectMeta"]["name"]
                       for obj in d_resources[check_plural]]
        LOG.debug("{0} names from dashboard: {1}"
                  .format(check_plural, d_res_names))

        k_resources = child_client.__getattribute__(check_plural).list_all()
        k_res_names = [obj.name for obj in k_resources]
        LOG.debug("{0} names from k8s api: {1}"
                  .format(check_plural, k_res_names))

        assert set(k_res_names) == set(d_res_names), (
            "Resources for <{0}> from k8s api and from "
            "kubernetes-dashboard {1} don't match"
            .format(check_plural, dashboard_url))

    check_resource_names = ["namespace", "pod", "service", "node"]
    for check_res in check_resource_names:
        wait_resources(check_res, d_client, child_client)
