import base64
import yaml
import os
import time

from si_tests.utils import waiters
from si_tests import settings

from si_tests import logger
from si_tests.clients.openstack import OpenstackClient
from si_tests.managers.aws_manager import AwsManager
from vconnector.core import VConnector
from pyVmomi import vim

LOG = logger.logger


class ProviderResources(object):
    def __init__(self, manager, cluster):
        """
        manager: <Manager> instance
        cluster: <KaaSCluster> instance
        """
        self.manager = manager
        self.cluster = cluster
        self.creds = \
            cluster.data['spec']['providerSpec']['value']['credentials']
        self.uid = \
            cluster.data['metadata']['annotations']['kaas.mirantis.com/uid']
        self.cluster_region = cluster.region_name
        self.pv_names = [x.name for x in
                         self.cluster.k8sclient.pvolumes.list_all()
                         if 'local-pv' not in x.name]
        self.nodes_names = \
            [x.name for x in self.cluster.k8sclient.nodes.list()]
        self.__resources = {}
        self.not_deleted_resources = {'checked': False}

    def get_resources(self):
        return {"NotImplementedError": True}

    def get_nodes(self):
        """General method to return nodes"""
        raise NotImplementedError("Invoking get_nodes from ProviderResources")

    @property
    def resources(self):
        try:
            if not self.__resources:
                self.__resources = self.get_resources()
            return self.__resources
        except Exception as e:
            LOG.error(e)

    def get_resources_yaml(self):
        LOG.debug("PROVIDER RESOURCES INFO: {}".format(self.resources))
        return yaml.dump(self.resources)

    def save_artifact(self):
        artifact_name = os.path.join(
            settings.ARTIFACTS_DIR,
            "{}_provider_resources.yaml".format(self.cluster.name)
        )
        LOG.info(f"Save cluster {self.cluster.name} provider "
                 f"resources to {artifact_name}")
        with open(artifact_name, 'w') as f:
            try:
                f.write(self.get_resources_yaml())
            except Exception as e:
                LOG.error(e)

    @property
    def no_resources(self):
        for k, v in self.resources.items():
            if v:
                return False
        else:
            return True


class OSProviderResources(ProviderResources):
    """Openstack provider specific methods"""
    __client = None

    def __init__(self, manager, cluster):
        super().__init__(manager, cluster)
        self.__client = self.client

    def get_password_from_secret(self, secret_name):
        secret = self.manager.api.secrets.get(
            secret_name, namespace=self.cluster.namespace)
        pwd_data = secret.read().to_dict()['data']['value']
        return base64.b64decode(pwd_data).decode("utf-8")

    @property
    def client(self):
        if not self.__client:
            oscreds = \
                self.manager.api.kaas_openstackcredentials.get(
                    self.creds, self.cluster.namespace).data['spec']
            oscreds['auth']['auth_url'] = oscreds['auth']['authURL']
            oscreds['auth']['password'] = \
                self.get_password_from_secret(
                    oscreds['auth']['password']['secret']['name']
                )
            oscreds['auth']['username'] = oscreds['auth']['userName']
            oscreds['auth']['project_name'] = oscreds['auth'].get('projectName')
            oscreds['auth']['project_id'] = oscreds['auth']['projectID']
            oscreds['auth']['user_domain_name'] = \
                oscreds['auth']['userDomainName']
            self.__client = OpenstackClient(oscreds)
        return self.__client

    # overriding method
    def get_nodes(self):
        bastion_node = []
        if self.cluster.is_management or self.cluster.is_regional:
            provider_status = \
                self.cluster.data["status"].get("providerStatus", {})
            if 'bastion' in provider_status and provider_status.get('bastion'):
                bastion_ip = provider_status["bastion"]['publicIP']
                bastion_node = self.client.get_server_by_fip(bastion_ip)
                if bastion_node:
                    LOG.debug("Bastion node : {}".format(bastion_node))
                    bastion_node = [bastion_node._info]
        return [x._info for x in self.client.server_list()
                if x.name in self.nodes_names] + bastion_node

    def get_resources(self):
        resources = {}
        resources['vms'] = self.get_nodes()
        resources['volumes'] = self.get_volumes()
        resources['lbs'] = self.get_lbs()
        resources['fips'] = self.get_fips()
        resources['nets'] = self.get_networks()
        resources['subnets'] = self.get_subnets()
        resources['routers'] = self.get_routers()
        resources['sgs'] = self.get_sgs()
        return resources

    # k8s pvc
    def get_volumes(self):
        # Openstack: Cinder
        return [pv._info for pv in
                self.client.cinder.volumes.list() if
                pv.name in self.pv_names]

    def get_lbs(self):
        # Openstack: Octavia
        net_ids = [x['id'] for x in self.get_networks()]
        return [lb for lb in
                self.client.octavia.load_balancer_list()['loadbalancers']
                if lb['vip_network_id'] in net_ids]

    def get_fips(self):
        # OS only
        return [fip for fip in
                self.client.neutron.list_floatingips()['floatingips'] if
                self.uid in fip.get('tags')]

    def get_networks(self):
        return [net for net in
                self.client.neutron.list_networks()['networks'] if
                self.uid in net.get('tags')]

    def get_subnets(self):
        return [subnet for subnet in
                self.client.neutron.list_subnets()['subnets'] if
                self.uid in subnet.get('tags')]

    def get_routers(self):
        return [router for router in
                self.client.neutron.list_routers()['routers'] if
                self.uid in router.get('tags')]

    def get_sgs(self):
        neutron_cl = self.client.neutron
        return [sg for sg in
                neutron_cl.list_security_groups()['security_groups'] if
                self.uid in sg.get('tags')]

    def check_resources_deletion(self, cluster_resources=None, volumes_cleanup_check=False):
        """
        This method checks cluster_resources existence in
        all_resources for current provider and returns a dict
        with existed resources.
        """

        not_deleted_resources_map = {
            'vms': [], 'subnets': [], 'volumes': [], 'sgs': [],
            'fips': [], 'lbs': [], 'routers': [], 'nets': []
        }
        assert cluster_resources, ("Cluster resources are not passed. "
                                   "Nothing to check")
        assert type(cluster_resources) == dict, (
            "Wrong cluster_resources format. Dict expected. Example: "
            "{}".format(not_deleted_resources_map))

        all_resources = self.client.get_all_resources()
        for vm in cluster_resources.get('vms', []):
            if vm.get('id') in [
                    vm.get('id') for vm in all_resources.get('vms')]:
                not_deleted_resources_map['vms'].append(vm)

        for subnet in cluster_resources.get('subnets', []):
            if subnet.get('id') in [
                subnet.get('id') for subnet in
                    all_resources.get('subnets')]:
                not_deleted_resources_map['subnets'].append(subnet)

        for vol in cluster_resources.get('volumes', []):
            vol_id = vol.get('id')
            if vol_id in [
                    vol.get('id') for vol in all_resources.get('volumes')]:
                current_volume = [
                        vol for vol in all_resources.get(
                            'volumes', []) if vol.get(
                                'id') == vol_id][0]
                if current_volume.get('status') == 'in-use':
                    not_deleted_resources_map['volumes'].append(vol)
                elif not volumes_cleanup_check and current_volume.get('status') == 'available':
                    LOG.warning("Vol {} was detached, but not deleted. "
                                "Please, delete it manually "
                                "if needed".format(vol_id))
                else:
                    LOG.warning("Vol {} has state {}. Assuming "
                                "volume not deleted.".format(
                                    vol_id, current_volume.get(
                                        'status')))
                    not_deleted_resources_map['volumes'].append(vol)

        for lb in cluster_resources.get('lbs', []):
            lb_id = lb.get('id')
            if lb_id in [lb.get('id') for lb in all_resources.get('lbs')]:
                not_deleted_resources_map['lbs'].append(lb)

        for fip in cluster_resources.get('fips', []):
            if fip.get('id') in [
                    fip.get('id') for fip in all_resources.get('fips')]:
                not_deleted_resources_map['fips'].append(fip)

        for net in cluster_resources.get('nets', []):
            if net.get('id') in [
                    net.get('id') for net in all_resources.get('nets')]:
                not_deleted_resources_map['nets'].append(net)

        for router in cluster_resources.get('routers', []):
            if router.get('id') in [
                    router.get('id') for router in all_resources.get(
                    'routers')]:
                not_deleted_resources_map['routers'].append(router)

        for sg in cluster_resources.get('sgs', []):
            if sg.get('id') in [
                    sg.get('id') for sg in all_resources.get('sgs')]:
                not_deleted_resources_map['sgs'].append(sg)

        not_deleted_resources = {k: v for k, v in
                                 not_deleted_resources_map.items() if v}

        self.not_deleted_resources = not_deleted_resources

        return not_deleted_resources


class AWSProviderResources(ProviderResources):
    """AWS provider specific methods"""
    __client = None
    __filter = None
    __cluster_uid_tag_name = None
    __cluster_uid_tag = None
    __cluster_name_tag = None
    __vpc_id = None
    __vpc = None

    def __init__(self, manager, cluster):
        super().__init__(manager, cluster)
        self.__client = self.client
        self.__cluster_uid_tag_name = 'kubernetes.io/cluster/{}'.format(self.uid)
        self.__filter = [
            {'Name': 'tag:{}'.format(self.__cluster_uid_tag_name),
             'Values': ['owned']}
        ]

    @property
    def client(self):
        if not self.__client:
            region = self.cluster.data['spec']['providerSpec'][
                'value']['region']
            aws_key_id, aws_secret_access_key = \
                self.manager.get_aws_credential(self.creds,
                                                self.cluster.namespace,
                                                region=self.cluster_region)
            self.__client = AwsManager(
                aws_access_key_id=aws_key_id,
                aws_secret_access_key=aws_secret_access_key,
                region_name=region)
        return self.__client

    @property
    def vpc_id(self):
        if not self.__vpc_id:
            self.__vpc_id = self.cluster.data['status']['providerStatus']['vpc']['id']
        return self.__vpc_id

    @property
    def vpc(self):
        if not self.__vpc:
            self.__vpc = self.client.get_vpc(self.vpc_id)
        return self.__vpc

    @property
    def cluster_uid_tag(self):
        if not self.__cluster_uid_tag:
            self.__cluster_uid_tag = {
                'Key': '{}'.format(self.__cluster_uid_tag_name),
                'Value': 'owned'
            }
        return self.__cluster_uid_tag

    @property
    def cluster_name_tag(self):
        if not self.__cluster_name_tag:
            self.__cluster_name_tag = {
                'Key': 'cluster-name',
                'Value': '{}'.format(self.cluster.name)
            }
        return self.__cluster_name_tag

    def get_resources(self):
        resources = {}
        resources['vms'] = self.get_nodes()
        resources['vpc'] = self.get_vpc()
        resources['volumes'] = self.get_volumes()
        resources['lbs'] = self.get_lbs()
        resources['subnets'] = self.get_subnets()
        resources['route_tables'] = self.get_route_tables()
        resources['sgs'] = self.get_sgs()
        resources['network_interfaces'] = self.get_network_interfaces()
        resources['internet_gateways'] = self.get_internet_gateways()
        resources['ips'] = self.get_addresses()
        resources['nat_gateways'] = self.get_nat_gateways()
        return resources

    # overriding method
    def get_nodes(self):
        return self.client.get_instances(self.__filter)

    def get_volumes(self):
        cluster_volumes = [
            volume.spec.aws_elastic_block_store for volume in
            self.cluster.k8sclient.pvolumes.list_raw().items if
            volume.spec.aws_elastic_block_store]
        aws_cluster_volumes_ids = [volume.volume_id.split('/')[-1]
                                   for volume in cluster_volumes]
        filter = [{'Name': 'volume-id',
                   'Values': aws_cluster_volumes_ids}]
        return self.client.get_volumes(filter)['Volumes']

    def get_vpc(self):
        filter = [{'Name': 'vpc-id',
                   'Values': [self.vpc._id]}]
        return self.client.get_vpcs(filter)['Vpcs']

    def get_network_interfaces(self):
        filter = [{'Name': 'vpc-id',
                   'Values': [self.vpc._id]}]
        return self.client.get_network_interfaces(filter)['NetworkInterfaces']

    def get_internet_gateways(self):
        return self.client.get_internet_gateways(
            self.__filter)['InternetGateways']

    def get_network_acls(self):
        filter = [{'Name': 'vpc-id',
                   'Values': [self.vpc._id]}]
        return self.client.get_network_acls(filter)['NetworkAcls']

    def get_subnets(self):
        return self.client.get_subnets(self.__filter)['Subnets']

    def get_route_tables(self):
        return self.client.get_route_tables(self.__filter)['RouteTables']

    def get_sgs(self):
        return self.client.get_security_groups(self.__filter)['SecurityGroups']

    def get_addresses(self):
        return self.client.get_addresses(self.__filter)['Addresses']

    def get_lbs(self):
        return [lb for lb in self.client.get_lbs()
                if lb['VPCId'] == self.vpc._id]

    def get_nat_gateways(self):
        return self.client.get_nat_gateways(self.__filter)['NatGateways']

    def get_default_sg_for_cluster_vpc(self):
        filter = [{
            "Name": "group-name",
            "Values": ["default"]
        }]
        sgs = self.client.get_security_groups(filter)['SecurityGroups']
        cluster_sgs = [sg for sg in sgs if sg['VpcId'] == self.vpc_id]
        assert len(cluster_sgs) == 1, (f"Should be only 1 default security group for a cluster vpc: {self.vpc_id}")
        return cluster_sgs[0]

    def check_resources_deletion(self, cluster_resources=None, volumes_cleanup_check=False):
        """
        This method checks cluster_resources existence in
        all resources for current provider and returns a dict
        of existed resources

        """

        not_deleted_resources_map = {
            'vms': [], 'subnets': [], 'route_tables': [],
            'network_interfaces': [], 'volumes': [],
            'internet_gateways': [], 'nat_gateways': [],
            'sgs': [], 'fips': [], 'vpcs': [], 'lbs': [],
        }

        assert cluster_resources, ("Cluster resources are not passed. "
                                   "Nothing to check")
        assert type(cluster_resources) == dict, (
            "Wrong cluster_resources format. Dict expected. Example: "
            "{}".format(not_deleted_resources_map))

        expected_statuses = ['terminated', 'detached', 'deleted',
                             'disassociated', 'stopped']
        all_resources = self.client.get_all_resources()
        for vm in cluster_resources.get('vms', []):
            vm_id = vm.get('InstanceId')
            if vm_id in [vm.get('InstanceId') for vm in all_resources.get(
                    'vms', []) if vm.get('State').get(
                        'Name') not in expected_statuses]:
                not_deleted_resources_map['vms'].append(vm)

        for subnet in cluster_resources.get('subnets', []):
            if subnet.get('SubnetId') in [subnet.get('SubnetId') for subnet
                                          in all_resources.get('subnets')]:
                not_deleted_resources_map['subnets'].append(subnet)

        for ip in cluster_resources.get('ips', []):
            if ip.get('AllocationId') in [ip.get('AllocationId') for ip in
                                          all_resources.get('ips')]:
                not_deleted_resources_map['fips'].append(ip)

        for sg in cluster_resources.get('sgs', []):
            if sg.get('GroupId') in [sg.get('GroupId') for sg in
                                     all_resources.get('sgs')]:
                not_deleted_resources_map['sgs'].append(sg)

        for lb in cluster_resources.get('lbs', []):
            if lb.get('LoadBalancerName') in [
                lb.get('LoadBalancerName') for lb in all_resources.get(
                    'lbs')]:
                not_deleted_resources_map['lbs'].append(lb)

        for rt in cluster_resources.get('route_tables', []):
            rt_id = rt.get('RouteTableId')
            if rt_id in [rt.get('RouteTableId') for rt in
                         all_resources.get('route_tables', [])]:
                current_rt = [rt for rt in all_resources.get(
                    'route_tables', []) if rt.get(
                        'RouteTableId') == rt_id][0]
                if len(current_rt.get('Associations', [])) > 0:
                    not_deleted_resources_map[
                            'route_tables'].append(rt)
                else:
                    LOG.warning("Route table {} was disassociated "
                                "but still exists".format(rt_id))
                    not_deleted_resources_map[
                            'route_tables'].append(rt)

        for n_iface in cluster_resources.get('network_interfaces', []):
            iface_id = n_iface.get('NetworkInterfaceId')
            if iface_id in [iface.get('NetworkInterfaceId') for iface in
                            all_resources.get('network_interfaces', [])]:
                current_iface = [iface for iface in all_resources.get(
                    'network_interfaces', []) if iface.get(
                        'NetworkInterfaceId') == iface_id][0]
                if len(current_iface.get('Attachment', [])) > 0:
                    not_deleted_resources_map[
                            'network_interfaces'].append(n_iface)
                else:
                    LOG.warning("Network interface {} was detached "
                                "but still exists".format(iface_id))
                    not_deleted_resources_map[
                            'network_interfaces'].append(n_iface)

        for gw in cluster_resources.get('internet_gateways', []):
            gw_id = gw.get('InternetGatewayId')
            if gw_id in [gw.get('InternetGatewayId') for gw in
                         all_resources.get('internet_gateways', [])]:
                current_gw = [gw for gw in all_resources.get(
                    'internet_gateways', []) if gw.get(
                        'InternetGatewayId') == gw_id][0]
                if len(current_gw.get('Attachments', [])) > 0:
                    not_deleted_resources_map['internet_gateways'].append(
                        gw)
                else:
                    LOG.warning("Internet gateway {} was detached "
                                "but still exists".format(gw_id))
                    not_deleted_resources_map['internet_gateways'].append(
                        gw)

        for nat_gw in cluster_resources.get('nat_gateways', []):
            nat_gw_id = nat_gw.get('NatGatewayId')
            if nat_gw_id in [nat_gw.get('NatGatewayId') for nat_gw in
                             all_resources.get('nat_gateways', []) if
                             nat_gw.get(
                                 'State') not in expected_statuses]:
                not_deleted_resources_map['nat_gateways'].append(
                    nat_gw)

        for vpc in cluster_resources.get('vpc', []):
            if vpc.get('VpcId') in [vpc.get('VpcId') for vpc in
                                    all_resources.get('vpcs')]:
                not_deleted_resources_map['vpcs'].append(vpc)

        for vol in cluster_resources.get('volumes', []):
            vol_id = vol.get('VolumeId')
            if vol_id in [vol.get('VolumeId') for vol in
                          all_resources.get('volumes', [])]:
                current_volume = [
                        vol for vol in all_resources.get(
                            'volumes', []) if vol.get(
                                'VolumeId') == vol_id][0]
                if current_volume.get('State') == 'in-use':
                    not_deleted_resources_map['volumes'].append(vol)
                elif current_volume.get('State') == 'available':
                    LOG.warning("Vol {} was detached, but not deleted. "
                                "Please, delete it manually "
                                "if needed".format(vol_id))
                else:
                    LOG.warning("Vol {} has unexpected state. Please, "
                                "check manually".format(vol_id))

        not_deleted_resources = {k: v for k, v in
                                 not_deleted_resources_map.items() if v}

        self.not_deleted_resources = not_deleted_resources

        return not_deleted_resources

    def create_volume(self, name="", **kwargs):
        required_parameter = 'AvailabilityZone'
        if required_parameter not in kwargs:
            raise ValueError(f"{required_parameter} is a required parameter")

        volume = self.client.create_volume(
            TagSpecifications=[{
                'ResourceType': 'volume',
                'Tags': [
                    {
                        'Key': 'Name',
                        'Value': f'{name}'
                    },
                    self.cluster_uid_tag,
                    self.cluster_name_tag
                ]
            }],
            **kwargs
        )
        waiter = self.client.get_waiter('volume_available')
        waiter.wait(VolumeIds=[volume['VolumeId']])
        return volume

    def create_subnet(self, name="", **kwargs):
        subnet = self.client.create_subnet(
            VpcId=self.vpc_id,
            TagSpecifications=[{
                'ResourceType': 'subnet',
                'Tags': [
                    {
                        'Key': 'Name',
                        'Value': f'{name}'
                    },
                    self.cluster_uid_tag,
                    self.cluster_name_tag
                ]
            }],
            **kwargs
        )
        subnet_id = subnet['Subnet']['SubnetId']

        waiter = self.client.get_waiter('subnet_available')
        waiter.wait(SubnetIds=[subnet_id])
        return subnet

    def create_network_interface(self, name="", **kwargs):
        required_parameter = 'SubnetId'
        if required_parameter not in kwargs:
            raise ValueError(f"{required_parameter} is a required parameter")

        network_interface = self.client.create_network_interface(
            TagSpecifications=[{
                'ResourceType': 'network-interface',
                'Tags': [
                    {
                        'Key': 'Name',
                        'Value': f'{name}'
                    },
                    self.cluster_uid_tag,
                    self.cluster_name_tag
                ]
            }],
            **kwargs
        )

        network_interface_id = network_interface['NetworkInterface']['NetworkInterfaceId']
        waiter = self.client.get_waiter('network_interface_available')
        waiter.wait(NetworkInterfaceIds=[network_interface_id])
        return network_interface

    def attach_network_interface(self, **kwargs):
        required_parameters = ['DeviceIndex', 'InstanceId', 'NetworkInterfaceId']
        if any(key not in kwargs for key in required_parameters):
            raise ValueError(f"{required_parameters} are required parameters")

        response = self.client.attach_network_interface(**kwargs)
        return response

    def attach_volume(self, **kwargs):
        required_parameters = ['Device', 'InstanceId', 'VolumeId']
        if any(key not in kwargs for key in required_parameters):
            raise ValueError(f"{required_parameters} are required parameters")

        response = self.client.attach_volume(**kwargs)
        LOG.info("Need to wait some time for disk appears in devices")
        time.sleep(20)
        return response

    def modify_network_interface_attribute(self, **kwargs):
        required_parameter = 'NetworkInterfaceId'
        if required_parameter not in kwargs:
            raise ValueError(f"{required_parameter} is a required parameter")

        response = self.client.modify_network_interface_attribute(**kwargs)
        return response


class VsphereProviderResources(ProviderResources):
    """Vsphere provider specific methods"""
    __client = None

    def __init__(self, manager, cluster):
        super().__init__(manager, cluster)
        self.__client = self.client
        self.datastorage = self.cluster.data[
            'spec']['providerSpec']['value'][
            'vsphere']['cloudProviderDatastore']

    def get_password_from_secret(self, secret_name):
        secret = self.manager.api.secrets.get(
            secret_name, namespace=self.cluster.namespace)
        pwd_data = secret.read().to_dict()['data']['value']
        return base64.b64decode(pwd_data).decode("utf-8")

    @property
    def client(self):
        if not self.__client:
            vscreds = \
                self.manager.api.kaas_vspherecredentials.get(
                    self.creds, self.cluster.namespace).data['spec']
            vhost = vscreds['vsphere']['server']
            vuser = vscreds['clusterApi']['username']
            vpassword = \
                self.get_password_from_secret(
                    vscreds['clusterApi']['password']['secret']['name'])
            self.__client = VConnector(vuser, vpassword, vhost)
        return self.__client

    def get_resources(self):
        resources = {}
        resources['vms'] = self.get_nodes()
        resources['volumes'] = self.get_volumes()
        return resources

    def get_all_resources(self):
        all_resources = {}
        all_resources['vms'] = self.get_all_nodes()
        all_resources['volumes'] = self.get_all_volumes()
        return all_resources

    def get_nodes(self):
        return [x.name for x in self.client.get_vm_view().view
                if x.name in self.nodes_names]

    def get_all_nodes(self):
        return [x.name for x in self.client.get_vm_view().view]

    def get_volumes(self, all_volumes=False):
        def wait_task(task):
            if task.info.state != "success":
                if task.info.state == 'error':
                    LOG.error(task.info.error.msg)
                    raise AttributeError
                return False
            else:
                return True
        dsname = self.datastorage
        datacenter = self.client.si.content.rootFolder.childEntity[0]
        datastores = datacenter.datastore
        ds = [ds for ds in datastores if ds.summary.name == dsname][0]
        search = vim.HostDatastoreBrowserSearchSpec()
        search.matchPattern = "*.vmdk"
        search_ds = ds.browser.SearchDatastoreSubFolders_Task(
            datastorePath=f"[{ds.name}]\\kubevols", searchSpec=search)
        pv_files = []
        try:
            waiters.wait(lambda: wait_task(search_ds), timeout=12,
                         timeout_msg=f"Datastore {ds.name} has no volumes")
            results = search_ds.info.result
            for rs in results:
                for f in rs.file:
                    if all_volumes:
                        if "-flat" not in f.path:
                            pv_files.append(f.path)
                    else:
                        for pv_name in self.pv_names:
                            if pv_name in f.path and "-flat" not in f.path:
                                pv_files.append(f.path)
        except AttributeError:
            LOG.error("Error searching for volumes in datastore")
        except TimeoutError:
            LOG.error("Waiting datastore Timeout error")
        return pv_files

    def get_all_volumes(self):
        return self.get_volumes(all_volumes=True)

    def check_resources_deletion(self, cluster_resources=None, volumes_cleanup_check=False):
        """
        This method checks cluster_resources existence in
        all_resources for current provider and returns a dict
        with existed resources.
        """

        not_deleted_resources_map = {
            'vms': [], 'volumes': [],
        }
        assert type(cluster_resources) == dict, (
            "Wrong cluster_resources format. Dict expected. Example: "
            f"{not_deleted_resources_map}")

        all_resources = self.get_all_resources()
        for vm in cluster_resources.get('vms', []):
            if vm in [vm for vm in all_resources.get('vms')]:
                not_deleted_resources_map['vms'].append(vm)
        for vol in cluster_resources.get('volumes', []):
            if vol in [vol for vol in all_resources.get('volumes')]:
                not_deleted_resources_map['volumes'].append(vol)

        not_deleted_resources = {k: v for k, v in
                                 not_deleted_resources_map.items() if v}
        self.not_deleted_resources = not_deleted_resources

        return not_deleted_resources
