import os
import uuid
from typing import TYPE_CHECKING
import yaml
from yaml import scanner as yaml_scanner

from si_tests import logger
from si_tests.deployments.utils import commons, file_utils
from si_tests.utils import utils

if TYPE_CHECKING:
    from si_tests.managers.kaas_manager import Cluster

LOG = logger.logger


class ClusterCheckMosManager(object):
    def __init__(self, cluster: "Cluster"):
        self._cluster: "Cluster" = cluster

    @property
    def cluster(self) -> "Cluster":
        return self._cluster

    def check_vrouter_pods(self, tf_pod_pref, os_manager):
        tf_ns = os_manager.tf_namespace
        ds_list = os_manager.api.daemonsets.list_starts_with(tf_pod_pref, tf_ns)
        LOG.info(f"Total vRouter DaemonSets: {len(ds_list)}")
        # container name for validation
        c_name = 'agent'
        for ds in ds_list:
            ds_data = ds.read()
            status = ds_data.status
            LOG.info(f"Verify DaemonSet {ds.name}: {status.number_ready}"
                     f"/{status.desired_number_scheduled} (ready/desired)")
            # Skip DaemonSets without scheduled pods
            if status.desired_number_scheduled == 0:
                continue
            pod_list = os_manager.api.pods.list_starts_with(ds.name, tf_ns)
            pods = [pod for pod in pod_list if pod.get_owner_reference() ==
                    f"DaemonSet/{ds.name}"]
            containers = ds_data.spec.template.spec.containers
            ds_c_spec_list = [c for c in containers if c.name == c_name]
            if not ds_c_spec_list:
                raise Exception(
                    f"DaemonSet {ds.name}: container {c_name} "
                    f"wasn't found in template")
            ds_c_spec = ds_c_spec_list.pop()
            pod_containers = []
            pod_name = None
            for pod in pods:
                pod_data = pod.read()
                pod_containers = [c for c in pod_data.spec.containers if
                                  c.name ==
                                  c_name]
                if pod_containers:
                    pod_name = pod.name
                    break
            if not pod_containers:
                raise Exception(
                    f"Container {c_name} wasn't found in pods belongs "
                    f"to DaemonSet {ds.name}")
            container = pod_containers.pop()
            LOG.info(f"Check {c_name} image in pod {pod_name} belonged to "
                     f"DaemonSet {ds.name}:\n"
                     f"DaemonSet image: {ds_c_spec.image}\n"
                     f"Pod image: {container.image}")
            assert container.image == ds_c_spec.image, \
                f"vRouter pod {pod_name} wasn't updated.\n" \
                f"{container.image} != {ds_c_spec.image}"

    @staticmethod
    def check_cassandra_nodes_config(os_manager, actualize_nodes_config=False):
        """
        Get current actual cassandra nodes, and compare with configured

        When 'actualize_nodes_config' is 'True' - delete missed nodes from config
                                      if 'False' - raise an error
        """

        tf_ns = os_manager.tf_namespace
        cassandra_pods = os_manager.api.pods.list_starts_with('tf-cassandra-config', tf_ns)
        LOG.info(f"Total tf-cassandra-config pods: {len(cassandra_pods)}")
        actual_cassandra_ips = [p.data['status']['pod_ip'] for p in cassandra_pods]
        LOG.info(f"List of actual cassandra nodes IPs: {actual_cassandra_ips}")

        cmd = ['/bin/sh', '-c', "nodetool -h ::FFFF:127.0.0.1 status 2>/dev/null|"
                                "grep rack|awk '{print $1\";\"$2\";\"$7}'"]
        resutl_cmd = cassandra_pods[0].exec(cmd, container='cassandra')
        assert len(resutl_cmd) > 0, "Cant read cassandra nodes list, command returned empty data"
        LOG.info(f"List of cassandra nodes in config:\n{resutl_cmd}")

        cassandra_nodes_to_delete = []
        for cassandra_node in resutl_cmd.splitlines():
            cassandra_node_status = cassandra_node.split(";")[0]
            cassandra_ip = cassandra_node.split(";")[1]
            cassandra_node_id = cassandra_node.split(";")[2]
            assert utils.is_valid_ip(cassandra_ip), \
                "Wrong information gathered. Returned result does not contain node ip"
            if cassandra_ip not in actual_cassandra_ips or cassandra_node_status.startswith("D"):
                cassandra_nodes_to_delete.append(cassandra_node_id)
                LOG.info(f"Unexpected cassandra node found {cassandra_node_id} {cassandra_ip} {cassandra_node_status}")

        if actualize_nodes_config:
            if len(cassandra_nodes_to_delete) > 0:
                for cassandra_node_to_delete in cassandra_nodes_to_delete:
                    LOG.info(f"Delete cassandra node {cassandra_node_to_delete} from config")
                    cassandra_pods[0].exec(
                        ['/bin/sh', '-c', f"nodetool -h ::FFFF:127.0.0.1 removenode force {cassandra_node_to_delete}"],
                        container='cassandra')
            else:
                LOG.info("Missing cassandra down nodes to delete")
        elif not actualize_nodes_config and len(cassandra_nodes_to_delete) > 0:
            raise Exception('Cassandra nodes config contain not actual nodes with Down status!\n'
                            f"List of cassandra nodes in config:\n{resutl_cmd}")

    @staticmethod
    def created_stack_tf_lb(request, openstack_client_manager, custom_params=None, finalizer=True):
        stack_id = uuid.uuid4()
        stack_name = f"si-test-tf-lb-{stack_id}"
        template_path = file_utils.join(os.path.dirname(os.path.abspath(__file__)),
                                        "../templates/heat_mosk_workloads/tf_loadbalancer.yaml")
        ssh_keys = utils.generate_keys()
        stack_params = {
            "id": str(stack_id),
            "ssh_public_key": f"ssh-rsa {ssh_keys['public']}",
            "ssh_private_key": ssh_keys["private"],
        }
        if custom_params:
            stack_params = stack_params | custom_params
        openstack_client_manager.create_stack(
            request, stack_name, template_path, stack_params=stack_params,
            finalizer=finalizer
        )
        return openstack_client_manager.stack.show([stack_name])

    @staticmethod
    def is_network_present(openstack_client_manager, network_name):
        network = openstack_client_manager.exec_os_cli(f"network show {network_name}", deserialize=True)
        return True if network else False

    @staticmethod
    def is_lb_functional(openstack_client_manager, num_members: int, url):
        # Simple check of LoadBalancer. Not applicable for some types of LB
        cmd = ['/bin/sh', '-c', f"curl -s {url}"]
        raw_responses = []
        unique_responses = 0
        commons.LOG.info("LB responses:")
        for i in range(num_members * 2):
            content = openstack_client_manager.client.exec(cmd)
            commons.LOG.info(f"{content}")
            if content and content not in raw_responses:
                unique_responses += 1
            raw_responses.append(content)
        if unique_responses == num_members:
            return True
        commons.LOG.warning(f"LoadBalancer returned {unique_responses} unique "
                            f"responses while number of members is {num_members}")
        return False

    @staticmethod
    def check_contrail_api_readiness(os_client_manager):
        cmd_net_list = ['/bin/sh', '-c', 'openstack network list -f yaml 2>/dev/null']
        try:
            yaml.safe_load(os_client_manager.client.exec(cmd_net_list))
            return True
        except yaml_scanner.ScannerError as e:
            LOG.warning(f"Got error {e} during parsing command status'")
            return False
