import exec_helpers
import yaml

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters

import packet

LOG = logger.logger


def get_ssh_client(ip):
    auth = exec_helpers.SSHAuth(
        username="mcc-user",
        key_filename=settings.KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE,
    )
    return exec_helpers.SSHClient(host=ip, auth=auth)


def _configure_network(ssh, node_id, conf, vlan_ids):
    LOG.info(f"Applying network config {conf}")
    floating_id, stor_frontend_id, stor_backend_id, neutron_tunnel_id = vlan_ids
    post_up = f"""#!/bin/bash
if [[ $IFACE == "bond0" ]]; then
    ip link add name public link bond0 type vlan id {floating_id}
    ip link add name stor-frontend link bond0 type vlan id {stor_frontend_id}
    ip link add name stor-backend link bond0 type vlan id {stor_backend_id}
    ip link add name neutron-tunnel link bond0 type vlan id {neutron_tunnel_id}
fi
"""
    post_down = """#!/bin/bash
if [[ $IFACE == "bond0" ]]; then
    ip link delete public type vlan
    ip link delete stor-frontend type vlan
    ip link delete stor-backend type vlan
    ip link delete neutron-tunnel type vlan
fi
"""
    enable_sh = f"""#!/bin/bash
set -x
set -e
cat << EOF > /etc/sysctl.d/101-openstack-ci.conf
net.bridge.bridge-nf-call-ip6tables=0
net.bridge.bridge-nf-call-iptables=0
net.bridge.bridge-nf-call-arptables=0
net.ipv4.conf.default.rp_filter=0
EOF
sysctl -p /etc/sysctl.d/101-openstack-ci.conf
# Public interface for openstack floating network vlan
ip l add link bond0 name public type vlan id {floating_id}
ip link set dev public up
# Interface for ceph public network
ip l add link bond0 name stor-frontend type vlan id {stor_frontend_id}
ip a add 172.31.1.{node_id}/24 dev stor-frontend
ip link set dev stor-frontend up
# Interface for ceph private network
ip l add link bond0 name stor-backend type vlan id {stor_backend_id}
ip a add 172.31.2.{node_id}/24 dev stor-backend
ip link set dev stor-backend up
# Interface for openstack tenant networks
ip l add link bond0 name neutron-tunnel type vlan id {neutron_tunnel_id}
ip a add 172.31.3.{node_id}/24 dev neutron-tunnel
ip link set dev neutron-tunnel up
ip link add name br-public type bridge
# Bridge used to connect openstack instances to floating network
ip a add 172.31.4.{node_id}/24 dev br-public
ip link set dev br-public up
ip link add veth-br type veth peer name veth-phy
ip link set veth-br up
ip link set veth-phy up
ip link set dev public master br-public
ip link set dev veth-br master br-public
# firewall configuration for floating network routing
ufw route allow from 172.31.4.0/24 to any
ufw route allow from any to 172.31.4.0/24
# nat for floating network
cat << EOF >> /etc/ufw/before.rules
*nat
:POSTROUTING ACCEPT [0:0]
-A POSTROUTING -s 172.31.4.0/24 -o bond0 -j MASQUERADE
COMMIT
EOF
ufw reload
"""
    tmp_conf = f"/tmp/mos_net.conf_{node_id}"
    tmp_post_up = f"/tmp/post_up_{node_id}"
    tmp_post_down = f"/tmp/post_down_{node_id}"
    tmp_enable_sh = "/tmp/enable.sh"

    with ssh.sudo(enforce=True):
        # firstly upload to tmp as due to permissions there is no
        # ability to write directly to /etc
        with ssh.open(tmp_conf, "w") as f:
            f.write(conf)
        with ssh.open(tmp_post_up, "w") as f:
            f.write(post_up)
        with ssh.open(tmp_post_down, "w") as f:
            f.write(post_down)
        with ssh.open(tmp_enable_sh, "w") as f:
            f.write(enable_sh)

        ssh.check_call("apt update && apt install vlan bridge-utils -y", verbose=True)
        ssh.check_call("modprobe 8021q", verbose=True)

        ssh.check_call(
            "echo 'source /etc/network/interfaces.d/*' | tee -a /etc/network/interfaces",
            verbose=True,
        )
        for config in [tmp_conf, tmp_post_up, tmp_post_down, tmp_enable_sh]:
            if config != tmp_conf:
                ssh.check_call(f"chmod +x {config}")
        ssh.check_call("/tmp/enable.sh")
        ssh.check_call(f"cp {tmp_conf} /etc/network/interfaces.d/mos_net.conf")
        ssh.check_call(f"cp {tmp_post_up} /etc/network/if-up.d/")
        ssh.check_call(f"cp {tmp_post_down} /etc/network/if-post-down.d/")


def configure_node_networking(ssh, node_ids, vlan_ids):
    node_id = node_ids.pop(0)
    net_conf = f"""
auto public
iface public inet manual

auto br-public
iface br-public inet static
    pre-up ip link add veth-br type veth peer name veth-phy
    pre-up ip link set veth-br up
    pre-up ip link set veth-phy up
    address 172.31.4.{node_id}
    netmask 255.255.255.0
    bridge_ports public veth-br
    post-down ip link del veth-br

auto stor-frontend
iface stor-frontend inet static
    address 172.31.1.{node_id}
    netmask 255.255.255.0

auto stor-backend
iface stor-backend inet static
    address 172.31.2.{node_id}
    netmask 255.255.255.0

auto neutron-tunnel
iface neutron-tunnel inet static
    address 172.31.3.{node_id}
    netmask 255.255.255.0
"""
    _configure_network(ssh, node_id, net_conf, vlan_ids)


def get_machine_equinix_device(manager, machine):
    providerID = machine.spec["providerSpec"]["value"]["providerID"]
    eq_id = providerID.split("://")[1]
    return manager.get_device(eq_id)


def get_equinix_device_bond_id(device):
    for port in device.network_ports:
        if port["type"] == "NetworkBondPort":
            return port["id"]


def get_machine_first_not_bootable_device(cluster, machine_name):
    res = None
    machine = cluster.get_machine(name=machine_name)
    storage = machine.data.get("status", {}).get("providerStatus", {}).get("hardware", {}).get("storage", [])
    for device in storage:
        if not device.get("isBoot", False):
            res = device["name"]
            break
    return res


def configure_equinix_mos_ceph(ns, ctl_nodes, cmp_nodes):
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    cluster = ns.get_cluster(cluster_name)
    nodes = {}
    for node in cmp_nodes:
        # equinix not always creates bootable partition on sda device
        waiters.wait(
            lambda: get_machine_first_not_bootable_device(cluster, node.name) is not None,
            timeout=600,
            interval=30,
            timeout_msg="Storage hardware failed to be discovered",
        )
        osd_device_name = get_machine_first_not_bootable_device(cluster, node.name)
        nodes[node.name] = {"storageDevices": [{"config": {"deviceClass": "ssd"}, "name": osd_device_name}]}
    # Due to https://mirantis.jira.com/browse/PRODX-16961,
    # rgw cannot be deployed on master nodes when controlplane is colocated
    # so placing rgw (which is tied to mon label) on first 3 osd nodes which are computes in mos case
    if settings.KAAS_EQUINIX_CHILD_DEDICATED_CONTROLPLANE:
        for node in ctl_nodes:
            nodes[node.name] = {"roles": ["mon", "mgr"]}
    else:
        for node in cmp_nodes[0:3]:
            nodes[node.name]["roles"] = ["mon", "mgr"]

    pools = """
    - default: false
      deviceClass: ssd
      name: volumes
      replicated:
        size: 2
      role: volumes
    - default: false
      deviceClass: ssd
      name: vms
      replicated:
        size: 2
      role: vms
    - default: false
      deviceClass: ssd
      name: backup
      replicated:
        size: 2
      role: backup
    - default: false
      deviceClass: ssd
      name: images
      replicated:
        size: 2
      role: images
    - default: false
      deviceClass: ssd
      name: other
      replicated:
        size: 2
      role: other
    - default: true
      deviceClass: ssd
      erasureCoded:
        codingChunks: 0
        dataChunks: 0
      name: kubernetes
      replicated:
        size: 3
      role: kubernetes-ssd
    """
    rgw = yaml.safe_load(
        """
      dataPool:
        deviceClass: ssd
        erasureCoded:
          codingChunks: 1
          dataChunks: 2
        failureDomain: host
      gateway:
        allNodes: false
        instances: 1
        port: 80
        securePort: 8443
      healthCheck:
        bucket: {}
      metadataPool:
        deviceClass: ssd
        failureDomain: host
        replicated:
          size: 2
      name: openstack-store
      preservePoolsOnDelete: false
      zone:
        name: ""
    """
    )
    network = """
      hostNetwork: true
      clusterNet: 172.31.2.0/24
      publicNet: 172.31.1.0/24
    """
    ns.create_ceph_cluster(
        name=cluster_name,
        cluster_name=cluster_name,
        hdd_type="ssd",
        nodes_data=nodes,
        pools=yaml.safe_load(pools),
        rgw=rgw,
        network=yaml.safe_load(network)
    )


def configure_equinix_mos_networking(cluster):
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME

    eq_manager = packet.Manager(auth_token=settings.KAAS_EQUINIX_USER_API_TOKEN)
    eq_project_id = settings.KAAS_EQUINIX_PROJECT_ID
    facility = cluster.spec["providerSpec"]["value"]["facility"]

    # create vlans
    networks = ["floating", "stor_frontend", "stor_backend", "neutron_tunnel"]
    vlan_ids = []
    vlan_uids = []
    # 4 vlans - floating, stor_frontend, stor_backend, neutron_tunnel
    for v in networks:
        # description is expected by cleanup job
        vlan = eq_manager.create_vlan(
            project_id=eq_project_id,
            facility=facility,
            description=f"kaas.mirantis.com/cluster:{cluster_name}",
        )
        LOG.info(f"Created vlan {vlan.vxlan} for network {v}")
        vlan_uids.append(vlan.id)
        vlan_ids.append(vlan.vxlan)

    nodes = cluster.get_machines()
    # node ids are needed for node ips generation
    node_ids = list(range(10, 10 + len(nodes)))
    for m in nodes:
        LOG.info(f"Configuring master node {m.name} ip {m.public_ip}")
        device = get_machine_equinix_device(eq_manager, m)
        bond_id = get_equinix_device_bond_id(device)
        # neutron-tunnel network is not needed on masters
        for vlan_uid in vlan_uids:
            eq_manager.assign_port(bond_id, vlan_uid)
        ssh_client = get_ssh_client(m.public_ip)
        configure_node_networking(ssh_client, node_ids, vlan_ids)
