#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import si_tests.utils.waiters as helpers
from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.clients.k8s.base import K8sBaseManager
from si_tests import logger

LOG = logger.logger


class K8sJob(K8sNamespacedResource):
    resource_type = 'job'

    def _read(self, **kwargs):
        return self._manager.api.read_namespaced_job(
            self.name, self.namespace, **kwargs)

    def _create(self, body, **kwargs):
        return self._manager.api.create_namespaced_job(
            self.namespace, body, **kwargs)

    def _patch(self, body, **kwargs):
        return self._manager.api.patch_namespaced_job(
            self.name, self.namespace, body, **kwargs)

    def _replace(self, body, **kwargs):
        return self._manager.api.replace_namespaced_job(
            self.name, self.namespace, body, **kwargs)

    def _delete(self, async_del=False, propagation_policy='Background', timeout=120, **kwargs):
        uid = self.uid
        LOG.info(f"Deleting job {self.namespace}/{self.name} with UID: {uid}")
        self._manager.api.delete_namespaced_job(
            self.name, self.namespace, propagation_policy=propagation_policy, **kwargs)
        if not async_del:
            helpers.wait(lambda: not self.exists(),
                         timeout=timeout,
                         timeout_msg=f"Timeout deleting job {self.namespace}/{self.name}")

    def exists(self):
        jobs = self._manager.list_raw(self.namespace).to_dict()['items']
        job = [x for x in jobs if x['metadata']['uid'] == self.uid]
        return True if job else False

    def wait_succeded(self, timeout=500, interval=3):
        helpers.wait(lambda: self.read().status.succeeded == 1,
                     timeout=timeout, interval=interval,
                     timeout_msg='Timeout waiting, job {0} phase is not '
                                 'succeded'.format(self.name))

    def wait_finished(self, timeout=500, interval=3):
        helpers.wait(lambda: (self.read().status.succeeded == 1) or (self.read().status.failed == 1),
                     timeout=timeout, interval=interval,
                     timeout_msg='Timeout waiting, job {0} phase is not '
                                 'succeded'.format(self.name))

    @property
    def succeeded(self):
        status = self.read().status
        return True if status.succeeded == 1 else False


class K8sJobManager(K8sBaseManager):
    resource_class = K8sJob

    @property
    def api(self):
        return self._cluster.api_batch

    def _list(self, namespace, **kwargs):
        return self.api.list_namespaced_job(namespace, **kwargs)

    def _list_all(self, **kwargs):
        return self.api.list_job_for_all_namespaces(**kwargs)

    def check_jobs_completed(self, target_namespaces=None,
                             excluded_jobs=[], jobs_prefix=''):
        jobs = self.list_raw().to_dict()['items']
        failed_jobs = {}
        cron_jobs = []
        for job in jobs:
            name = job['metadata']['name']
            namespace = job['metadata']['namespace']
            status = job.get('status') or {}
            owner_references = [item.get('kind') for item in job['metadata'].get('owner_references') or {}]
            if target_namespaces and namespace not in target_namespaces:
                continue
            exclude = False
            for one in excluded_jobs:
                if name.startswith(one):
                    exclude = True
                    break
            if exclude:
                continue
            if jobs_prefix and not name.startswith(jobs_prefix):
                continue
            if 'CronJob' in owner_references:
                cron_jobs.append(name)
                continue
            if status.get('active', {}) is not None \
                    or status.get('succeeded', {}) != 1:
                failed_jobs[name] = {'active': status.get('active', {}),
                                     'succeeded': status.get('succeeded', {})}
        if len(failed_jobs) > 0:
            LOG.warning(f"Next jobs did not finish or failed: {failed_jobs}")
            return False
        LOG.info(f"Skipped CronJobs: {cron_jobs}")
        return True
