import time

from si_tests import logger
from keycloak import exceptions
from keycloak import KeycloakAdmin
from keycloak import KeycloakOpenID
from si_tests import settings

LOG = logger.logger


def relogin_keycloak_401(func):
    """Relogin to keycloak only once in case of KeycloakAuthenticationError 401"""
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except exceptions.KeycloakAuthenticationError as e:
            if e.response_code == 401:
                LOG.debug(f"Keycloak 'Unauthorized' error:\n{e}")
                LOG.info("Keycloak 'Unauthorized' error, "
                         "trying to refresh authorization")
                time.sleep(3)
                cls = args[0]
                cls.login()
                return func(*args, **kwargs)
            raise e
    return wrapper


class KeycloakAdminClient(object):

    def __init__(self, ip,
                 user=settings.KEYCLOAK_USER,
                 password=settings.KEYCLOAK_PASSWORD,
                 verify=settings.IAM_CACERT, realm='master',
                 client_id='admin-cli', grant_type='password'):
        self.url = "https://" + ip
        self.user = user
        self.password = password
        # realm is master by default
        # self.realm = realm
        # client_id is admin-cli by default
        # self.client_id = client_id
        # grant_type is password by default
        # self.grant_type = grant_type
        self.cacert_path = verify
        # Init Keycloak client
        self.login()

    def login(self):
        self.client = KeycloakAdmin(
            server_url="{}/auth/".format(self.url),
            username=self.user, password=self.password,
            verify=self.cacert_path,
        )
        # switching to iam realm
        self.client.realm_name = 'iam'

    @relogin_keycloak_401
    def get_token(self):
        return self.client.token

    @relogin_keycloak_401
    def user_create(self, username, password, email=None,
                    firstname="si", lastname="test", enabled=True):
        """Returns user ID of existing or created user."""
        return self.client.create_user({
            "email": email,
            "username": username,
            "enabled": enabled,
            "firstName": firstname,
            "lastName": lastname,
            "credentials": [{"value": password, "type": "password"}]
        })

    @relogin_keycloak_401
    def user_delete(self, username):
        id = self.client.get_user_id(username)
        self.client.delete_user(id)

    @relogin_keycloak_401
    def user_update(self, username, body):
        id = self.client.get_user_id(username)
        self.client.update_user(id, body)

    @relogin_keycloak_401
    def user_get(self, username):
        id = self.client.get_user_id(username)
        return self.client.get_user(id)

    @relogin_keycloak_401
    def user_get_by_id(self, userid):
        return self.client.get_user(userid)

    @relogin_keycloak_401
    def get_id_by_username(self, username):
        return self.client.get_user_id(username)

    @relogin_keycloak_401
    def roles_get(self, roles):
        all_roles = self.client.get_realm_roles()
        body = []
        for role in all_roles:
            if role['name'] in roles:
                body.append(role)
        return body

    @relogin_keycloak_401
    def get_realm_roles(self):
        return self.client.get_realm_roles()

    @relogin_keycloak_401
    def get_user_roles(self, user_id):
        return self.client.get_realm_roles_of_user(user_id)

    @relogin_keycloak_401
    def assign_roles(self, username, roles):
        user_id = self.user_get(username)['id']
        roles_list = self.roles_get(roles)
        self.client.assign_realm_roles(user_id, roles_list)
        return True

    def sync_user(self, username, password):
        # sync from ldap
        raise NotImplementedError("To be implemented")


class KeycloakUserClient(object):

    def __init__(self, ip, user, password,
                 verify=settings.IAM_CACERT, client_id='iam',
                 realm_name='iam', client_secret_key=""):
        self.user = user
        self.password = password
        self.client = KeycloakOpenID(server_url="https://{}/auth/".format(ip),
                                     client_id=client_id,
                                     realm_name=realm_name,
                                     client_secret_key=client_secret_key,
                                     verify=verify)
        # Get WellKnow
        # config_well_know = keycloak_openid.well_know()

        self.token = self.client.token(self.user, self.password)

        # Get Token
    def get_token(self):
        return self.token['access_token']

    def refresh_token(self):
        return self.client.refresh_token(self.token['refresh_token'])

    def get_openid_token(self):
        return self.client.token(self.user, self.password, scope='openid')

    def get_offline_openid_token(self):
        return self.client.token(self.user, self.password, scope='openid offline_access')
