#    Author: Alex Savatieiev (osavatieiev@mirantis.com; a.savex@gmail.com)
#    Copyright 2019-2022 Mirantis, Inc.
from cfg_checker.common import logger_cli
from cfg_checker.common.settings import ENV_TYPE_SALT, ENV_TYPE_KUBE
from cfg_checker.common.exception import CheckerException
from cfg_checker.helpers import args_utils
from cfg_checker.modules.network import checker, mapper, pinger


command_help = "Network infrastructure checks and reports"
supported_envs = [ENV_TYPE_SALT, ENV_TYPE_KUBE]


def mtu_type(value):
    try:
        _mtu = int(value)
    except ValueError:
        _mtu = value
    return _mtu


def _selectClass(_env, strClassHint="checker"):
    _class = None
    if _env == ENV_TYPE_SALT:
        if strClassHint == "checker":
            _class = checker.SaltNetworkChecker
        elif strClassHint == "mapper":
            _class = mapper.SaltNetworkMapper
        elif strClassHint == "pinger":
            _class = pinger.SaltNetworkPinger
    elif _env == ENV_TYPE_KUBE:
        if strClassHint == "checker":
            _class = checker.KubeNetworkChecker
        elif strClassHint == "mapper":
            _class = mapper.KubeNetworkMapper
        elif strClassHint == "pinger":
            _class = pinger.KubeNetworkPinger
    if not _class:
        raise CheckerException(
            "Unknown hint for selecting Network handler Class: '{}'".format(
                strClassHint
            )
        )
    else:
        return _class


def init_parser(_parser):
    # network subparser
    net_subparsers = _parser.add_subparsers(dest='type')

    net_check_parser = net_subparsers.add_parser(
        'check',
        help="Do network check and print the result"
    )

    net_check_parser.add_argument(
        '--detailed',
        action="store_true", default=False,
        help="Print error details after summary"
    )

    net_check_parser.add_argument(
        '--skip-ifs',
        metavar='skip_ifs', default="docker",
        help="String with keywords to skip networks which has interfaces "
        "names with keywords as substrings. Example: 'eno' keyword will "
        "cause to skip interface named 'eno1np0'. Example: 'docker'"
    )

    net_report_parser = net_subparsers.add_parser(
        'report',
        help="Generate network check report"
    )

    net_report_parser.add_argument(
        '--html',
        metavar='network_html_filename',
        help="HTML filename to save report"
    )

    net_report_parser.add_argument(
        '--skip-ifs',
        metavar='skip_ifs', default="docker",
        help="String with keywords to skip networks which has interfaces "
        "names with keywords as substrings. Example: 'eno' keyword will "
        "cause to skip interface named 'eno1np0'. Example: 'docker'"
    )

    net_ping_parser = net_subparsers.add_parser(
        'ping',
        help="Ping all nodes with each other using network CIDR"
    )

    net_ping_parser.add_argument(
        '--cidr',
        metavar='network_ping_cidr',
        action="append",
        help="Subnet CIDR to ping nodes in. Can be used multiple times"
    )
    net_ping_parser.add_argument(
        '--mtu',
        metavar='network_ping_mtu', default=64, type=mtu_type,
        help="MTU size to use. Ping will be done for MTU - 20 - 8. Default: 64"
    )
    net_ping_parser.add_argument(
        '--detailed',
        action="store_true", default=False,
        help="Print detailed report at the end"
    )

    net_subparsers.add_parser(
        'map',
        help="Print network map"
    )

    net_subparsers.add_parser(
        'list',
        help="List networks in reclass"
    )

    return _parser


def do_check(args, config):
    # Net Checks
    # should not print map, etc...
    # Just bare summary and errors
    # Check if there is supported env found
    _env = args_utils.check_supported_env(
        [ENV_TYPE_SALT, ENV_TYPE_KUBE],
        args,
        config
    )
    # prepare skip keywords
    _skip_ifs_keywords = []
    for _str in args.skip_ifs.split(","):
        _skip_ifs_keywords.append(_str)
    logger_cli.info(
        "-> Interface keywords skip list is '{}'".format(
            ", ".join(_skip_ifs_keywords)
        )
    )
    # Start command
    logger_cli.info("# Network check to console")
    _skip, _skip_file = args_utils.get_skip_args(args)
    _class = _selectClass(_env)
    netChecker = _class(
        config,
        skip_list=_skip,
        skip_list_file=_skip_file
    )
    netChecker.check_networks(skip_keywords=_skip_ifs_keywords)

    # save what was collected
    netChecker.errors.save_iteration_data()

    # print a report
    netChecker.print_summary()

    # if set, print details
    if args.detailed:
        netChecker.print_error_details()


def do_report(args, config):
    # Network Report
    # Check if there is supported env found
    _env = args_utils.check_supported_env(
        [ENV_TYPE_SALT, ENV_TYPE_KUBE],
        args,
        config
    )
    # Start command
    logger_cli.info("# Network report (check, node map)")

    _skip_ifs_keywords = []
    for _str in args.skip_ifs.split(","):
        _skip_ifs_keywords.append(_str)
    logger_cli.info(
        "-> Interface keywords skip list is '{}'".format(
            ", ".join(_skip_ifs_keywords)
        )
    )

    _filename = args_utils.get_arg(args, 'html')
    _skip, _skip_file = args_utils.get_skip_args(args)
    _class = _selectClass(_env)
    netChecker = _class(
        config,
        skip_list=_skip,
        skip_list_file=_skip_file
    )
    netChecker.check_networks(skip_keywords=_skip_ifs_keywords, map=False)

    # save what was collected
    netChecker.errors.save_iteration_data()
    netChecker.create_html_report(_filename)

    return


def do_map(args, config):
    # Network Map
    # Check if there is supported env found
    _env = args_utils.check_supported_env(
        [ENV_TYPE_SALT, ENV_TYPE_KUBE],
        args,
        config
    )
    # Start command
    logger_cli.info("# Network report")
    _skip, _skip_file = args_utils.get_skip_args(args)
    _class = _selectClass(_env, strClassHint="mapper")
    networkMap = _class(
        config,
        skip_list=_skip,
        skip_list_file=_skip_file
    )
    networkMap.map_networks()
    networkMap.create_map()
    networkMap.print_map()

    return


def do_list(args, config):
    # Network List
    # Check if there is supported env found
    _env = args_utils.check_supported_env(
        [ENV_TYPE_SALT, ENV_TYPE_KUBE],
        args,
        config
    )
    # Start command
    _skip, _skip_file = args_utils.get_skip_args(args)
    _class = _selectClass(_env, strClassHint="mapper")
    _map = _class(
        config,
        skip_list=_skip,
        skip_list_file=_skip_file
    )
    logger_cli.info("# Mapping networks")
    if _env == ENV_TYPE_SALT:
        reclass = _map.map_network(_map.RECLASS)
        _s = [str(_n) for _n in reclass.keys()]
        logger_cli.info("\n# Reclass networks list")
        logger_cli.info("\n".join(_s))

    runtime = _map.map_network(_map.RUNTIME)
    logger_cli.info("\n# Runtime networks list")
    for _net, _nodes in runtime.items():
        _names = set()
        _mtus = set()
        for _node_name, _interfaces in _nodes.items():
            for _if in _interfaces:
                _names.add(_if["name"])
                _mtus.add(_if["mtu"])
        logger_cli.info(
            "{:21}: {}, {}".format(
                str(_net),
                "/".join(list(_names)),
                "/".join(list(_mtus))
            )
        )
    # Done
    return


def do_ping(args, config):
    # Network pinger
    # Checks if selected nodes are pingable
    # with a desireble parameters: MTU, Frame, etc
    # Check if there is supported env found
    _env = args_utils.check_supported_env(
        [ENV_TYPE_SALT, ENV_TYPE_KUBE],
        args,
        config
    )
    # Start command
    # prepare parameters
    if not args.cidr:
        logger_cli.error("\n# Use mcp-check network list to get list of CIDRs")
    _cidr = args_utils.get_arg(args, "cidr")
    _skip, _skip_file = args_utils.get_skip_args(args)
    _mtu = args_utils.get_arg(args, "mtu")
    if isinstance(_mtu, str):
        if _mtu == "auto":
            logger_cli.info(
                "# Supplied MTU is 'auto'. Network's MTU value will be used"
            )
            _mtu = 0
        else:
            raise CheckerException(
                "Bad MTU value supplied: '{}'.\n"
                "Allowed values are: 'auto'/0 (autodetect), 64 - 9100".format(
                    _mtu
                )
            )
    elif isinstance(_mtu, int):
        if _mtu < 64 and _mtu > 0:
            logger_cli.warn(
                "WARNING: Supplied MTU value is low: '{}'"
                "Defaulting to '64'".format(_mtu)
            )
            _mtu = 64
        elif _mtu == 0:
            logger_cli.info(
                "# MTU set to '0'. Network's MTU value will be used."
            )
        elif _mtu > 9100:
            logger_cli.warn(
                "WARNING: Supplied MTU value is >9100: '{}'"
                "Defaulting to '9100'".format(_mtu)
            )
            _mtu = 9100
        elif _mtu > 63 or _mtu < 9101:
            logger_cli.info(
                "# MTU set to '{}'.".format(_mtu)
            )
        else:
            raise CheckerException(
                "Negative MTU values not supported: '{}'".format(
                    _mtu
                )
            )

    # init mapper
    _skip, _skip_file = args_utils.get_skip_args(args)
    _class = _selectClass(_env, strClassHint="mapper")
    _mapper = _class(
        config,
        skip_list=_skip,
        skip_list_file=_skip_file
    )
    # init pinger
    _class = _selectClass(_env, strClassHint="pinger")
    _pinger = _class(
        _mapper,
        detailed=args.detailed,
        skip_list=_skip,
        skip_list_file=_skip_file
    )
    # Iterate networks to be checked
    logger_cli.info("# Checking {} networks".format(len(_cidr)))
    _results = {}
    for _t_cidr in _cidr:
        logger_cli.info("-> '{}'".format(_t_cidr))
        _results[_t_cidr] = _pinger.ping_nodes(_t_cidr, _mtu)

    logger_cli.info("\n# Summary")
    _dataset = []
    for _cidr, _data in _results.items():
        if not _data:
            # no need to save the iterations and summary
            _dataset.append(False)
        else:
            # print a report for cidr
            if args.detailed:
                # if set, print details
                _pinger.print_details(_cidr, _data)
            else:
                _pinger.print_summary(_cidr, _data)

    if any(_dataset):
        # save what was collected
        _pinger.mapper.errors.save_iteration_data()
    return


def do_trace(args, config):
    # Network packet tracer
    # Check if packet is delivered to proper network host
    logger_cli.info("# Packet Tracer not yet implemented")

    # TODO: Packet tracer

    return
